/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicecatalog.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a request operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecordDetail implements SdkPojo, Serializable, ToCopyableBuilder<RecordDetail.Builder, RecordDetail> {
    private static final SdkField<String> RECORD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecordId").getter(getter(RecordDetail::recordId)).setter(setter(Builder::recordId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordId").build()).build();

    private static final SdkField<String> PROVISIONED_PRODUCT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProvisionedProductName").getter(getter(RecordDetail::provisionedProductName))
            .setter(setter(Builder::provisionedProductName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedProductName").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(RecordDetail::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(RecordDetail::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedTime").getter(getter(RecordDetail::updatedTime)).setter(setter(Builder::updatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedTime").build()).build();

    private static final SdkField<String> PROVISIONED_PRODUCT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProvisionedProductType").getter(getter(RecordDetail::provisionedProductType))
            .setter(setter(Builder::provisionedProductType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedProductType").build())
            .build();

    private static final SdkField<String> RECORD_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecordType").getter(getter(RecordDetail::recordType)).setter(setter(Builder::recordType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordType").build()).build();

    private static final SdkField<String> PROVISIONED_PRODUCT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProvisionedProductId").getter(getter(RecordDetail::provisionedProductId))
            .setter(setter(Builder::provisionedProductId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedProductId").build())
            .build();

    private static final SdkField<String> PRODUCT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProductId").getter(getter(RecordDetail::productId)).setter(setter(Builder::productId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductId").build()).build();

    private static final SdkField<String> PROVISIONING_ARTIFACT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProvisioningArtifactId").getter(getter(RecordDetail::provisioningArtifactId))
            .setter(setter(Builder::provisioningArtifactId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisioningArtifactId").build())
            .build();

    private static final SdkField<String> PATH_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("PathId")
            .getter(getter(RecordDetail::pathId)).setter(setter(Builder::pathId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PathId").build()).build();

    private static final SdkField<List<RecordError>> RECORD_ERRORS_FIELD = SdkField
            .<List<RecordError>> builder(MarshallingType.LIST)
            .memberName("RecordErrors")
            .getter(getter(RecordDetail::recordErrors))
            .setter(setter(Builder::recordErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordErrors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RecordError> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecordError::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RecordTag>> RECORD_TAGS_FIELD = SdkField
            .<List<RecordTag>> builder(MarshallingType.LIST)
            .memberName("RecordTags")
            .getter(getter(RecordDetail::recordTags))
            .setter(setter(Builder::recordTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RecordTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecordTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LAUNCH_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LaunchRoleArn").getter(getter(RecordDetail::launchRoleArn)).setter(setter(Builder::launchRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RECORD_ID_FIELD,
            PROVISIONED_PRODUCT_NAME_FIELD, STATUS_FIELD, CREATED_TIME_FIELD, UPDATED_TIME_FIELD, PROVISIONED_PRODUCT_TYPE_FIELD,
            RECORD_TYPE_FIELD, PROVISIONED_PRODUCT_ID_FIELD, PRODUCT_ID_FIELD, PROVISIONING_ARTIFACT_ID_FIELD, PATH_ID_FIELD,
            RECORD_ERRORS_FIELD, RECORD_TAGS_FIELD, LAUNCH_ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String recordId;

    private final String provisionedProductName;

    private final String status;

    private final Instant createdTime;

    private final Instant updatedTime;

    private final String provisionedProductType;

    private final String recordType;

    private final String provisionedProductId;

    private final String productId;

    private final String provisioningArtifactId;

    private final String pathId;

    private final List<RecordError> recordErrors;

    private final List<RecordTag> recordTags;

    private final String launchRoleArn;

    private RecordDetail(BuilderImpl builder) {
        this.recordId = builder.recordId;
        this.provisionedProductName = builder.provisionedProductName;
        this.status = builder.status;
        this.createdTime = builder.createdTime;
        this.updatedTime = builder.updatedTime;
        this.provisionedProductType = builder.provisionedProductType;
        this.recordType = builder.recordType;
        this.provisionedProductId = builder.provisionedProductId;
        this.productId = builder.productId;
        this.provisioningArtifactId = builder.provisioningArtifactId;
        this.pathId = builder.pathId;
        this.recordErrors = builder.recordErrors;
        this.recordTags = builder.recordTags;
        this.launchRoleArn = builder.launchRoleArn;
    }

    /**
     * <p>
     * The identifier of the record.
     * </p>
     * 
     * @return The identifier of the record.
     */
    public final String recordId() {
        return recordId;
    }

    /**
     * <p>
     * The user-friendly name of the provisioned product.
     * </p>
     * 
     * @return The user-friendly name of the provisioned product.
     */
    public final String provisionedProductName() {
        return provisionedProductName;
    }

    /**
     * <p>
     * The status of the provisioned product.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATED</code> - The request was created but the operation has not started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - The requested operation is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested operation failed
     * and some remediation is occurring. For example, a rollback.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code> - The requested operation has successfully completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the error messages
     * returned.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RecordStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the provisioned product.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATED</code> - The request was created but the operation has not started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - The requested operation is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested operation
     *         failed and some remediation is occurring. For example, a rollback.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code> - The requested operation has successfully completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the error
     *         messages returned.
     *         </p>
     *         </li>
     * @see RecordStatus
     */
    public final RecordStatus status() {
        return RecordStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the provisioned product.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATED</code> - The request was created but the operation has not started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - The requested operation is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested operation failed
     * and some remediation is occurring. For example, a rollback.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code> - The requested operation has successfully completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the error messages
     * returned.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RecordStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the provisioned product.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATED</code> - The request was created but the operation has not started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - The requested operation is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested operation
     *         failed and some remediation is occurring. For example, a rollback.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code> - The requested operation has successfully completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the error
     *         messages returned.
     *         </p>
     *         </li>
     * @see RecordStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The UTC time stamp of the creation time.
     * </p>
     * 
     * @return The UTC time stamp of the creation time.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The time when the record was last updated.
     * </p>
     * 
     * @return The time when the record was last updated.
     */
    public final Instant updatedTime() {
        return updatedTime;
    }

    /**
     * <p>
     * The type of provisioned product. The supported values are <code>CFN_STACK</code>, <code>CFN_STACKSET</code>,
     * <code>TERRAFORM_OPEN_SOURCE</code>, <code>TERRAFORM_CLOUD</code>, and <code>EXTERNAL</code>.
     * </p>
     * 
     * @return The type of provisioned product. The supported values are <code>CFN_STACK</code>,
     *         <code>CFN_STACKSET</code>, <code>TERRAFORM_OPEN_SOURCE</code>, <code>TERRAFORM_CLOUD</code>, and
     *         <code>EXTERNAL</code>.
     */
    public final String provisionedProductType() {
        return provisionedProductType;
    }

    /**
     * <p>
     * The record type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PROVISION_PRODUCT</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATE_PROVISIONED_PRODUCT</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TERMINATE_PROVISIONED_PRODUCT</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The record type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PROVISION_PRODUCT</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATE_PROVISIONED_PRODUCT</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TERMINATE_PROVISIONED_PRODUCT</code>
     *         </p>
     *         </li>
     */
    public final String recordType() {
        return recordType;
    }

    /**
     * <p>
     * The identifier of the provisioned product.
     * </p>
     * 
     * @return The identifier of the provisioned product.
     */
    public final String provisionedProductId() {
        return provisionedProductId;
    }

    /**
     * <p>
     * The product identifier.
     * </p>
     * 
     * @return The product identifier.
     */
    public final String productId() {
        return productId;
    }

    /**
     * <p>
     * The identifier of the provisioning artifact.
     * </p>
     * 
     * @return The identifier of the provisioning artifact.
     */
    public final String provisioningArtifactId() {
        return provisioningArtifactId;
    }

    /**
     * <p>
     * The path identifier.
     * </p>
     * 
     * @return The path identifier.
     */
    public final String pathId() {
        return pathId;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecordErrors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecordErrors() {
        return recordErrors != null && !(recordErrors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The errors that occurred.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecordErrors} method.
     * </p>
     * 
     * @return The errors that occurred.
     */
    public final List<RecordError> recordErrors() {
        return recordErrors;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecordTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRecordTags() {
        return recordTags != null && !(recordTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecordTags} method.
     * </p>
     * 
     * @return One or more tags.
     */
    public final List<RecordTag> recordTags() {
        return recordTags;
    }

    /**
     * <p>
     * The ARN of the launch role associated with the provisioned product.
     * </p>
     * 
     * @return The ARN of the launch role associated with the provisioned product.
     */
    public final String launchRoleArn() {
        return launchRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(recordId());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedProductName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(updatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedProductType());
        hashCode = 31 * hashCode + Objects.hashCode(recordType());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedProductId());
        hashCode = 31 * hashCode + Objects.hashCode(productId());
        hashCode = 31 * hashCode + Objects.hashCode(provisioningArtifactId());
        hashCode = 31 * hashCode + Objects.hashCode(pathId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecordErrors() ? recordErrors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRecordTags() ? recordTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(launchRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecordDetail)) {
            return false;
        }
        RecordDetail other = (RecordDetail) obj;
        return Objects.equals(recordId(), other.recordId())
                && Objects.equals(provisionedProductName(), other.provisionedProductName())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(updatedTime(), other.updatedTime())
                && Objects.equals(provisionedProductType(), other.provisionedProductType())
                && Objects.equals(recordType(), other.recordType())
                && Objects.equals(provisionedProductId(), other.provisionedProductId())
                && Objects.equals(productId(), other.productId())
                && Objects.equals(provisioningArtifactId(), other.provisioningArtifactId())
                && Objects.equals(pathId(), other.pathId()) && hasRecordErrors() == other.hasRecordErrors()
                && Objects.equals(recordErrors(), other.recordErrors()) && hasRecordTags() == other.hasRecordTags()
                && Objects.equals(recordTags(), other.recordTags()) && Objects.equals(launchRoleArn(), other.launchRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecordDetail").add("RecordId", recordId())
                .add("ProvisionedProductName", provisionedProductName()).add("Status", statusAsString())
                .add("CreatedTime", createdTime()).add("UpdatedTime", updatedTime())
                .add("ProvisionedProductType", provisionedProductType()).add("RecordType", recordType())
                .add("ProvisionedProductId", provisionedProductId()).add("ProductId", productId())
                .add("ProvisioningArtifactId", provisioningArtifactId()).add("PathId", pathId())
                .add("RecordErrors", hasRecordErrors() ? recordErrors() : null)
                .add("RecordTags", hasRecordTags() ? recordTags() : null).add("LaunchRoleArn", launchRoleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RecordId":
            return Optional.ofNullable(clazz.cast(recordId()));
        case "ProvisionedProductName":
            return Optional.ofNullable(clazz.cast(provisionedProductName()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "UpdatedTime":
            return Optional.ofNullable(clazz.cast(updatedTime()));
        case "ProvisionedProductType":
            return Optional.ofNullable(clazz.cast(provisionedProductType()));
        case "RecordType":
            return Optional.ofNullable(clazz.cast(recordType()));
        case "ProvisionedProductId":
            return Optional.ofNullable(clazz.cast(provisionedProductId()));
        case "ProductId":
            return Optional.ofNullable(clazz.cast(productId()));
        case "ProvisioningArtifactId":
            return Optional.ofNullable(clazz.cast(provisioningArtifactId()));
        case "PathId":
            return Optional.ofNullable(clazz.cast(pathId()));
        case "RecordErrors":
            return Optional.ofNullable(clazz.cast(recordErrors()));
        case "RecordTags":
            return Optional.ofNullable(clazz.cast(recordTags()));
        case "LaunchRoleArn":
            return Optional.ofNullable(clazz.cast(launchRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecordDetail, T> g) {
        return obj -> g.apply((RecordDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecordDetail> {
        /**
         * <p>
         * The identifier of the record.
         * </p>
         * 
         * @param recordId
         *        The identifier of the record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordId(String recordId);

        /**
         * <p>
         * The user-friendly name of the provisioned product.
         * </p>
         * 
         * @param provisionedProductName
         *        The user-friendly name of the provisioned product.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedProductName(String provisionedProductName);

        /**
         * <p>
         * The status of the provisioned product.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATED</code> - The request was created but the operation has not started.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - The requested operation is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested operation
         * failed and some remediation is occurring. For example, a rollback.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code> - The requested operation has successfully completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the error
         * messages returned.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the provisioned product.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATED</code> - The request was created but the operation has not started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - The requested operation is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested
         *        operation failed and some remediation is occurring. For example, a rollback.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code> - The requested operation has successfully completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the
         *        error messages returned.
         *        </p>
         *        </li>
         * @see RecordStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the provisioned product.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATED</code> - The request was created but the operation has not started.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - The requested operation is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested operation
         * failed and some remediation is occurring. For example, a rollback.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code> - The requested operation has successfully completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the error
         * messages returned.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the provisioned product.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATED</code> - The request was created but the operation has not started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - The requested operation is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS_IN_ERROR</code> - The provisioned product is under change but the requested
         *        operation failed and some remediation is occurring. For example, a rollback.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code> - The requested operation has successfully completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The requested operation has unsuccessfully completed. Investigate using the
         *        error messages returned.
         *        </p>
         *        </li>
         * @see RecordStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordStatus
         */
        Builder status(RecordStatus status);

        /**
         * <p>
         * The UTC time stamp of the creation time.
         * </p>
         * 
         * @param createdTime
         *        The UTC time stamp of the creation time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The time when the record was last updated.
         * </p>
         * 
         * @param updatedTime
         *        The time when the record was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedTime(Instant updatedTime);

        /**
         * <p>
         * The type of provisioned product. The supported values are <code>CFN_STACK</code>, <code>CFN_STACKSET</code>,
         * <code>TERRAFORM_OPEN_SOURCE</code>, <code>TERRAFORM_CLOUD</code>, and <code>EXTERNAL</code>.
         * </p>
         * 
         * @param provisionedProductType
         *        The type of provisioned product. The supported values are <code>CFN_STACK</code>,
         *        <code>CFN_STACKSET</code>, <code>TERRAFORM_OPEN_SOURCE</code>, <code>TERRAFORM_CLOUD</code>, and
         *        <code>EXTERNAL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedProductType(String provisionedProductType);

        /**
         * <p>
         * The record type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PROVISION_PRODUCT</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATE_PROVISIONED_PRODUCT</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TERMINATE_PROVISIONED_PRODUCT</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param recordType
         *        The record type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PROVISION_PRODUCT</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATE_PROVISIONED_PRODUCT</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TERMINATE_PROVISIONED_PRODUCT</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordType(String recordType);

        /**
         * <p>
         * The identifier of the provisioned product.
         * </p>
         * 
         * @param provisionedProductId
         *        The identifier of the provisioned product.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedProductId(String provisionedProductId);

        /**
         * <p>
         * The product identifier.
         * </p>
         * 
         * @param productId
         *        The product identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productId(String productId);

        /**
         * <p>
         * The identifier of the provisioning artifact.
         * </p>
         * 
         * @param provisioningArtifactId
         *        The identifier of the provisioning artifact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisioningArtifactId(String provisioningArtifactId);

        /**
         * <p>
         * The path identifier.
         * </p>
         * 
         * @param pathId
         *        The path identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pathId(String pathId);

        /**
         * <p>
         * The errors that occurred.
         * </p>
         * 
         * @param recordErrors
         *        The errors that occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordErrors(Collection<RecordError> recordErrors);

        /**
         * <p>
         * The errors that occurred.
         * </p>
         * 
         * @param recordErrors
         *        The errors that occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordErrors(RecordError... recordErrors);

        /**
         * <p>
         * The errors that occurred.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.servicecatalog.model.RecordError.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.servicecatalog.model.RecordError#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.servicecatalog.model.RecordError.Builder#build()} is called
         * immediately and its result is passed to {@link #recordErrors(List<RecordError>)}.
         * 
         * @param recordErrors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.servicecatalog.model.RecordError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordErrors(java.util.Collection<RecordError>)
         */
        Builder recordErrors(Consumer<RecordError.Builder>... recordErrors);

        /**
         * <p>
         * One or more tags.
         * </p>
         * 
         * @param recordTags
         *        One or more tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordTags(Collection<RecordTag> recordTags);

        /**
         * <p>
         * One or more tags.
         * </p>
         * 
         * @param recordTags
         *        One or more tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordTags(RecordTag... recordTags);

        /**
         * <p>
         * One or more tags.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.servicecatalog.model.RecordTag.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.servicecatalog.model.RecordTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.servicecatalog.model.RecordTag.Builder#build()} is called immediately
         * and its result is passed to {@link #recordTags(List<RecordTag>)}.
         * 
         * @param recordTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.servicecatalog.model.RecordTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordTags(java.util.Collection<RecordTag>)
         */
        Builder recordTags(Consumer<RecordTag.Builder>... recordTags);

        /**
         * <p>
         * The ARN of the launch role associated with the provisioned product.
         * </p>
         * 
         * @param launchRoleArn
         *        The ARN of the launch role associated with the provisioned product.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchRoleArn(String launchRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private String recordId;

        private String provisionedProductName;

        private String status;

        private Instant createdTime;

        private Instant updatedTime;

        private String provisionedProductType;

        private String recordType;

        private String provisionedProductId;

        private String productId;

        private String provisioningArtifactId;

        private String pathId;

        private List<RecordError> recordErrors = DefaultSdkAutoConstructList.getInstance();

        private List<RecordTag> recordTags = DefaultSdkAutoConstructList.getInstance();

        private String launchRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(RecordDetail model) {
            recordId(model.recordId);
            provisionedProductName(model.provisionedProductName);
            status(model.status);
            createdTime(model.createdTime);
            updatedTime(model.updatedTime);
            provisionedProductType(model.provisionedProductType);
            recordType(model.recordType);
            provisionedProductId(model.provisionedProductId);
            productId(model.productId);
            provisioningArtifactId(model.provisioningArtifactId);
            pathId(model.pathId);
            recordErrors(model.recordErrors);
            recordTags(model.recordTags);
            launchRoleArn(model.launchRoleArn);
        }

        public final String getRecordId() {
            return recordId;
        }

        public final void setRecordId(String recordId) {
            this.recordId = recordId;
        }

        @Override
        public final Builder recordId(String recordId) {
            this.recordId = recordId;
            return this;
        }

        public final String getProvisionedProductName() {
            return provisionedProductName;
        }

        public final void setProvisionedProductName(String provisionedProductName) {
            this.provisionedProductName = provisionedProductName;
        }

        @Override
        public final Builder provisionedProductName(String provisionedProductName) {
            this.provisionedProductName = provisionedProductName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(RecordStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getUpdatedTime() {
            return updatedTime;
        }

        public final void setUpdatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
        }

        @Override
        public final Builder updatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
            return this;
        }

        public final String getProvisionedProductType() {
            return provisionedProductType;
        }

        public final void setProvisionedProductType(String provisionedProductType) {
            this.provisionedProductType = provisionedProductType;
        }

        @Override
        public final Builder provisionedProductType(String provisionedProductType) {
            this.provisionedProductType = provisionedProductType;
            return this;
        }

        public final String getRecordType() {
            return recordType;
        }

        public final void setRecordType(String recordType) {
            this.recordType = recordType;
        }

        @Override
        public final Builder recordType(String recordType) {
            this.recordType = recordType;
            return this;
        }

        public final String getProvisionedProductId() {
            return provisionedProductId;
        }

        public final void setProvisionedProductId(String provisionedProductId) {
            this.provisionedProductId = provisionedProductId;
        }

        @Override
        public final Builder provisionedProductId(String provisionedProductId) {
            this.provisionedProductId = provisionedProductId;
            return this;
        }

        public final String getProductId() {
            return productId;
        }

        public final void setProductId(String productId) {
            this.productId = productId;
        }

        @Override
        public final Builder productId(String productId) {
            this.productId = productId;
            return this;
        }

        public final String getProvisioningArtifactId() {
            return provisioningArtifactId;
        }

        public final void setProvisioningArtifactId(String provisioningArtifactId) {
            this.provisioningArtifactId = provisioningArtifactId;
        }

        @Override
        public final Builder provisioningArtifactId(String provisioningArtifactId) {
            this.provisioningArtifactId = provisioningArtifactId;
            return this;
        }

        public final String getPathId() {
            return pathId;
        }

        public final void setPathId(String pathId) {
            this.pathId = pathId;
        }

        @Override
        public final Builder pathId(String pathId) {
            this.pathId = pathId;
            return this;
        }

        public final List<RecordError.Builder> getRecordErrors() {
            List<RecordError.Builder> result = RecordErrorsCopier.copyToBuilder(this.recordErrors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecordErrors(Collection<RecordError.BuilderImpl> recordErrors) {
            this.recordErrors = RecordErrorsCopier.copyFromBuilder(recordErrors);
        }

        @Override
        public final Builder recordErrors(Collection<RecordError> recordErrors) {
            this.recordErrors = RecordErrorsCopier.copy(recordErrors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordErrors(RecordError... recordErrors) {
            recordErrors(Arrays.asList(recordErrors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordErrors(Consumer<RecordError.Builder>... recordErrors) {
            recordErrors(Stream.of(recordErrors).map(c -> RecordError.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<RecordTag.Builder> getRecordTags() {
            List<RecordTag.Builder> result = RecordTagsCopier.copyToBuilder(this.recordTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecordTags(Collection<RecordTag.BuilderImpl> recordTags) {
            this.recordTags = RecordTagsCopier.copyFromBuilder(recordTags);
        }

        @Override
        public final Builder recordTags(Collection<RecordTag> recordTags) {
            this.recordTags = RecordTagsCopier.copy(recordTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordTags(RecordTag... recordTags) {
            recordTags(Arrays.asList(recordTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordTags(Consumer<RecordTag.Builder>... recordTags) {
            recordTags(Stream.of(recordTags).map(c -> RecordTag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getLaunchRoleArn() {
            return launchRoleArn;
        }

        public final void setLaunchRoleArn(String launchRoleArn) {
            this.launchRoleArn = launchRoleArn;
        }

        @Override
        public final Builder launchRoleArn(String launchRoleArn) {
            this.launchRoleArn = launchRoleArn;
            return this;
        }

        @Override
        public RecordDetail build() {
            return new RecordDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
