/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a scheduled action. You can use a scheduled action to trigger some Amazon Redshift API operations on a
 * schedule. For information about which API operations can be scheduled, see <a>ScheduledActionType</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ModifyScheduledActionResponse extends RedshiftResponse implements
        ToCopyableBuilder<ModifyScheduledActionResponse.Builder, ModifyScheduledActionResponse> {
    private static final SdkField<String> SCHEDULED_ACTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScheduledActionName").getter(getter(ModifyScheduledActionResponse::scheduledActionName))
            .setter(setter(Builder::scheduledActionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduledActionName").build())
            .build();

    private static final SdkField<ScheduledActionType> TARGET_ACTION_FIELD = SdkField
            .<ScheduledActionType> builder(MarshallingType.SDK_POJO).memberName("TargetAction")
            .getter(getter(ModifyScheduledActionResponse::targetAction)).setter(setter(Builder::targetAction))
            .constructor(ScheduledActionType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetAction").build()).build();

    private static final SdkField<String> SCHEDULE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Schedule").getter(getter(ModifyScheduledActionResponse::schedule)).setter(setter(Builder::schedule))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schedule").build()).build();

    private static final SdkField<String> IAM_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRole").getter(getter(ModifyScheduledActionResponse::iamRole)).setter(setter(Builder::iamRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRole").build()).build();

    private static final SdkField<String> SCHEDULED_ACTION_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ScheduledActionDescription")
            .getter(getter(ModifyScheduledActionResponse::scheduledActionDescription))
            .setter(setter(Builder::scheduledActionDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduledActionDescription").build())
            .build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(ModifyScheduledActionResponse::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<List<Instant>> NEXT_INVOCATIONS_FIELD = SdkField
            .<List<Instant>> builder(MarshallingType.LIST)
            .memberName("NextInvocations")
            .getter(getter(ModifyScheduledActionResponse::nextInvocations))
            .setter(setter(Builder::nextInvocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextInvocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("ScheduledActionTime")
                            .memberFieldInfo(
                                    SdkField.<Instant> builder(MarshallingType.INSTANT)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("ScheduledActionTime").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(ModifyScheduledActionResponse::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(ModifyScheduledActionResponse::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCHEDULED_ACTION_NAME_FIELD,
            TARGET_ACTION_FIELD, SCHEDULE_FIELD, IAM_ROLE_FIELD, SCHEDULED_ACTION_DESCRIPTION_FIELD, STATE_FIELD,
            NEXT_INVOCATIONS_FIELD, START_TIME_FIELD, END_TIME_FIELD));

    private final String scheduledActionName;

    private final ScheduledActionType targetAction;

    private final String schedule;

    private final String iamRole;

    private final String scheduledActionDescription;

    private final String state;

    private final List<Instant> nextInvocations;

    private final Instant startTime;

    private final Instant endTime;

    private ModifyScheduledActionResponse(BuilderImpl builder) {
        super(builder);
        this.scheduledActionName = builder.scheduledActionName;
        this.targetAction = builder.targetAction;
        this.schedule = builder.schedule;
        this.iamRole = builder.iamRole;
        this.scheduledActionDescription = builder.scheduledActionDescription;
        this.state = builder.state;
        this.nextInvocations = builder.nextInvocations;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
    }

    /**
     * <p>
     * The name of the scheduled action.
     * </p>
     * 
     * @return The name of the scheduled action.
     */
    public final String scheduledActionName() {
        return scheduledActionName;
    }

    /**
     * <p>
     * A JSON format string of the Amazon Redshift API operation with input parameters.
     * </p>
     * <p>
     * "
     * <code>{\"ResizeCluster\":{\"NodeType\":\"ra3.4xlarge\",\"ClusterIdentifier\":\"my-test-cluster\",\"NumberOfNodes\":3}}</code>
     * ".
     * </p>
     * 
     * @return A JSON format string of the Amazon Redshift API operation with input parameters. </p>
     *         <p>
     *         "
     *         <code>{\"ResizeCluster\":{\"NodeType\":\"ra3.4xlarge\",\"ClusterIdentifier\":\"my-test-cluster\",\"NumberOfNodes\":3}}</code>
     *         ".
     */
    public final ScheduledActionType targetAction() {
        return targetAction;
    }

    /**
     * <p>
     * The schedule for a one-time (at format) or recurring (cron format) scheduled action. Schedule invocations must be
     * separated by at least one hour.
     * </p>
     * <p>
     * Format of at expressions is "<code>at(yyyy-mm-ddThh:mm:ss)</code>". For example, "
     * <code>at(2016-03-04T17:27:00)</code>".
     * </p>
     * <p>
     * Format of cron expressions is "<code>cron(Minutes Hours Day-of-month Month Day-of-week Year)</code>
     * ". For example, "<code>cron(0 10 ? * MON *)</code>
     * ". For more information, see <a href="https://docs.aws.amazon.com
     * /AmazonCloudWatch/latest/events/ScheduledEvents.html#CronExpressions">Cron Expressions</a> in the <i>Amazon
     * CloudWatch Events User Guide</i>.
     * </p>
     * 
     * @return The schedule for a one-time (at format) or recurring (cron format) scheduled action. Schedule invocations
     *         must be separated by at least one hour.</p>
     *         <p>
     *         Format of at expressions is "<code>at(yyyy-mm-ddThh:mm:ss)</code>". For example, "
     *         <code>at(2016-03-04T17:27:00)</code>".
     *         </p>
     *         <p>
     *         Format of cron expressions is "<code>cron(Minutes Hours Day-of-month Month Day-of-week Year)</code>
     *         ". For example, "<code>cron(0 10 ? * MON *)</code>
     *         ". For more information, see <a href="https://docs.aws.amazon
     *         .com/AmazonCloudWatch/latest/events/ScheduledEvents.html#CronExpressions">Cron Expressions</a> in the
     *         <i>Amazon CloudWatch Events User Guide</i>.
     */
    public final String schedule() {
        return schedule;
    }

    /**
     * <p>
     * The IAM role to assume to run the scheduled action. This IAM role must have permission to run the Amazon Redshift
     * API operation in the scheduled action. This IAM role must allow the Amazon Redshift scheduler (Principal
     * scheduler.redshift.amazonaws.com) to assume permissions on your behalf. For more information about the IAM role
     * to use with the Amazon Redshift scheduler, see <a
     * href="https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-identity-based.html">Using
     * Identity-Based Policies for Amazon Redshift</a> in the <i>Amazon Redshift Cluster Management Guide</i>.
     * </p>
     * 
     * @return The IAM role to assume to run the scheduled action. This IAM role must have permission to run the Amazon
     *         Redshift API operation in the scheduled action. This IAM role must allow the Amazon Redshift scheduler
     *         (Principal scheduler.redshift.amazonaws.com) to assume permissions on your behalf. For more information
     *         about the IAM role to use with the Amazon Redshift scheduler, see <a
     *         href="https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-identity-based.html"
     *         >Using Identity-Based Policies for Amazon Redshift</a> in the <i>Amazon Redshift Cluster Management
     *         Guide</i>.
     */
    public final String iamRole() {
        return iamRole;
    }

    /**
     * <p>
     * The description of the scheduled action.
     * </p>
     * 
     * @return The description of the scheduled action.
     */
    public final String scheduledActionDescription() {
        return scheduledActionDescription;
    }

    /**
     * <p>
     * The state of the scheduled action. For example, <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ScheduledActionState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the scheduled action. For example, <code>DISABLED</code>.
     * @see ScheduledActionState
     */
    public final ScheduledActionState state() {
        return ScheduledActionState.fromValue(state);
    }

    /**
     * <p>
     * The state of the scheduled action. For example, <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ScheduledActionState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the scheduled action. For example, <code>DISABLED</code>.
     * @see ScheduledActionState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the NextInvocations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNextInvocations() {
        return nextInvocations != null && !(nextInvocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of times when the scheduled action will run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNextInvocations} method.
     * </p>
     * 
     * @return List of times when the scheduled action will run.
     */
    public final List<Instant> nextInvocations() {
        return nextInvocations;
    }

    /**
     * <p>
     * The start time in UTC when the schedule is active. Before this time, the scheduled action does not trigger.
     * </p>
     * 
     * @return The start time in UTC when the schedule is active. Before this time, the scheduled action does not
     *         trigger.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The end time in UTC when the schedule is no longer active. After this time, the scheduled action does not
     * trigger.
     * </p>
     * 
     * @return The end time in UTC when the schedule is no longer active. After this time, the scheduled action does not
     *         trigger.
     */
    public final Instant endTime() {
        return endTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(scheduledActionName());
        hashCode = 31 * hashCode + Objects.hashCode(targetAction());
        hashCode = 31 * hashCode + Objects.hashCode(schedule());
        hashCode = 31 * hashCode + Objects.hashCode(iamRole());
        hashCode = 31 * hashCode + Objects.hashCode(scheduledActionDescription());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasNextInvocations() ? nextInvocations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModifyScheduledActionResponse)) {
            return false;
        }
        ModifyScheduledActionResponse other = (ModifyScheduledActionResponse) obj;
        return Objects.equals(scheduledActionName(), other.scheduledActionName())
                && Objects.equals(targetAction(), other.targetAction()) && Objects.equals(schedule(), other.schedule())
                && Objects.equals(iamRole(), other.iamRole())
                && Objects.equals(scheduledActionDescription(), other.scheduledActionDescription())
                && Objects.equals(stateAsString(), other.stateAsString()) && hasNextInvocations() == other.hasNextInvocations()
                && Objects.equals(nextInvocations(), other.nextInvocations()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ModifyScheduledActionResponse").add("ScheduledActionName", scheduledActionName())
                .add("TargetAction", targetAction()).add("Schedule", schedule()).add("IamRole", iamRole())
                .add("ScheduledActionDescription", scheduledActionDescription()).add("State", stateAsString())
                .add("NextInvocations", hasNextInvocations() ? nextInvocations() : null).add("StartTime", startTime())
                .add("EndTime", endTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ScheduledActionName":
            return Optional.ofNullable(clazz.cast(scheduledActionName()));
        case "TargetAction":
            return Optional.ofNullable(clazz.cast(targetAction()));
        case "Schedule":
            return Optional.ofNullable(clazz.cast(schedule()));
        case "IamRole":
            return Optional.ofNullable(clazz.cast(iamRole()));
        case "ScheduledActionDescription":
            return Optional.ofNullable(clazz.cast(scheduledActionDescription()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "NextInvocations":
            return Optional.ofNullable(clazz.cast(nextInvocations()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ModifyScheduledActionResponse, T> g) {
        return obj -> g.apply((ModifyScheduledActionResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RedshiftResponse.Builder, SdkPojo, CopyableBuilder<Builder, ModifyScheduledActionResponse> {
        /**
         * <p>
         * The name of the scheduled action.
         * </p>
         * 
         * @param scheduledActionName
         *        The name of the scheduled action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduledActionName(String scheduledActionName);

        /**
         * <p>
         * A JSON format string of the Amazon Redshift API operation with input parameters.
         * </p>
         * <p>
         * "
         * <code>{\"ResizeCluster\":{\"NodeType\":\"ra3.4xlarge\",\"ClusterIdentifier\":\"my-test-cluster\",\"NumberOfNodes\":3}}</code>
         * ".
         * </p>
         * 
         * @param targetAction
         *        A JSON format string of the Amazon Redshift API operation with input parameters. </p>
         *        <p>
         *        "
         *        <code>{\"ResizeCluster\":{\"NodeType\":\"ra3.4xlarge\",\"ClusterIdentifier\":\"my-test-cluster\",\"NumberOfNodes\":3}}</code>
         *        ".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetAction(ScheduledActionType targetAction);

        /**
         * <p>
         * A JSON format string of the Amazon Redshift API operation with input parameters.
         * </p>
         * <p>
         * "
         * <code>{\"ResizeCluster\":{\"NodeType\":\"ra3.4xlarge\",\"ClusterIdentifier\":\"my-test-cluster\",\"NumberOfNodes\":3}}</code>
         * ".
         * </p>
         * This is a convenience method that creates an instance of the {@link ScheduledActionType.Builder} avoiding the
         * need to create one manually via {@link ScheduledActionType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScheduledActionType.Builder#build()} is called immediately and
         * its result is passed to {@link #targetAction(ScheduledActionType)}.
         * 
         * @param targetAction
         *        a consumer that will call methods on {@link ScheduledActionType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetAction(ScheduledActionType)
         */
        default Builder targetAction(Consumer<ScheduledActionType.Builder> targetAction) {
            return targetAction(ScheduledActionType.builder().applyMutation(targetAction).build());
        }

        /**
         * <p>
         * The schedule for a one-time (at format) or recurring (cron format) scheduled action. Schedule invocations
         * must be separated by at least one hour.
         * </p>
         * <p>
         * Format of at expressions is "<code>at(yyyy-mm-ddThh:mm:ss)</code>". For example, "
         * <code>at(2016-03-04T17:27:00)</code>".
         * </p>
         * <p>
         * Format of cron expressions is "<code>cron(Minutes Hours Day-of-month Month Day-of-week Year)</code>
         * ". For example, "<code>cron(0 10 ? * MON *)</code>
         * ". For more information, see <a href="https://docs.aws.amazon
         * .com/AmazonCloudWatch/latest/events/ScheduledEvents.html#CronExpressions">Cron Expressions</a> in the
         * <i>Amazon CloudWatch Events User Guide</i>.
         * </p>
         * 
         * @param schedule
         *        The schedule for a one-time (at format) or recurring (cron format) scheduled action. Schedule
         *        invocations must be separated by at least one hour.</p>
         *        <p>
         *        Format of at expressions is "<code>at(yyyy-mm-ddThh:mm:ss)</code>". For example, "
         *        <code>at(2016-03-04T17:27:00)</code>".
         *        </p>
         *        <p>
         *        Format of cron expressions is "<code>cron(Minutes Hours Day-of-month Month Day-of-week Year)</code>
         *        ". For example, "<code>cron(0 10 ? * MON *)</code>
         *        ". For more information, see <a href="https://docs.aws
         *        .amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html#CronExpressions">Cron Expressions</a>
         *        in the <i>Amazon CloudWatch Events User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedule(String schedule);

        /**
         * <p>
         * The IAM role to assume to run the scheduled action. This IAM role must have permission to run the Amazon
         * Redshift API operation in the scheduled action. This IAM role must allow the Amazon Redshift scheduler
         * (Principal scheduler.redshift.amazonaws.com) to assume permissions on your behalf. For more information about
         * the IAM role to use with the Amazon Redshift scheduler, see <a
         * href="https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-identity-based.html">Using
         * Identity-Based Policies for Amazon Redshift</a> in the <i>Amazon Redshift Cluster Management Guide</i>.
         * </p>
         * 
         * @param iamRole
         *        The IAM role to assume to run the scheduled action. This IAM role must have permission to run the
         *        Amazon Redshift API operation in the scheduled action. This IAM role must allow the Amazon Redshift
         *        scheduler (Principal scheduler.redshift.amazonaws.com) to assume permissions on your behalf. For more
         *        information about the IAM role to use with the Amazon Redshift scheduler, see <a href=
         *        "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-identity-based.html"
         *        >Using Identity-Based Policies for Amazon Redshift</a> in the <i>Amazon Redshift Cluster Management
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRole(String iamRole);

        /**
         * <p>
         * The description of the scheduled action.
         * </p>
         * 
         * @param scheduledActionDescription
         *        The description of the scheduled action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduledActionDescription(String scheduledActionDescription);

        /**
         * <p>
         * The state of the scheduled action. For example, <code>DISABLED</code>.
         * </p>
         * 
         * @param state
         *        The state of the scheduled action. For example, <code>DISABLED</code>.
         * @see ScheduledActionState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScheduledActionState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the scheduled action. For example, <code>DISABLED</code>.
         * </p>
         * 
         * @param state
         *        The state of the scheduled action. For example, <code>DISABLED</code>.
         * @see ScheduledActionState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScheduledActionState
         */
        Builder state(ScheduledActionState state);

        /**
         * <p>
         * List of times when the scheduled action will run.
         * </p>
         * 
         * @param nextInvocations
         *        List of times when the scheduled action will run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextInvocations(Collection<Instant> nextInvocations);

        /**
         * <p>
         * List of times when the scheduled action will run.
         * </p>
         * 
         * @param nextInvocations
         *        List of times when the scheduled action will run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextInvocations(Instant... nextInvocations);

        /**
         * <p>
         * The start time in UTC when the schedule is active. Before this time, the scheduled action does not trigger.
         * </p>
         * 
         * @param startTime
         *        The start time in UTC when the schedule is active. Before this time, the scheduled action does not
         *        trigger.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The end time in UTC when the schedule is no longer active. After this time, the scheduled action does not
         * trigger.
         * </p>
         * 
         * @param endTime
         *        The end time in UTC when the schedule is no longer active. After this time, the scheduled action does
         *        not trigger.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);
    }

    static final class BuilderImpl extends RedshiftResponse.BuilderImpl implements Builder {
        private String scheduledActionName;

        private ScheduledActionType targetAction;

        private String schedule;

        private String iamRole;

        private String scheduledActionDescription;

        private String state;

        private List<Instant> nextInvocations = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private Instant endTime;

        private BuilderImpl() {
        }

        private BuilderImpl(ModifyScheduledActionResponse model) {
            super(model);
            scheduledActionName(model.scheduledActionName);
            targetAction(model.targetAction);
            schedule(model.schedule);
            iamRole(model.iamRole);
            scheduledActionDescription(model.scheduledActionDescription);
            state(model.state);
            nextInvocations(model.nextInvocations);
            startTime(model.startTime);
            endTime(model.endTime);
        }

        public final String getScheduledActionName() {
            return scheduledActionName;
        }

        public final void setScheduledActionName(String scheduledActionName) {
            this.scheduledActionName = scheduledActionName;
        }

        @Override
        public final Builder scheduledActionName(String scheduledActionName) {
            this.scheduledActionName = scheduledActionName;
            return this;
        }

        public final ScheduledActionType.Builder getTargetAction() {
            return targetAction != null ? targetAction.toBuilder() : null;
        }

        public final void setTargetAction(ScheduledActionType.BuilderImpl targetAction) {
            this.targetAction = targetAction != null ? targetAction.build() : null;
        }

        @Override
        public final Builder targetAction(ScheduledActionType targetAction) {
            this.targetAction = targetAction;
            return this;
        }

        public final String getSchedule() {
            return schedule;
        }

        public final void setSchedule(String schedule) {
            this.schedule = schedule;
        }

        @Override
        public final Builder schedule(String schedule) {
            this.schedule = schedule;
            return this;
        }

        public final String getIamRole() {
            return iamRole;
        }

        public final void setIamRole(String iamRole) {
            this.iamRole = iamRole;
        }

        @Override
        public final Builder iamRole(String iamRole) {
            this.iamRole = iamRole;
            return this;
        }

        public final String getScheduledActionDescription() {
            return scheduledActionDescription;
        }

        public final void setScheduledActionDescription(String scheduledActionDescription) {
            this.scheduledActionDescription = scheduledActionDescription;
        }

        @Override
        public final Builder scheduledActionDescription(String scheduledActionDescription) {
            this.scheduledActionDescription = scheduledActionDescription;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ScheduledActionState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Collection<Instant> getNextInvocations() {
            if (nextInvocations instanceof SdkAutoConstructList) {
                return null;
            }
            return nextInvocations;
        }

        public final void setNextInvocations(Collection<Instant> nextInvocations) {
            this.nextInvocations = ScheduledActionTimeListCopier.copy(nextInvocations);
        }

        @Override
        public final Builder nextInvocations(Collection<Instant> nextInvocations) {
            this.nextInvocations = ScheduledActionTimeListCopier.copy(nextInvocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nextInvocations(Instant... nextInvocations) {
            nextInvocations(Arrays.asList(nextInvocations));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        @Override
        public ModifyScheduledActionResponse build() {
            return new ModifyScheduledActionResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
