/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that determines what the type of layout will be used on a sheet.
 * </p>
 * <p>
 * This is a union type structure. For this structure to be valid, only one of the attributes can be defined.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LayoutConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LayoutConfiguration.Builder, LayoutConfiguration> {
    private static final SdkField<GridLayoutConfiguration> GRID_LAYOUT_FIELD = SdkField
            .<GridLayoutConfiguration> builder(MarshallingType.SDK_POJO).memberName("GridLayout")
            .getter(getter(LayoutConfiguration::gridLayout)).setter(setter(Builder::gridLayout))
            .constructor(GridLayoutConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GridLayout").build()).build();

    private static final SdkField<FreeFormLayoutConfiguration> FREE_FORM_LAYOUT_FIELD = SdkField
            .<FreeFormLayoutConfiguration> builder(MarshallingType.SDK_POJO).memberName("FreeFormLayout")
            .getter(getter(LayoutConfiguration::freeFormLayout)).setter(setter(Builder::freeFormLayout))
            .constructor(FreeFormLayoutConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FreeFormLayout").build()).build();

    private static final SdkField<SectionBasedLayoutConfiguration> SECTION_BASED_LAYOUT_FIELD = SdkField
            .<SectionBasedLayoutConfiguration> builder(MarshallingType.SDK_POJO).memberName("SectionBasedLayout")
            .getter(getter(LayoutConfiguration::sectionBasedLayout)).setter(setter(Builder::sectionBasedLayout))
            .constructor(SectionBasedLayoutConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SectionBasedLayout").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GRID_LAYOUT_FIELD,
            FREE_FORM_LAYOUT_FIELD, SECTION_BASED_LAYOUT_FIELD));

    private static final long serialVersionUID = 1L;

    private final GridLayoutConfiguration gridLayout;

    private final FreeFormLayoutConfiguration freeFormLayout;

    private final SectionBasedLayoutConfiguration sectionBasedLayout;

    private LayoutConfiguration(BuilderImpl builder) {
        this.gridLayout = builder.gridLayout;
        this.freeFormLayout = builder.freeFormLayout;
        this.sectionBasedLayout = builder.sectionBasedLayout;
    }

    /**
     * <p>
     * A type of layout that can be used on a sheet. In a grid layout, visuals snap to a grid with standard spacing and
     * alignment. Dashboards are displayed as designed, with options to fit to screen or view at actual size. A grid
     * layout can be configured to behave in one of two ways when the viewport is resized: <code>FIXED</code> or
     * <code>RESPONSIVE</code>.
     * </p>
     * 
     * @return A type of layout that can be used on a sheet. In a grid layout, visuals snap to a grid with standard
     *         spacing and alignment. Dashboards are displayed as designed, with options to fit to screen or view at
     *         actual size. A grid layout can be configured to behave in one of two ways when the viewport is resized:
     *         <code>FIXED</code> or <code>RESPONSIVE</code>.
     */
    public final GridLayoutConfiguration gridLayout() {
        return gridLayout;
    }

    /**
     * <p>
     * A free-form is optimized for a fixed width and has more control over the exact placement of layout elements.
     * </p>
     * 
     * @return A free-form is optimized for a fixed width and has more control over the exact placement of layout
     *         elements.
     */
    public final FreeFormLayoutConfiguration freeFormLayout() {
        return freeFormLayout;
    }

    /**
     * <p>
     * A section based layout organizes visuals into multiple sections and has customized header, footer and page break.
     * </p>
     * 
     * @return A section based layout organizes visuals into multiple sections and has customized header, footer and
     *         page break.
     */
    public final SectionBasedLayoutConfiguration sectionBasedLayout() {
        return sectionBasedLayout;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gridLayout());
        hashCode = 31 * hashCode + Objects.hashCode(freeFormLayout());
        hashCode = 31 * hashCode + Objects.hashCode(sectionBasedLayout());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LayoutConfiguration)) {
            return false;
        }
        LayoutConfiguration other = (LayoutConfiguration) obj;
        return Objects.equals(gridLayout(), other.gridLayout()) && Objects.equals(freeFormLayout(), other.freeFormLayout())
                && Objects.equals(sectionBasedLayout(), other.sectionBasedLayout());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LayoutConfiguration").add("GridLayout", gridLayout()).add("FreeFormLayout", freeFormLayout())
                .add("SectionBasedLayout", sectionBasedLayout()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GridLayout":
            return Optional.ofNullable(clazz.cast(gridLayout()));
        case "FreeFormLayout":
            return Optional.ofNullable(clazz.cast(freeFormLayout()));
        case "SectionBasedLayout":
            return Optional.ofNullable(clazz.cast(sectionBasedLayout()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LayoutConfiguration, T> g) {
        return obj -> g.apply((LayoutConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LayoutConfiguration> {
        /**
         * <p>
         * A type of layout that can be used on a sheet. In a grid layout, visuals snap to a grid with standard spacing
         * and alignment. Dashboards are displayed as designed, with options to fit to screen or view at actual size. A
         * grid layout can be configured to behave in one of two ways when the viewport is resized: <code>FIXED</code>
         * or <code>RESPONSIVE</code>.
         * </p>
         * 
         * @param gridLayout
         *        A type of layout that can be used on a sheet. In a grid layout, visuals snap to a grid with standard
         *        spacing and alignment. Dashboards are displayed as designed, with options to fit to screen or view at
         *        actual size. A grid layout can be configured to behave in one of two ways when the viewport is
         *        resized: <code>FIXED</code> or <code>RESPONSIVE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gridLayout(GridLayoutConfiguration gridLayout);

        /**
         * <p>
         * A type of layout that can be used on a sheet. In a grid layout, visuals snap to a grid with standard spacing
         * and alignment. Dashboards are displayed as designed, with options to fit to screen or view at actual size. A
         * grid layout can be configured to behave in one of two ways when the viewport is resized: <code>FIXED</code>
         * or <code>RESPONSIVE</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link GridLayoutConfiguration.Builder} avoiding
         * the need to create one manually via {@link GridLayoutConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GridLayoutConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #gridLayout(GridLayoutConfiguration)}.
         * 
         * @param gridLayout
         *        a consumer that will call methods on {@link GridLayoutConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gridLayout(GridLayoutConfiguration)
         */
        default Builder gridLayout(Consumer<GridLayoutConfiguration.Builder> gridLayout) {
            return gridLayout(GridLayoutConfiguration.builder().applyMutation(gridLayout).build());
        }

        /**
         * <p>
         * A free-form is optimized for a fixed width and has more control over the exact placement of layout elements.
         * </p>
         * 
         * @param freeFormLayout
         *        A free-form is optimized for a fixed width and has more control over the exact placement of layout
         *        elements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder freeFormLayout(FreeFormLayoutConfiguration freeFormLayout);

        /**
         * <p>
         * A free-form is optimized for a fixed width and has more control over the exact placement of layout elements.
         * </p>
         * This is a convenience method that creates an instance of the {@link FreeFormLayoutConfiguration.Builder}
         * avoiding the need to create one manually via {@link FreeFormLayoutConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FreeFormLayoutConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #freeFormLayout(FreeFormLayoutConfiguration)}.
         * 
         * @param freeFormLayout
         *        a consumer that will call methods on {@link FreeFormLayoutConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #freeFormLayout(FreeFormLayoutConfiguration)
         */
        default Builder freeFormLayout(Consumer<FreeFormLayoutConfiguration.Builder> freeFormLayout) {
            return freeFormLayout(FreeFormLayoutConfiguration.builder().applyMutation(freeFormLayout).build());
        }

        /**
         * <p>
         * A section based layout organizes visuals into multiple sections and has customized header, footer and page
         * break.
         * </p>
         * 
         * @param sectionBasedLayout
         *        A section based layout organizes visuals into multiple sections and has customized header, footer and
         *        page break.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sectionBasedLayout(SectionBasedLayoutConfiguration sectionBasedLayout);

        /**
         * <p>
         * A section based layout organizes visuals into multiple sections and has customized header, footer and page
         * break.
         * </p>
         * This is a convenience method that creates an instance of the {@link SectionBasedLayoutConfiguration.Builder}
         * avoiding the need to create one manually via {@link SectionBasedLayoutConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SectionBasedLayoutConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #sectionBasedLayout(SectionBasedLayoutConfiguration)}.
         * 
         * @param sectionBasedLayout
         *        a consumer that will call methods on {@link SectionBasedLayoutConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sectionBasedLayout(SectionBasedLayoutConfiguration)
         */
        default Builder sectionBasedLayout(Consumer<SectionBasedLayoutConfiguration.Builder> sectionBasedLayout) {
            return sectionBasedLayout(SectionBasedLayoutConfiguration.builder().applyMutation(sectionBasedLayout).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private GridLayoutConfiguration gridLayout;

        private FreeFormLayoutConfiguration freeFormLayout;

        private SectionBasedLayoutConfiguration sectionBasedLayout;

        private BuilderImpl() {
        }

        private BuilderImpl(LayoutConfiguration model) {
            gridLayout(model.gridLayout);
            freeFormLayout(model.freeFormLayout);
            sectionBasedLayout(model.sectionBasedLayout);
        }

        public final GridLayoutConfiguration.Builder getGridLayout() {
            return gridLayout != null ? gridLayout.toBuilder() : null;
        }

        public final void setGridLayout(GridLayoutConfiguration.BuilderImpl gridLayout) {
            this.gridLayout = gridLayout != null ? gridLayout.build() : null;
        }

        @Override
        public final Builder gridLayout(GridLayoutConfiguration gridLayout) {
            this.gridLayout = gridLayout;
            return this;
        }

        public final FreeFormLayoutConfiguration.Builder getFreeFormLayout() {
            return freeFormLayout != null ? freeFormLayout.toBuilder() : null;
        }

        public final void setFreeFormLayout(FreeFormLayoutConfiguration.BuilderImpl freeFormLayout) {
            this.freeFormLayout = freeFormLayout != null ? freeFormLayout.build() : null;
        }

        @Override
        public final Builder freeFormLayout(FreeFormLayoutConfiguration freeFormLayout) {
            this.freeFormLayout = freeFormLayout;
            return this;
        }

        public final SectionBasedLayoutConfiguration.Builder getSectionBasedLayout() {
            return sectionBasedLayout != null ? sectionBasedLayout.toBuilder() : null;
        }

        public final void setSectionBasedLayout(SectionBasedLayoutConfiguration.BuilderImpl sectionBasedLayout) {
            this.sectionBasedLayout = sectionBasedLayout != null ? sectionBasedLayout.build() : null;
        }

        @Override
        public final Builder sectionBasedLayout(SectionBasedLayoutConfiguration sectionBasedLayout) {
            this.sectionBasedLayout = sectionBasedLayout;
            return this;
        }

        @Override
        public LayoutConfiguration build() {
            return new LayoutConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
