/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the status about a <a>CreateAccount</a> or <a>CreateGovCloudAccount</a> request to create an Amazon Web
 * Services account or an Amazon Web Services GovCloud (US) account in an organization.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAccountStatus implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateAccountStatus.Builder, CreateAccountStatus> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(CreateAccountStatus::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ACCOUNT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountName").getter(getter(CreateAccountStatus::accountName)).setter(setter(Builder::accountName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountName").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(CreateAccountStatus::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<Instant> REQUESTED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RequestedTimestamp").getter(getter(CreateAccountStatus::requestedTimestamp))
            .setter(setter(Builder::requestedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedTimestamp").build())
            .build();

    private static final SdkField<Instant> COMPLETED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletedTimestamp").getter(getter(CreateAccountStatus::completedTimestamp))
            .setter(setter(Builder::completedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletedTimestamp").build())
            .build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(CreateAccountStatus::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> GOV_CLOUD_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GovCloudAccountId").getter(getter(CreateAccountStatus::govCloudAccountId))
            .setter(setter(Builder::govCloudAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GovCloudAccountId").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(CreateAccountStatus::failureReasonAsString))
            .setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ACCOUNT_NAME_FIELD,
            STATE_FIELD, REQUESTED_TIMESTAMP_FIELD, COMPLETED_TIMESTAMP_FIELD, ACCOUNT_ID_FIELD, GOV_CLOUD_ACCOUNT_ID_FIELD,
            FAILURE_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String accountName;

    private final String state;

    private final Instant requestedTimestamp;

    private final Instant completedTimestamp;

    private final String accountId;

    private final String govCloudAccountId;

    private final String failureReason;

    private CreateAccountStatus(BuilderImpl builder) {
        this.id = builder.id;
        this.accountName = builder.accountName;
        this.state = builder.state;
        this.requestedTimestamp = builder.requestedTimestamp;
        this.completedTimestamp = builder.completedTimestamp;
        this.accountId = builder.accountId;
        this.govCloudAccountId = builder.govCloudAccountId;
        this.failureReason = builder.failureReason;
    }

    /**
     * <p>
     * The unique identifier (ID) that references this request. You get this value from the response of the initial
     * <a>CreateAccount</a> request to create the account.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a create account request ID string requires
     * "car-" followed by from 8 to 32 lowercase letters or digits.
     * </p>
     * 
     * @return The unique identifier (ID) that references this request. You get this value from the response of the
     *         initial <a>CreateAccount</a> request to create the account.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a create account request ID string
     *         requires "car-" followed by from 8 to 32 lowercase letters or digits.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The account name given to the account when it was created.
     * </p>
     * 
     * @return The account name given to the account when it was created.
     */
    public final String accountName() {
        return accountName;
    }

    /**
     * <p>
     * The status of the asynchronous request to create an Amazon Web Services account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CreateAccountState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The status of the asynchronous request to create an Amazon Web Services account.
     * @see CreateAccountState
     */
    public final CreateAccountState state() {
        return CreateAccountState.fromValue(state);
    }

    /**
     * <p>
     * The status of the asynchronous request to create an Amazon Web Services account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CreateAccountState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The status of the asynchronous request to create an Amazon Web Services account.
     * @see CreateAccountState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The date and time that the request was made for the account creation.
     * </p>
     * 
     * @return The date and time that the request was made for the account creation.
     */
    public final Instant requestedTimestamp() {
        return requestedTimestamp;
    }

    /**
     * <p>
     * The date and time that the account was created and the request completed.
     * </p>
     * 
     * @return The date and time that the account was created and the request completed.
     */
    public final Instant completedTimestamp() {
        return completedTimestamp;
    }

    /**
     * <p>
     * If the account was created successfully, the unique identifier (ID) of the new account.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly 12
     * digits.
     * </p>
     * 
     * @return If the account was created successfully, the unique identifier (ID) of the new account.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly
     *         12 digits.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * If the account was created successfully, the unique identifier (ID) of the new account in the Amazon Web Services
     * GovCloud (US) Region.
     * </p>
     * 
     * @return If the account was created successfully, the unique identifier (ID) of the new account in the Amazon Web
     *         Services GovCloud (US) Region.
     */
    public final String govCloudAccountId() {
        return govCloudAccountId;
    }

    /**
     * <p>
     * If the request failed, a description of the reason for the failure.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of accounts
     * in your organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
     * </p>
     * </li>
     * <li>
     * <p>
     * EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account with that
     * email address already exists.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to receive
     * business license validation.
     * </p>
     * </li>
     * <li>
     * <p>
     * GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not be created
     * because this Region already includes an account with that email address.
     * </p>
     * </li>
     * <li>
     * <p>
     * IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization can't complete
     * business license validation because it doesn't have valid identity data.
     * </p>
     * </li>
     * <li>
     * <p>
     * INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
     * </p>
     * </li>
     * <li>
     * <p>
     * INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
     * </p>
     * </li>
     * <li>
     * <p>
     * INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have a supported
     * payment method associated with the account. Amazon Web Services does not support cards issued by financial
     * institutions in Russia or Belarus. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your Amazon Web
     * Services payments</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If the
     * problem persists, contact Amazon Web Services Customer Support.
     * </p>
     * </li>
     * <li>
     * <p>
     * MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not received
     * Business Validation.
     * </p>
     * </li>
     * <li>
     * <p>
     * MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method, such as a
     * credit card.
     * </p>
     * </li>
     * <li>
     * <p>
     * PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in the process
     * of completing business license validation.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an unknown issue
     * with business license validation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link CreateAccountFailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #failureReasonAsString}.
     * </p>
     * 
     * @return If the request failed, a description of the reason for the failure.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of
     *         accounts in your organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account with
     *         that email address already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to receive
     *         business license validation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not be
     *         created because this Region already includes an account with that email address.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization can't
     *         complete business license validation because it doesn't have valid identity data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have a
     *         supported payment method associated with the account. Amazon Web Services does not support cards issued
     *         by financial institutions in Russia or Belarus. For more information, see <a
     *         href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your
     *         Amazon Web Services payments</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If
     *         the problem persists, contact Amazon Web Services Customer Support.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not received
     *         Business Validation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method, such
     *         as a credit card.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in the
     *         process of completing business license validation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an unknown
     *         issue with business license validation.
     *         </p>
     *         </li>
     * @see CreateAccountFailureReason
     */
    public final CreateAccountFailureReason failureReason() {
        return CreateAccountFailureReason.fromValue(failureReason);
    }

    /**
     * <p>
     * If the request failed, a description of the reason for the failure.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of accounts
     * in your organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
     * </p>
     * </li>
     * <li>
     * <p>
     * EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account with that
     * email address already exists.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to receive
     * business license validation.
     * </p>
     * </li>
     * <li>
     * <p>
     * GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not be created
     * because this Region already includes an account with that email address.
     * </p>
     * </li>
     * <li>
     * <p>
     * IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization can't complete
     * business license validation because it doesn't have valid identity data.
     * </p>
     * </li>
     * <li>
     * <p>
     * INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
     * </p>
     * </li>
     * <li>
     * <p>
     * INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
     * </p>
     * </li>
     * <li>
     * <p>
     * INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have a supported
     * payment method associated with the account. Amazon Web Services does not support cards issued by financial
     * institutions in Russia or Belarus. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your Amazon Web
     * Services payments</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If the
     * problem persists, contact Amazon Web Services Customer Support.
     * </p>
     * </li>
     * <li>
     * <p>
     * MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not received
     * Business Validation.
     * </p>
     * </li>
     * <li>
     * <p>
     * MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method, such as a
     * credit card.
     * </p>
     * </li>
     * <li>
     * <p>
     * PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in the process
     * of completing business license validation.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an unknown issue
     * with business license validation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link CreateAccountFailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #failureReasonAsString}.
     * </p>
     * 
     * @return If the request failed, a description of the reason for the failure.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of
     *         accounts in your organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account with
     *         that email address already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to receive
     *         business license validation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not be
     *         created because this Region already includes an account with that email address.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization can't
     *         complete business license validation because it doesn't have valid identity data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have a
     *         supported payment method associated with the account. Amazon Web Services does not support cards issued
     *         by financial institutions in Russia or Belarus. For more information, see <a
     *         href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your
     *         Amazon Web Services payments</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If
     *         the problem persists, contact Amazon Web Services Customer Support.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not received
     *         Business Validation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method, such
     *         as a credit card.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in the
     *         process of completing business license validation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an unknown
     *         issue with business license validation.
     *         </p>
     *         </li>
     * @see CreateAccountFailureReason
     */
    public final String failureReasonAsString() {
        return failureReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(accountName());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(requestedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(completedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(govCloudAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(failureReasonAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAccountStatus)) {
            return false;
        }
        CreateAccountStatus other = (CreateAccountStatus) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(accountName(), other.accountName())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(requestedTimestamp(), other.requestedTimestamp())
                && Objects.equals(completedTimestamp(), other.completedTimestamp())
                && Objects.equals(accountId(), other.accountId())
                && Objects.equals(govCloudAccountId(), other.govCloudAccountId())
                && Objects.equals(failureReasonAsString(), other.failureReasonAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateAccountStatus").add("Id", id())
                .add("AccountName", accountName() == null ? null : "*** Sensitive Data Redacted ***")
                .add("State", stateAsString()).add("RequestedTimestamp", requestedTimestamp())
                .add("CompletedTimestamp", completedTimestamp()).add("AccountId", accountId())
                .add("GovCloudAccountId", govCloudAccountId()).add("FailureReason", failureReasonAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "AccountName":
            return Optional.ofNullable(clazz.cast(accountName()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "RequestedTimestamp":
            return Optional.ofNullable(clazz.cast(requestedTimestamp()));
        case "CompletedTimestamp":
            return Optional.ofNullable(clazz.cast(completedTimestamp()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "GovCloudAccountId":
            return Optional.ofNullable(clazz.cast(govCloudAccountId()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReasonAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateAccountStatus, T> g) {
        return obj -> g.apply((CreateAccountStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateAccountStatus> {
        /**
         * <p>
         * The unique identifier (ID) that references this request. You get this value from the response of the initial
         * <a>CreateAccount</a> request to create the account.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a create account request ID string
         * requires "car-" followed by from 8 to 32 lowercase letters or digits.
         * </p>
         * 
         * @param id
         *        The unique identifier (ID) that references this request. You get this value from the response of the
         *        initial <a>CreateAccount</a> request to create the account.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a create account request ID string
         *        requires "car-" followed by from 8 to 32 lowercase letters or digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The account name given to the account when it was created.
         * </p>
         * 
         * @param accountName
         *        The account name given to the account when it was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountName(String accountName);

        /**
         * <p>
         * The status of the asynchronous request to create an Amazon Web Services account.
         * </p>
         * 
         * @param state
         *        The status of the asynchronous request to create an Amazon Web Services account.
         * @see CreateAccountState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreateAccountState
         */
        Builder state(String state);

        /**
         * <p>
         * The status of the asynchronous request to create an Amazon Web Services account.
         * </p>
         * 
         * @param state
         *        The status of the asynchronous request to create an Amazon Web Services account.
         * @see CreateAccountState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreateAccountState
         */
        Builder state(CreateAccountState state);

        /**
         * <p>
         * The date and time that the request was made for the account creation.
         * </p>
         * 
         * @param requestedTimestamp
         *        The date and time that the request was made for the account creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedTimestamp(Instant requestedTimestamp);

        /**
         * <p>
         * The date and time that the account was created and the request completed.
         * </p>
         * 
         * @param completedTimestamp
         *        The date and time that the account was created and the request completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedTimestamp(Instant completedTimestamp);

        /**
         * <p>
         * If the account was created successfully, the unique identifier (ID) of the new account.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly 12
         * digits.
         * </p>
         * 
         * @param accountId
         *        If the account was created successfully, the unique identifier (ID) of the new account.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires
         *        exactly 12 digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * If the account was created successfully, the unique identifier (ID) of the new account in the Amazon Web
         * Services GovCloud (US) Region.
         * </p>
         * 
         * @param govCloudAccountId
         *        If the account was created successfully, the unique identifier (ID) of the new account in the Amazon
         *        Web Services GovCloud (US) Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder govCloudAccountId(String govCloudAccountId);

        /**
         * <p>
         * If the request failed, a description of the reason for the failure.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of
         * accounts in your organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
         * </p>
         * </li>
         * <li>
         * <p>
         * EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account with that
         * email address already exists.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to receive
         * business license validation.
         * </p>
         * </li>
         * <li>
         * <p>
         * GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not be
         * created because this Region already includes an account with that email address.
         * </p>
         * </li>
         * <li>
         * <p>
         * IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization can't
         * complete business license validation because it doesn't have valid identity data.
         * </p>
         * </li>
         * <li>
         * <p>
         * INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
         * </p>
         * </li>
         * <li>
         * <p>
         * INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
         * </p>
         * </li>
         * <li>
         * <p>
         * INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have a
         * supported payment method associated with the account. Amazon Web Services does not support cards issued by
         * financial institutions in Russia or Belarus. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your Amazon
         * Web Services payments</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If the
         * problem persists, contact Amazon Web Services Customer Support.
         * </p>
         * </li>
         * <li>
         * <p>
         * MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not received
         * Business Validation.
         * </p>
         * </li>
         * <li>
         * <p>
         * MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method, such as a
         * credit card.
         * </p>
         * </li>
         * <li>
         * <p>
         * PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in the
         * process of completing business license validation.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an unknown issue
         * with business license validation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureReason
         *        If the request failed, a description of the reason for the failure.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of
         *        accounts in your organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account
         *        with that email address already exists.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to
         *        receive business license validation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not
         *        be created because this Region already includes an account with that email address.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization
         *        can't complete business license validation because it doesn't have valid identity data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have
         *        a supported payment method associated with the account. Amazon Web Services does not support cards
         *        issued by financial institutions in Russia or Belarus. For more information, see <a
         *        href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your
         *        Amazon Web Services payments</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If
         *        the problem persists, contact Amazon Web Services Customer Support.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not
         *        received Business Validation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method,
         *        such as a credit card.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in
         *        the process of completing business license validation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an
         *        unknown issue with business license validation.
         *        </p>
         *        </li>
         * @see CreateAccountFailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreateAccountFailureReason
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * If the request failed, a description of the reason for the failure.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of
         * accounts in your organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
         * </p>
         * </li>
         * <li>
         * <p>
         * EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account with that
         * email address already exists.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to receive
         * business license validation.
         * </p>
         * </li>
         * <li>
         * <p>
         * GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not be
         * created because this Region already includes an account with that email address.
         * </p>
         * </li>
         * <li>
         * <p>
         * IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization can't
         * complete business license validation because it doesn't have valid identity data.
         * </p>
         * </li>
         * <li>
         * <p>
         * INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
         * </p>
         * </li>
         * <li>
         * <p>
         * INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
         * </p>
         * </li>
         * <li>
         * <p>
         * INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have a
         * supported payment method associated with the account. Amazon Web Services does not support cards issued by
         * financial institutions in Russia or Belarus. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your Amazon
         * Web Services payments</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If the
         * problem persists, contact Amazon Web Services Customer Support.
         * </p>
         * </li>
         * <li>
         * <p>
         * MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not received
         * Business Validation.
         * </p>
         * </li>
         * <li>
         * <p>
         * MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method, such as a
         * credit card.
         * </p>
         * </li>
         * <li>
         * <p>
         * PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in the
         * process of completing business license validation.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an unknown issue
         * with business license validation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureReason
         *        If the request failed, a description of the reason for the failure.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ACCOUNT_LIMIT_EXCEEDED: The account couldn't be created because you reached the limit on the number of
         *        accounts in your organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        CONCURRENT_ACCOUNT_MODIFICATION: You already submitted a request with the same information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EMAIL_ALREADY_EXISTS: The account could not be created because another Amazon Web Services account
         *        with that email address already exists.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization failed to
         *        receive business license validation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        GOVCLOUD_ACCOUNT_ALREADY_EXISTS: The account in the Amazon Web Services GovCloud (US) Region could not
         *        be created because this Region already includes an account with that email address.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IDENTITY_INVALID_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization
         *        can't complete business license validation because it doesn't have valid identity data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INVALID_ADDRESS: The account could not be created because the address you provided is not valid.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INVALID_EMAIL: The account could not be created because the email address you provided is not valid.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INVALID_PAYMENT_INSTRUMENT: The Amazon Web Services account that owns your organization does not have
         *        a supported payment method associated with the account. Amazon Web Services does not support cards
         *        issued by financial institutions in Russia or Belarus. For more information, see <a
         *        href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/manage-general.html">Managing your
         *        Amazon Web Services payments</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INTERNAL_FAILURE: The account could not be created because of an internal failure. Try again later. If
         *        the problem persists, contact Amazon Web Services Customer Support.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MISSING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has not
         *        received Business Validation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MISSING_PAYMENT_INSTRUMENT: You must configure the management account with a valid payment method,
         *        such as a credit card.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PENDING_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization is still in
         *        the process of completing business license validation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN_BUSINESS_VALIDATION: The Amazon Web Services account that owns your organization has an
         *        unknown issue with business license validation.
         *        </p>
         *        </li>
         * @see CreateAccountFailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreateAccountFailureReason
         */
        Builder failureReason(CreateAccountFailureReason failureReason);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String accountName;

        private String state;

        private Instant requestedTimestamp;

        private Instant completedTimestamp;

        private String accountId;

        private String govCloudAccountId;

        private String failureReason;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAccountStatus model) {
            id(model.id);
            accountName(model.accountName);
            state(model.state);
            requestedTimestamp(model.requestedTimestamp);
            completedTimestamp(model.completedTimestamp);
            accountId(model.accountId);
            govCloudAccountId(model.govCloudAccountId);
            failureReason(model.failureReason);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getAccountName() {
            return accountName;
        }

        public final void setAccountName(String accountName) {
            this.accountName = accountName;
        }

        @Override
        public final Builder accountName(String accountName) {
            this.accountName = accountName;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(CreateAccountState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Instant getRequestedTimestamp() {
            return requestedTimestamp;
        }

        public final void setRequestedTimestamp(Instant requestedTimestamp) {
            this.requestedTimestamp = requestedTimestamp;
        }

        @Override
        public final Builder requestedTimestamp(Instant requestedTimestamp) {
            this.requestedTimestamp = requestedTimestamp;
            return this;
        }

        public final Instant getCompletedTimestamp() {
            return completedTimestamp;
        }

        public final void setCompletedTimestamp(Instant completedTimestamp) {
            this.completedTimestamp = completedTimestamp;
        }

        @Override
        public final Builder completedTimestamp(Instant completedTimestamp) {
            this.completedTimestamp = completedTimestamp;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getGovCloudAccountId() {
            return govCloudAccountId;
        }

        public final void setGovCloudAccountId(String govCloudAccountId) {
            this.govCloudAccountId = govCloudAccountId;
        }

        @Override
        public final Builder govCloudAccountId(String govCloudAccountId) {
            this.govCloudAccountId = govCloudAccountId;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        @Override
        public final Builder failureReason(CreateAccountFailureReason failureReason) {
            this.failureReason(failureReason == null ? null : failureReason.toString());
            return this;
        }

        @Override
        public CreateAccountStatus build() {
            return new CreateAccountStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
