/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mobile;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mobile.internal.MobileServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mobile.model.AccountActionRequiredException;
import software.amazon.awssdk.services.mobile.model.BadRequestException;
import software.amazon.awssdk.services.mobile.model.CreateProjectRequest;
import software.amazon.awssdk.services.mobile.model.CreateProjectResponse;
import software.amazon.awssdk.services.mobile.model.DeleteProjectRequest;
import software.amazon.awssdk.services.mobile.model.DeleteProjectResponse;
import software.amazon.awssdk.services.mobile.model.DescribeBundleRequest;
import software.amazon.awssdk.services.mobile.model.DescribeBundleResponse;
import software.amazon.awssdk.services.mobile.model.DescribeProjectRequest;
import software.amazon.awssdk.services.mobile.model.DescribeProjectResponse;
import software.amazon.awssdk.services.mobile.model.ExportBundleRequest;
import software.amazon.awssdk.services.mobile.model.ExportBundleResponse;
import software.amazon.awssdk.services.mobile.model.ExportProjectRequest;
import software.amazon.awssdk.services.mobile.model.ExportProjectResponse;
import software.amazon.awssdk.services.mobile.model.InternalFailureException;
import software.amazon.awssdk.services.mobile.model.LimitExceededException;
import software.amazon.awssdk.services.mobile.model.ListBundlesRequest;
import software.amazon.awssdk.services.mobile.model.ListBundlesResponse;
import software.amazon.awssdk.services.mobile.model.ListProjectsRequest;
import software.amazon.awssdk.services.mobile.model.ListProjectsResponse;
import software.amazon.awssdk.services.mobile.model.MobileException;
import software.amazon.awssdk.services.mobile.model.NotFoundException;
import software.amazon.awssdk.services.mobile.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mobile.model.TooManyRequestsException;
import software.amazon.awssdk.services.mobile.model.UnauthorizedException;
import software.amazon.awssdk.services.mobile.model.UpdateProjectRequest;
import software.amazon.awssdk.services.mobile.model.UpdateProjectResponse;
import software.amazon.awssdk.services.mobile.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DescribeBundleRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ExportBundleRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ExportProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ListBundlesRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MobileClient}.
 *
 * @see MobileClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMobileClient implements MobileClient {
    private static final Logger log = Logger.loggerFor(DefaultMobileClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMobileClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an AWS Mobile Hub project.
     * </p>
     *
     * @param createProjectRequest
     *        Request structure used to request a project be created.
     * @return Result of the CreateProject operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws NotFoundException
     *         No entity can be found with the specified identifier.
     * @throws LimitExceededException
     *         There are too many AWS Mobile Hub projects in the account or the account has exceeded the maximum number
     *         of resources in some AWS service. You should create another sub-account using AWS Organizations or remove
     *         some resources and retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProjectResponse createProject(CreateProjectRequest createProjectRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, BadRequestException, NotFoundException,
            LimitExceededException, AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");

            return clientHandler.execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                    .withOperationName("CreateProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delets a project in AWS Mobile Hub.
     * </p>
     *
     * @param deleteProjectRequest
     *        Request structure used to request a project be deleted.
     * @return Result of the DeleteProject operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws NotFoundException
     *         No entity can be found with the specified identifier.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProjectResponse deleteProject(DeleteProjectRequest deleteProjectRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, NotFoundException, AwsServiceException,
            SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");

            return clientHandler.execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                    .withOperationName("DeleteProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the bundle details for the requested bundle id.
     * </p>
     *
     * @param describeBundleRequest
     *        Request structure to request the details of a specific bundle.
     * @return Result of the DescribeBundle operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws NotFoundException
     *         No entity can be found with the specified identifier.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.DescribeBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DescribeBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBundleResponse describeBundle(DescribeBundleRequest describeBundleRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, BadRequestException, NotFoundException,
            AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBundleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBundleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBundleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBundle");

            return clientHandler.execute(new ClientExecutionParams<DescribeBundleRequest, DescribeBundleResponse>()
                    .withOperationName("DescribeBundle").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeBundleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBundleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a project in AWS Mobile Hub.
     * </p>
     *
     * @param describeProjectRequest
     *        Request structure used to request details about a project.
     * @return Result of the DescribeProject operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws NotFoundException
     *         No entity can be found with the specified identifier.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DescribeProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeProjectResponse describeProject(DescribeProjectRequest describeProjectRequest)
            throws InternalFailureException, ServiceUnavailableException, UnauthorizedException, TooManyRequestsException,
            BadRequestException, NotFoundException, AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");

            return clientHandler.execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                    .withOperationName("DescribeProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generates customized software development kit (SDK) and or tool packages used to integrate mobile web or mobile
     * app clients with backend AWS resources.
     * </p>
     *
     * @param exportBundleRequest
     *        Request structure used to request generation of custom SDK and tool packages required to integrate mobile
     *        web or app clients with backed AWS resources.
     * @return Result of the ExportBundle operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws NotFoundException
     *         No entity can be found with the specified identifier.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.ExportBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ExportBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExportBundleResponse exportBundle(ExportBundleRequest exportBundleRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, BadRequestException, NotFoundException,
            AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportBundleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExportBundleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportBundleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportBundle");

            return clientHandler.execute(new ClientExecutionParams<ExportBundleRequest, ExportBundleResponse>()
                    .withOperationName("ExportBundle").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(exportBundleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExportBundleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports project configuration to a snapshot which can be downloaded and shared. Note that mobile app push
     * credentials are encrypted in exported projects, so they can only be shared successfully within the same AWS
     * account.
     * </p>
     *
     * @param exportProjectRequest
     *        Request structure used in requests to export project configuration details.
     * @return Result of the ExportProject operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws NotFoundException
     *         No entity can be found with the specified identifier.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.ExportProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ExportProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExportProjectResponse exportProject(ExportProjectRequest exportProjectRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, BadRequestException, NotFoundException,
            AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExportProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportProject");

            return clientHandler.execute(new ClientExecutionParams<ExportProjectRequest, ExportProjectResponse>()
                    .withOperationName("ExportProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(exportProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExportProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all available bundles.
     * </p>
     *
     * @param listBundlesRequest
     *        Request structure to request all available bundles.
     * @return Result of the ListBundles operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.ListBundles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListBundles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBundlesResponse listBundles(ListBundlesRequest listBundlesRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, BadRequestException,
            AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBundlesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBundlesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBundlesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBundlesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBundles");

            return clientHandler.execute(new ClientExecutionParams<ListBundlesRequest, ListBundlesResponse>()
                    .withOperationName("ListBundles").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listBundlesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBundlesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists projects in AWS Mobile Hub.
     * </p>
     *
     * @param listProjectsRequest
     *        Request structure used to request projects list in AWS Mobile Hub.
     * @return Result of the ListProjects operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProjectsResponse listProjects(ListProjectsRequest listProjectsRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, BadRequestException,
            AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProjectsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");

            return clientHandler.execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                    .withOperationName("ListProjects").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listProjectsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProjectsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an existing project.
     * </p>
     *
     * @param updateProjectRequest
     *        Request structure used for requests to update project configuration.
     * @return Result of the UpdateProject operation returned by the service.
     * @throws InternalFailureException
     *         The service has encountered an unexpected error condition which prevents it from servicing the request.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable. The request should be retried after some time delay.
     * @throws UnauthorizedException
     *         Credentials of the caller are insufficient to authorize the request.
     * @throws TooManyRequestsException
     *         Too many requests have been received for this AWS account in too short a time. The request should be
     *         retried after some time delay.
     * @throws BadRequestException
     *         The request cannot be processed because some parameter is not valid or the project state prevents the
     *         operation from being performed.
     * @throws NotFoundException
     *         No entity can be found with the specified identifier.
     * @throws AccountActionRequiredException
     *         Account Action is required in order to continue the request.
     * @throws LimitExceededException
     *         There are too many AWS Mobile Hub projects in the account or the account has exceeded the maximum number
     *         of resources in some AWS service. You should create another sub-account using AWS Organizations or remove
     *         some resources and retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MobileException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MobileClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateProjectResponse updateProject(UpdateProjectRequest updateProjectRequest) throws InternalFailureException,
            ServiceUnavailableException, UnauthorizedException, TooManyRequestsException, BadRequestException, NotFoundException,
            AccountActionRequiredException, LimitExceededException, AwsServiceException, SdkClientException, MobileException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");

            return clientHandler.execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                    .withOperationName("UpdateProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MobileServiceClientConfigurationBuilder serviceConfigBuilder = new MobileServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MobileException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountActionRequiredException")
                                .exceptionBuilderSupplier(AccountActionRequiredException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    @Override
    public final MobileServiceClientConfiguration serviceClientConfiguration() {
        return new MobileServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
