/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mobile;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mobile.internal.MobileServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mobile.model.AccountActionRequiredException;
import software.amazon.awssdk.services.mobile.model.BadRequestException;
import software.amazon.awssdk.services.mobile.model.CreateProjectRequest;
import software.amazon.awssdk.services.mobile.model.CreateProjectResponse;
import software.amazon.awssdk.services.mobile.model.DeleteProjectRequest;
import software.amazon.awssdk.services.mobile.model.DeleteProjectResponse;
import software.amazon.awssdk.services.mobile.model.DescribeBundleRequest;
import software.amazon.awssdk.services.mobile.model.DescribeBundleResponse;
import software.amazon.awssdk.services.mobile.model.DescribeProjectRequest;
import software.amazon.awssdk.services.mobile.model.DescribeProjectResponse;
import software.amazon.awssdk.services.mobile.model.ExportBundleRequest;
import software.amazon.awssdk.services.mobile.model.ExportBundleResponse;
import software.amazon.awssdk.services.mobile.model.ExportProjectRequest;
import software.amazon.awssdk.services.mobile.model.ExportProjectResponse;
import software.amazon.awssdk.services.mobile.model.InternalFailureException;
import software.amazon.awssdk.services.mobile.model.LimitExceededException;
import software.amazon.awssdk.services.mobile.model.ListBundlesRequest;
import software.amazon.awssdk.services.mobile.model.ListBundlesResponse;
import software.amazon.awssdk.services.mobile.model.ListProjectsRequest;
import software.amazon.awssdk.services.mobile.model.ListProjectsResponse;
import software.amazon.awssdk.services.mobile.model.MobileException;
import software.amazon.awssdk.services.mobile.model.NotFoundException;
import software.amazon.awssdk.services.mobile.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mobile.model.TooManyRequestsException;
import software.amazon.awssdk.services.mobile.model.UnauthorizedException;
import software.amazon.awssdk.services.mobile.model.UpdateProjectRequest;
import software.amazon.awssdk.services.mobile.model.UpdateProjectResponse;
import software.amazon.awssdk.services.mobile.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DescribeBundleRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ExportBundleRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ExportProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ListBundlesRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MobileAsyncClient}.
 *
 * @see MobileAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMobileAsyncClient implements MobileAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMobileAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMobileAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an AWS Mobile Hub project.
     * </p>
     *
     * @param createProjectRequest
     *        Request structure used to request a project be created.
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>LimitExceededException There are too many AWS Mobile Hub projects in the account or the account has
     *         exceeded the maximum number of resources in some AWS service. You should create another sub-account using
     *         AWS Organizations or remove some resources and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createProjectRequest));
            CompletableFuture<CreateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delets a project in AWS Mobile Hub.
     * </p>
     *
     * @param deleteProjectRequest
     *        Request structure used to request a project be deleted.
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProjectRequest));
            CompletableFuture<DeleteProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the bundle details for the requested bundle id.
     * </p>
     *
     * @param describeBundleRequest
     *        Request structure to request the details of a specific bundle.
     * @return A Java Future containing the result of the DescribeBundle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.DescribeBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DescribeBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBundleResponse> describeBundle(DescribeBundleRequest describeBundleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBundleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBundle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBundleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBundleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBundleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBundleRequest, DescribeBundleResponse>()
                            .withOperationName("DescribeBundle").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeBundleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeBundleRequest));
            CompletableFuture<DescribeBundleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a project in AWS Mobile Hub.
     * </p>
     *
     * @param describeProjectRequest
     *        Request structure used to request details about a project.
     * @return A Java Future containing the result of the DescribeProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DescribeProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProjectResponse> describeProject(DescribeProjectRequest describeProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                            .withOperationName("DescribeProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeProjectRequest));
            CompletableFuture<DescribeProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates customized software development kit (SDK) and or tool packages used to integrate mobile web or mobile
     * app clients with backend AWS resources.
     * </p>
     *
     * @param exportBundleRequest
     *        Request structure used to request generation of custom SDK and tool packages required to integrate mobile
     *        web or app clients with backed AWS resources.
     * @return A Java Future containing the result of the ExportBundle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ExportBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ExportBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExportBundleResponse> exportBundle(ExportBundleRequest exportBundleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportBundleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportBundle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportBundleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportBundleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportBundleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportBundleRequest, ExportBundleResponse>()
                            .withOperationName("ExportBundle").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportBundleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportBundleRequest));
            CompletableFuture<ExportBundleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports project configuration to a snapshot which can be downloaded and shared. Note that mobile app push
     * credentials are encrypted in exported projects, so they can only be shared successfully within the same AWS
     * account.
     * </p>
     *
     * @param exportProjectRequest
     *        Request structure used in requests to export project configuration details.
     * @return A Java Future containing the result of the ExportProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ExportProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ExportProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExportProjectResponse> exportProject(ExportProjectRequest exportProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportProjectRequest, ExportProjectResponse>()
                            .withOperationName("ExportProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportProjectRequest));
            CompletableFuture<ExportProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all available bundles.
     * </p>
     *
     * @param listBundlesRequest
     *        Request structure to request all available bundles.
     * @return A Java Future containing the result of the ListBundles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ListBundles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListBundles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBundlesResponse> listBundles(ListBundlesRequest listBundlesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBundlesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBundlesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBundles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBundlesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBundlesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBundlesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBundlesRequest, ListBundlesResponse>()
                            .withOperationName("ListBundles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListBundlesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBundlesRequest));
            CompletableFuture<ListBundlesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists projects in AWS Mobile Hub.
     * </p>
     *
     * @param listProjectsRequest
     *        Request structure used to request projects list in AWS Mobile Hub.
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProjectsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProjectsRequest));
            CompletableFuture<ListProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an existing project.
     * </p>
     *
     * @param updateProjectRequest
     *        Request structure used for requests to update project configuration.
     * @return A Java Future containing the result of the UpdateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>AccountActionRequiredException Account Action is required in order to continue the request.</li>
     *         <li>LimitExceededException There are too many AWS Mobile Hub projects in the account or the account has
     *         exceeded the maximum number of resources in some AWS service. You should create another sub-account using
     *         AWS Organizations or remove some resources and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectResponse> updateProject(UpdateProjectRequest updateProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                            .withOperationName("UpdateProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateProjectRequest));
            CompletableFuture<UpdateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MobileServiceClientConfiguration serviceClientConfiguration() {
        return new MobileServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MobileException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountActionRequiredException")
                                .exceptionBuilderSupplier(AccountActionRequiredException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MobileServiceClientConfigurationBuilder serviceConfigBuilder = new MobileServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
