/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the origin resource of an Amazon Lightsail content delivery network (CDN) distribution.
 * </p>
 * <p>
 * An origin can be a Lightsail instance, bucket, container service, or load balancer. A distribution pulls content from
 * an origin, caches it, and serves it to viewers via a worldwide network of edge servers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InputOrigin implements SdkPojo, Serializable, ToCopyableBuilder<InputOrigin.Builder, InputOrigin> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(InputOrigin::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> REGION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("regionName").getter(getter(InputOrigin::regionNameAsString)).setter(setter(Builder::regionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionName").build()).build();

    private static final SdkField<String> PROTOCOL_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("protocolPolicy").getter(getter(InputOrigin::protocolPolicyAsString))
            .setter(setter(Builder::protocolPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocolPolicy").build()).build();

    private static final SdkField<Integer> RESPONSE_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("responseTimeout").getter(getter(InputOrigin::responseTimeout)).setter(setter(Builder::responseTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseTimeout").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, REGION_NAME_FIELD,
            PROTOCOL_POLICY_FIELD, RESPONSE_TIMEOUT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String regionName;

    private final String protocolPolicy;

    private final Integer responseTimeout;

    private InputOrigin(BuilderImpl builder) {
        this.name = builder.name;
        this.regionName = builder.regionName;
        this.protocolPolicy = builder.protocolPolicy;
        this.responseTimeout = builder.responseTimeout;
    }

    /**
     * <p>
     * The name of the origin resource.
     * </p>
     * 
     * @return The name of the origin resource.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The AWS Region name of the origin resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #regionName} will
     * return {@link RegionName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionNameAsString}.
     * </p>
     * 
     * @return The AWS Region name of the origin resource.
     * @see RegionName
     */
    public final RegionName regionName() {
        return RegionName.fromValue(regionName);
    }

    /**
     * <p>
     * The AWS Region name of the origin resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #regionName} will
     * return {@link RegionName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionNameAsString}.
     * </p>
     * 
     * @return The AWS Region name of the origin resource.
     * @see RegionName
     */
    public final String regionNameAsString() {
        return regionName;
    }

    /**
     * <p>
     * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to pull
     * content.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocolPolicy}
     * will return {@link OriginProtocolPolicyEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #protocolPolicyAsString}.
     * </p>
     * 
     * @return The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin
     *         to pull content.
     * @see OriginProtocolPolicyEnum
     */
    public final OriginProtocolPolicyEnum protocolPolicy() {
        return OriginProtocolPolicyEnum.fromValue(protocolPolicy);
    }

    /**
     * <p>
     * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to pull
     * content.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocolPolicy}
     * will return {@link OriginProtocolPolicyEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #protocolPolicyAsString}.
     * </p>
     * 
     * @return The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin
     *         to pull content.
     * @see OriginProtocolPolicyEnum
     */
    public final String protocolPolicyAsString() {
        return protocolPolicy;
    }

    /**
     * <p>
     * The amount of time, in seconds, that the distribution waits for a response after forwarding a request to the
     * origin. The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you don't specify
     * otherwise) is 30 seconds.
     * </p>
     * 
     * @return The amount of time, in seconds, that the distribution waits for a response after forwarding a request to
     *         the origin. The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you don't
     *         specify otherwise) is 30 seconds.
     */
    public final Integer responseTimeout() {
        return responseTimeout;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(regionNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(protocolPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(responseTimeout());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InputOrigin)) {
            return false;
        }
        InputOrigin other = (InputOrigin) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(regionNameAsString(), other.regionNameAsString())
                && Objects.equals(protocolPolicyAsString(), other.protocolPolicyAsString())
                && Objects.equals(responseTimeout(), other.responseTimeout());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InputOrigin").add("Name", name()).add("RegionName", regionNameAsString())
                .add("ProtocolPolicy", protocolPolicyAsString()).add("ResponseTimeout", responseTimeout()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "regionName":
            return Optional.ofNullable(clazz.cast(regionNameAsString()));
        case "protocolPolicy":
            return Optional.ofNullable(clazz.cast(protocolPolicyAsString()));
        case "responseTimeout":
            return Optional.ofNullable(clazz.cast(responseTimeout()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InputOrigin, T> g) {
        return obj -> g.apply((InputOrigin) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InputOrigin> {
        /**
         * <p>
         * The name of the origin resource.
         * </p>
         * 
         * @param name
         *        The name of the origin resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The AWS Region name of the origin resource.
         * </p>
         * 
         * @param regionName
         *        The AWS Region name of the origin resource.
         * @see RegionName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionName
         */
        Builder regionName(String regionName);

        /**
         * <p>
         * The AWS Region name of the origin resource.
         * </p>
         * 
         * @param regionName
         *        The AWS Region name of the origin resource.
         * @see RegionName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionName
         */
        Builder regionName(RegionName regionName);

        /**
         * <p>
         * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to
         * pull content.
         * </p>
         * 
         * @param protocolPolicy
         *        The protocol that your Amazon Lightsail distribution uses when establishing a connection with your
         *        origin to pull content.
         * @see OriginProtocolPolicyEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginProtocolPolicyEnum
         */
        Builder protocolPolicy(String protocolPolicy);

        /**
         * <p>
         * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to
         * pull content.
         * </p>
         * 
         * @param protocolPolicy
         *        The protocol that your Amazon Lightsail distribution uses when establishing a connection with your
         *        origin to pull content.
         * @see OriginProtocolPolicyEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginProtocolPolicyEnum
         */
        Builder protocolPolicy(OriginProtocolPolicyEnum protocolPolicy);

        /**
         * <p>
         * The amount of time, in seconds, that the distribution waits for a response after forwarding a request to the
         * origin. The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you don't specify
         * otherwise) is 30 seconds.
         * </p>
         * 
         * @param responseTimeout
         *        The amount of time, in seconds, that the distribution waits for a response after forwarding a request
         *        to the origin. The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you
         *        don't specify otherwise) is 30 seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseTimeout(Integer responseTimeout);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String regionName;

        private String protocolPolicy;

        private Integer responseTimeout;

        private BuilderImpl() {
        }

        private BuilderImpl(InputOrigin model) {
            name(model.name);
            regionName(model.regionName);
            protocolPolicy(model.protocolPolicy);
            responseTimeout(model.responseTimeout);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getRegionName() {
            return regionName;
        }

        public final void setRegionName(String regionName) {
            this.regionName = regionName;
        }

        @Override
        public final Builder regionName(String regionName) {
            this.regionName = regionName;
            return this;
        }

        @Override
        public final Builder regionName(RegionName regionName) {
            this.regionName(regionName == null ? null : regionName.toString());
            return this;
        }

        public final String getProtocolPolicy() {
            return protocolPolicy;
        }

        public final void setProtocolPolicy(String protocolPolicy) {
            this.protocolPolicy = protocolPolicy;
        }

        @Override
        public final Builder protocolPolicy(String protocolPolicy) {
            this.protocolPolicy = protocolPolicy;
            return this;
        }

        @Override
        public final Builder protocolPolicy(OriginProtocolPolicyEnum protocolPolicy) {
            this.protocolPolicy(protocolPolicy == null ? null : protocolPolicy.toString());
            return this;
        }

        public final Integer getResponseTimeout() {
            return responseTimeout;
        }

        public final void setResponseTimeout(Integer responseTimeout) {
            this.responseTimeout = responseTimeout;
        }

        @Override
        public final Builder responseTimeout(Integer responseTimeout) {
            this.responseTimeout = responseTimeout;
            return this;
        }

        @Override
        public InputOrigin build() {
            return new InputOrigin(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
