/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.elasticbeanstalk.ElasticBeanstalkAsyncClient;
import software.amazon.awssdk.services.elasticbeanstalk.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.elasticbeanstalk.model.DescribeEnvironmentsRequest;
import software.amazon.awssdk.services.elasticbeanstalk.model.DescribeEnvironmentsResponse;
import software.amazon.awssdk.services.elasticbeanstalk.model.ElasticBeanstalkRequest;
import software.amazon.awssdk.services.elasticbeanstalk.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultElasticBeanstalkAsyncWaiter implements ElasticBeanstalkAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final ElasticBeanstalkAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeEnvironmentsResponse> environmentExistsWaiter;

    private final AsyncWaiter<DescribeEnvironmentsResponse> environmentUpdatedWaiter;

    private final AsyncWaiter<DescribeEnvironmentsResponse> environmentTerminatedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultElasticBeanstalkAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ElasticBeanstalkAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.environmentExistsWaiter = AsyncWaiter.builder(DescribeEnvironmentsResponse.class)
                .acceptors(environmentExistsWaiterAcceptors())
                .overrideConfiguration(environmentExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.environmentUpdatedWaiter = AsyncWaiter.builder(DescribeEnvironmentsResponse.class)
                .acceptors(environmentUpdatedWaiterAcceptors())
                .overrideConfiguration(environmentUpdatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.environmentTerminatedWaiter = AsyncWaiter.builder(DescribeEnvironmentsResponse.class)
                .acceptors(environmentTerminatedWaiterAcceptors())
                .overrideConfiguration(environmentTerminatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEnvironmentsResponse>> waitUntilEnvironmentExists(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        return environmentExistsWaiter.runAsync(() -> client
                .describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEnvironmentsResponse>> waitUntilEnvironmentExists(
            DescribeEnvironmentsRequest describeEnvironmentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentExistsWaiter.runAsync(
                () -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)),
                environmentExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEnvironmentsResponse>> waitUntilEnvironmentTerminated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        return environmentTerminatedWaiter.runAsync(() -> client
                .describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEnvironmentsResponse>> waitUntilEnvironmentTerminated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentTerminatedWaiter.runAsync(
                () -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)),
                environmentTerminatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEnvironmentsResponse>> waitUntilEnvironmentUpdated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        return environmentUpdatedWaiter.runAsync(() -> client
                .describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEnvironmentsResponse>> waitUntilEnvironmentUpdated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentUpdatedWaiter.runAsync(
                () -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)),
                environmentUpdatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> environmentExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Ready"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Launching"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> environmentUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Ready"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Updating"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> environmentTerminatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Terminated"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Terminating"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration environmentExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentTerminatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ElasticBeanstalkAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ElasticBeanstalkRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ElasticBeanstalkAsyncWaiter.Builder {
        private ElasticBeanstalkAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public ElasticBeanstalkAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public ElasticBeanstalkAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ElasticBeanstalkAsyncWaiter.Builder client(ElasticBeanstalkAsyncClient client) {
            this.client = client;
            return this;
        }

        public ElasticBeanstalkAsyncWaiter build() {
            return new DefaultElasticBeanstalkAsyncWaiter(this);
        }
    }
}
