/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * CreateLocationSmbRequest
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLocationSmbRequest extends DataSyncRequest implements
        ToCopyableBuilder<CreateLocationSmbRequest.Builder, CreateLocationSmbRequest> {
    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(CreateLocationSmbRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<String> SERVER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerHostname").getter(getter(CreateLocationSmbRequest::serverHostname))
            .setter(setter(Builder::serverHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerHostname").build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("User")
            .getter(getter(CreateLocationSmbRequest::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("User").build()).build();

    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Domain")
            .getter(getter(CreateLocationSmbRequest::domain)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Password").getter(getter(CreateLocationSmbRequest::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(CreateLocationSmbRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SmbMountOptions> MOUNT_OPTIONS_FIELD = SdkField
            .<SmbMountOptions> builder(MarshallingType.SDK_POJO).memberName("MountOptions")
            .getter(getter(CreateLocationSmbRequest::mountOptions)).setter(setter(Builder::mountOptions))
            .constructor(SmbMountOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MountOptions").build()).build();

    private static final SdkField<List<TagListEntry>> TAGS_FIELD = SdkField
            .<List<TagListEntry>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateLocationSmbRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TagListEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagListEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBDIRECTORY_FIELD,
            SERVER_HOSTNAME_FIELD, USER_FIELD, DOMAIN_FIELD, PASSWORD_FIELD, AGENT_ARNS_FIELD, MOUNT_OPTIONS_FIELD, TAGS_FIELD));

    private final String subdirectory;

    private final String serverHostname;

    private final String user;

    private final String domain;

    private final String password;

    private final List<String> agentArns;

    private final SmbMountOptions mountOptions;

    private final List<TagListEntry> tags;

    private CreateLocationSmbRequest(BuilderImpl builder) {
        super(builder);
        this.subdirectory = builder.subdirectory;
        this.serverHostname = builder.serverHostname;
        this.user = builder.user;
        this.domain = builder.domain;
        this.password = builder.password;
        this.agentArns = builder.agentArns;
        this.mountOptions = builder.mountOptions;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Specifies the name of the share exported by your SMB file server where DataSync will read or write data. You can
     * include a subdirectory in the share path (for example, <code>/path/to/subdirectory</code>). Make sure that other
     * SMB clients in your network can also mount this path.
     * </p>
     * <p>
     * To copy all data in the subdirectory, DataSync must be able to mount the SMB share and access all of its data.
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     * >required permissions</a> for SMB locations.
     * </p>
     * 
     * @return Specifies the name of the share exported by your SMB file server where DataSync will read or write data.
     *         You can include a subdirectory in the share path (for example, <code>/path/to/subdirectory</code>). Make
     *         sure that other SMB clients in your network can also mount this path.</p>
     *         <p>
     *         To copy all data in the subdirectory, DataSync must be able to mount the SMB share and access all of its
     *         data. For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     *         >required permissions</a> for SMB locations.
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * <p>
     * Specifies the Domain Name Service (DNS) name or IP address of the SMB file server that your DataSync agent will
     * mount.
     * </p>
     * <note>
     * <p>
     * You can't specify an IP version 6 (IPv6) address.
     * </p>
     * </note>
     * 
     * @return Specifies the Domain Name Service (DNS) name or IP address of the SMB file server that your DataSync
     *         agent will mount.</p> <note>
     *         <p>
     *         You can't specify an IP version 6 (IPv6) address.
     *         </p>
     */
    public final String serverHostname() {
        return serverHostname;
    }

    /**
     * <p>
     * Specifies the user that can mount and access the files, folders, and file metadata in your SMB file server.
     * </p>
     * <p>
     * For information about choosing a user with the right level of access for your transfer, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     * >required permissions</a> for SMB locations.
     * </p>
     * 
     * @return Specifies the user that can mount and access the files, folders, and file metadata in your SMB file
     *         server.</p>
     *         <p>
     *         For information about choosing a user with the right level of access for your transfer, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     *         >required permissions</a> for SMB locations.
     */
    public final String user() {
        return user;
    }

    /**
     * <p>
     * Specifies the name of the Active Directory domain that your SMB file server belongs to.
     * </p>
     * <p>
     * If you have multiple Active Directory domains in your environment, configuring this parameter makes sure that
     * DataSync connects to the right file server.
     * </p>
     * 
     * @return Specifies the name of the Active Directory domain that your SMB file server belongs to. </p>
     *         <p>
     *         If you have multiple Active Directory domains in your environment, configuring this parameter makes sure
     *         that DataSync connects to the right file server.
     */
    public final String domain() {
        return domain;
    }

    /**
     * <p>
     * Specifies the password of the user who can mount your SMB file server and has permission to access the files and
     * folders involved in your transfer.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     * >required permissions</a> for SMB locations.
     * </p>
     * 
     * @return Specifies the password of the user who can mount your SMB file server and has permission to access the
     *         files and folders involved in your transfer.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     *         >required permissions</a> for SMB locations.
     */
    public final String password() {
        return password;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the DataSync agent (or agents) which you want to connect to your SMB file server. You specify an agent
     * by using its Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return Specifies the DataSync agent (or agents) which you want to connect to your SMB file server. You specify
     *         an agent by using its Amazon Resource Name (ARN).
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * <p>
     * Specifies the version of the SMB protocol that DataSync uses to access your SMB file server.
     * </p>
     * 
     * @return Specifies the version of the SMB protocol that DataSync uses to access your SMB file server.
     */
    public final SmbMountOptions mountOptions() {
        return mountOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
     * recommend creating at least a name tag for your location.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
     *         recommend creating at least a name tag for your location.
     */
    public final List<TagListEntry> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(serverHostname());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(mountOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLocationSmbRequest)) {
            return false;
        }
        CreateLocationSmbRequest other = (CreateLocationSmbRequest) obj;
        return Objects.equals(subdirectory(), other.subdirectory()) && Objects.equals(serverHostname(), other.serverHostname())
                && Objects.equals(user(), other.user()) && Objects.equals(domain(), other.domain())
                && Objects.equals(password(), other.password()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns()) && Objects.equals(mountOptions(), other.mountOptions())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateLocationSmbRequest").add("Subdirectory", subdirectory())
                .add("ServerHostname", serverHostname()).add("User", user()).add("Domain", domain())
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("MountOptions", mountOptions())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "ServerHostname":
            return Optional.ofNullable(clazz.cast(serverHostname()));
        case "User":
            return Optional.ofNullable(clazz.cast(user()));
        case "Domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "MountOptions":
            return Optional.ofNullable(clazz.cast(mountOptions()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLocationSmbRequest, T> g) {
        return obj -> g.apply((CreateLocationSmbRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateLocationSmbRequest> {
        /**
         * <p>
         * Specifies the name of the share exported by your SMB file server where DataSync will read or write data. You
         * can include a subdirectory in the share path (for example, <code>/path/to/subdirectory</code>). Make sure
         * that other SMB clients in your network can also mount this path.
         * </p>
         * <p>
         * To copy all data in the subdirectory, DataSync must be able to mount the SMB share and access all of its
         * data. For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         * >required permissions</a> for SMB locations.
         * </p>
         * 
         * @param subdirectory
         *        Specifies the name of the share exported by your SMB file server where DataSync will read or write
         *        data. You can include a subdirectory in the share path (for example,
         *        <code>/path/to/subdirectory</code>). Make sure that other SMB clients in your network can also mount
         *        this path.</p>
         *        <p>
         *        To copy all data in the subdirectory, DataSync must be able to mount the SMB share and access all of
         *        its data. For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         *        >required permissions</a> for SMB locations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * Specifies the Domain Name Service (DNS) name or IP address of the SMB file server that your DataSync agent
         * will mount.
         * </p>
         * <note>
         * <p>
         * You can't specify an IP version 6 (IPv6) address.
         * </p>
         * </note>
         * 
         * @param serverHostname
         *        Specifies the Domain Name Service (DNS) name or IP address of the SMB file server that your DataSync
         *        agent will mount.</p> <note>
         *        <p>
         *        You can't specify an IP version 6 (IPv6) address.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverHostname(String serverHostname);

        /**
         * <p>
         * Specifies the user that can mount and access the files, folders, and file metadata in your SMB file server.
         * </p>
         * <p>
         * For information about choosing a user with the right level of access for your transfer, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         * >required permissions</a> for SMB locations.
         * </p>
         * 
         * @param user
         *        Specifies the user that can mount and access the files, folders, and file metadata in your SMB file
         *        server.</p>
         *        <p>
         *        For information about choosing a user with the right level of access for your transfer, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         *        >required permissions</a> for SMB locations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);

        /**
         * <p>
         * Specifies the name of the Active Directory domain that your SMB file server belongs to.
         * </p>
         * <p>
         * If you have multiple Active Directory domains in your environment, configuring this parameter makes sure that
         * DataSync connects to the right file server.
         * </p>
         * 
         * @param domain
         *        Specifies the name of the Active Directory domain that your SMB file server belongs to. </p>
         *        <p>
         *        If you have multiple Active Directory domains in your environment, configuring this parameter makes
         *        sure that DataSync connects to the right file server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * Specifies the password of the user who can mount your SMB file server and has permission to access the files
         * and folders involved in your transfer.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         * >required permissions</a> for SMB locations.
         * </p>
         * 
         * @param password
         *        Specifies the password of the user who can mount your SMB file server and has permission to access the
         *        files and folders involved in your transfer.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         *        >required permissions</a> for SMB locations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * Specifies the DataSync agent (or agents) which you want to connect to your SMB file server. You specify an
         * agent by using its Amazon Resource Name (ARN).
         * </p>
         * 
         * @param agentArns
         *        Specifies the DataSync agent (or agents) which you want to connect to your SMB file server. You
         *        specify an agent by using its Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * Specifies the DataSync agent (or agents) which you want to connect to your SMB file server. You specify an
         * agent by using its Amazon Resource Name (ARN).
         * </p>
         * 
         * @param agentArns
         *        Specifies the DataSync agent (or agents) which you want to connect to your SMB file server. You
         *        specify an agent by using its Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * <p>
         * Specifies the version of the SMB protocol that DataSync uses to access your SMB file server.
         * </p>
         * 
         * @param mountOptions
         *        Specifies the version of the SMB protocol that DataSync uses to access your SMB file server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountOptions(SmbMountOptions mountOptions);

        /**
         * <p>
         * Specifies the version of the SMB protocol that DataSync uses to access your SMB file server.
         * </p>
         * This is a convenience method that creates an instance of the {@link SmbMountOptions.Builder} avoiding the
         * need to create one manually via {@link SmbMountOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SmbMountOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #mountOptions(SmbMountOptions)}.
         * 
         * @param mountOptions
         *        a consumer that will call methods on {@link SmbMountOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountOptions(SmbMountOptions)
         */
        default Builder mountOptions(Consumer<SmbMountOptions.Builder> mountOptions) {
            return mountOptions(SmbMountOptions.builder().applyMutation(mountOptions).build());
        }

        /**
         * <p>
         * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
         * recommend creating at least a name tag for your location.
         * </p>
         * 
         * @param tags
         *        Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources.
         *        We recommend creating at least a name tag for your location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<TagListEntry> tags);

        /**
         * <p>
         * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
         * recommend creating at least a name tag for your location.
         * </p>
         * 
         * @param tags
         *        Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources.
         *        We recommend creating at least a name tag for your location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(TagListEntry... tags);

        /**
         * <p>
         * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
         * recommend creating at least a name tag for your location.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.datasync.model.TagListEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<TagListEntry>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<TagListEntry>)
         */
        Builder tags(Consumer<TagListEntry.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String subdirectory;

        private String serverHostname;

        private String user;

        private String domain;

        private String password;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private SmbMountOptions mountOptions;

        private List<TagListEntry> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLocationSmbRequest model) {
            super(model);
            subdirectory(model.subdirectory);
            serverHostname(model.serverHostname);
            user(model.user);
            domain(model.domain);
            password(model.password);
            agentArns(model.agentArns);
            mountOptions(model.mountOptions);
            tags(model.tags);
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final String getServerHostname() {
            return serverHostname;
        }

        public final void setServerHostname(String serverHostname) {
            this.serverHostname = serverHostname;
        }

        @Override
        public final Builder serverHostname(String serverHostname) {
            this.serverHostname = serverHostname;
            return this;
        }

        public final String getUser() {
            return user;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final String getDomain() {
            return domain;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final String getPassword() {
            return password;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final SmbMountOptions.Builder getMountOptions() {
            return mountOptions != null ? mountOptions.toBuilder() : null;
        }

        public final void setMountOptions(SmbMountOptions.BuilderImpl mountOptions) {
            this.mountOptions = mountOptions != null ? mountOptions.build() : null;
        }

        @Override
        public final Builder mountOptions(SmbMountOptions mountOptions) {
            this.mountOptions = mountOptions;
            return this;
        }

        public final List<TagListEntry.Builder> getTags() {
            List<TagListEntry.Builder> result = InputTagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<TagListEntry.BuilderImpl> tags) {
            this.tags = InputTagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<TagListEntry> tags) {
            this.tags = InputTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(TagListEntry... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<TagListEntry.Builder>... tags) {
            tags(Stream.of(tags).map(c -> TagListEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateLocationSmbRequest build() {
            return new CreateLocationSmbRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
