/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon RDS recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RDSDBRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<RDSDBRecommendation.Builder, RDSDBRecommendation> {
    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceArn").getter(getter(RDSDBRecommendation::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceArn").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(RDSDBRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("engine")
            .getter(getter(RDSDBRecommendation::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("engine").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("engineVersion").getter(getter(RDSDBRecommendation::engineVersion))
            .setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("engineVersion").build()).build();

    private static final SdkField<String> CURRENT_DB_INSTANCE_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentDBInstanceClass").getter(getter(RDSDBRecommendation::currentDBInstanceClass))
            .setter(setter(Builder::currentDBInstanceClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentDBInstanceClass").build())
            .build();

    private static final SdkField<DBStorageConfiguration> CURRENT_STORAGE_CONFIGURATION_FIELD = SdkField
            .<DBStorageConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("currentStorageConfiguration")
            .getter(getter(RDSDBRecommendation::currentStorageConfiguration))
            .setter(setter(Builder::currentStorageConfiguration))
            .constructor(DBStorageConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentStorageConfiguration")
                    .build()).build();

    private static final SdkField<String> IDLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("idle")
            .getter(getter(RDSDBRecommendation::idleAsString)).setter(setter(Builder::idle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("idle").build()).build();

    private static final SdkField<String> INSTANCE_FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceFinding").getter(getter(RDSDBRecommendation::instanceFindingAsString))
            .setter(setter(Builder::instanceFinding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceFinding").build()).build();

    private static final SdkField<String> STORAGE_FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("storageFinding").getter(getter(RDSDBRecommendation::storageFindingAsString))
            .setter(setter(Builder::storageFinding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("storageFinding").build()).build();

    private static final SdkField<List<String>> INSTANCE_FINDING_REASON_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("instanceFindingReasonCodes")
            .getter(getter(RDSDBRecommendation::instanceFindingReasonCodesAsStrings))
            .setter(setter(Builder::instanceFindingReasonCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceFindingReasonCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> STORAGE_FINDING_REASON_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("storageFindingReasonCodes")
            .getter(getter(RDSDBRecommendation::storageFindingReasonCodesAsStrings))
            .setter(setter(Builder::storageFindingReasonCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("storageFindingReasonCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RDSDBInstanceRecommendationOption>> INSTANCE_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<RDSDBInstanceRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("instanceRecommendationOptions")
            .getter(getter(RDSDBRecommendation::instanceRecommendationOptions))
            .setter(setter(Builder::instanceRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RDSDBInstanceRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(RDSDBInstanceRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RDSDBStorageRecommendationOption>> STORAGE_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<RDSDBStorageRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("storageRecommendationOptions")
            .getter(getter(RDSDBRecommendation::storageRecommendationOptions))
            .setter(setter(Builder::storageRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("storageRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RDSDBStorageRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(RDSDBStorageRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RDSDBUtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<RDSDBUtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(RDSDBRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RDSDBUtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(RDSDBUtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RDSEffectiveRecommendationPreferences> EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD = SdkField
            .<RDSEffectiveRecommendationPreferences> builder(MarshallingType.SDK_POJO)
            .memberName("effectiveRecommendationPreferences")
            .getter(getter(RDSDBRecommendation::effectiveRecommendationPreferences))
            .setter(setter(Builder::effectiveRecommendationPreferences))
            .constructor(RDSEffectiveRecommendationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effectiveRecommendationPreferences")
                    .build()).build();

    private static final SdkField<Double> LOOKBACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookbackPeriodInDays").getter(getter(RDSDBRecommendation::lookbackPeriodInDays))
            .setter(setter(Builder::lookbackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookbackPeriodInDays").build())
            .build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(RDSDBRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(RDSDBRecommendation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_ARN_FIELD,
            ACCOUNT_ID_FIELD, ENGINE_FIELD, ENGINE_VERSION_FIELD, CURRENT_DB_INSTANCE_CLASS_FIELD,
            CURRENT_STORAGE_CONFIGURATION_FIELD, IDLE_FIELD, INSTANCE_FINDING_FIELD, STORAGE_FINDING_FIELD,
            INSTANCE_FINDING_REASON_CODES_FIELD, STORAGE_FINDING_REASON_CODES_FIELD, INSTANCE_RECOMMENDATION_OPTIONS_FIELD,
            STORAGE_RECOMMENDATION_OPTIONS_FIELD, UTILIZATION_METRICS_FIELD, EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD,
            LOOKBACK_PERIOD_IN_DAYS_FIELD, LAST_REFRESH_TIMESTAMP_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceArn;

    private final String accountId;

    private final String engine;

    private final String engineVersion;

    private final String currentDBInstanceClass;

    private final DBStorageConfiguration currentStorageConfiguration;

    private final String idle;

    private final String instanceFinding;

    private final String storageFinding;

    private final List<String> instanceFindingReasonCodes;

    private final List<String> storageFindingReasonCodes;

    private final List<RDSDBInstanceRecommendationOption> instanceRecommendationOptions;

    private final List<RDSDBStorageRecommendationOption> storageRecommendationOptions;

    private final List<RDSDBUtilizationMetric> utilizationMetrics;

    private final RDSEffectiveRecommendationPreferences effectiveRecommendationPreferences;

    private final Double lookbackPeriodInDays;

    private final Instant lastRefreshTimestamp;

    private final List<Tag> tags;

    private RDSDBRecommendation(BuilderImpl builder) {
        this.resourceArn = builder.resourceArn;
        this.accountId = builder.accountId;
        this.engine = builder.engine;
        this.engineVersion = builder.engineVersion;
        this.currentDBInstanceClass = builder.currentDBInstanceClass;
        this.currentStorageConfiguration = builder.currentStorageConfiguration;
        this.idle = builder.idle;
        this.instanceFinding = builder.instanceFinding;
        this.storageFinding = builder.storageFinding;
        this.instanceFindingReasonCodes = builder.instanceFindingReasonCodes;
        this.storageFindingReasonCodes = builder.storageFindingReasonCodes;
        this.instanceRecommendationOptions = builder.instanceRecommendationOptions;
        this.storageRecommendationOptions = builder.storageRecommendationOptions;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.effectiveRecommendationPreferences = builder.effectiveRecommendationPreferences;
        this.lookbackPeriodInDays = builder.lookbackPeriodInDays;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ARN of the current Amazon RDS.
     * </p>
     * <p>
     * The following is the format of the ARN:
     * </p>
     * <p>
     * <code>arn:aws:rds:{region}:{accountId}:db:{resourceName}</code>
     * </p>
     * 
     * @return The ARN of the current Amazon RDS. </p>
     *         <p>
     *         The following is the format of the ARN:
     *         </p>
     *         <p>
     *         <code>arn:aws:rds:{region}:{accountId}:db:{resourceName}</code>
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the Amazon RDS.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the Amazon RDS.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The engine of the RDS instance.
     * </p>
     * 
     * @return The engine of the RDS instance.
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * The database engine version.
     * </p>
     * 
     * @return The database engine version.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The DB instance class of the current RDS instance.
     * </p>
     * 
     * @return The DB instance class of the current RDS instance.
     */
    public final String currentDBInstanceClass() {
        return currentDBInstanceClass;
    }

    /**
     * <p>
     * The configuration of the current RDS storage.
     * </p>
     * 
     * @return The configuration of the current RDS storage.
     */
    public final DBStorageConfiguration currentStorageConfiguration() {
        return currentStorageConfiguration;
    }

    /**
     * <p>
     * This indicates if the RDS instance is idle or not.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #idle} will return
     * {@link Idle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #idleAsString}.
     * </p>
     * 
     * @return This indicates if the RDS instance is idle or not.
     * @see Idle
     */
    public final Idle idle() {
        return Idle.fromValue(idle);
    }

    /**
     * <p>
     * This indicates if the RDS instance is idle or not.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #idle} will return
     * {@link Idle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #idleAsString}.
     * </p>
     * 
     * @return This indicates if the RDS instance is idle or not.
     * @see Idle
     */
    public final String idleAsString() {
        return idle;
    }

    /**
     * <p>
     * The finding classification of an Amazon RDS instance.
     * </p>
     * <p>
     * Findings for Amazon RDS instance include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough resource
     * specifications, an Amazon RDS is considered under-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
     * specifications, an Amazon RDS is considered over-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the performance
     * requirements of your workload, the service is considered optimized.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceFinding}
     * will return {@link RDSInstanceFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceFindingAsString}.
     * </p>
     * 
     * @return The finding classification of an Amazon RDS instance. </p>
     *         <p>
     *         Findings for Amazon RDS instance include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough resource
     *         specifications, an Amazon RDS is considered under-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
     *         specifications, an Amazon RDS is considered over-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the
     *         performance requirements of your workload, the service is considered optimized.
     *         </p>
     *         </li>
     * @see RDSInstanceFinding
     */
    public final RDSInstanceFinding instanceFinding() {
        return RDSInstanceFinding.fromValue(instanceFinding);
    }

    /**
     * <p>
     * The finding classification of an Amazon RDS instance.
     * </p>
     * <p>
     * Findings for Amazon RDS instance include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough resource
     * specifications, an Amazon RDS is considered under-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
     * specifications, an Amazon RDS is considered over-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the performance
     * requirements of your workload, the service is considered optimized.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceFinding}
     * will return {@link RDSInstanceFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceFindingAsString}.
     * </p>
     * 
     * @return The finding classification of an Amazon RDS instance. </p>
     *         <p>
     *         Findings for Amazon RDS instance include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough resource
     *         specifications, an Amazon RDS is considered under-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
     *         specifications, an Amazon RDS is considered over-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the
     *         performance requirements of your workload, the service is considered optimized.
     *         </p>
     *         </li>
     * @see RDSInstanceFinding
     */
    public final String instanceFindingAsString() {
        return instanceFinding;
    }

    /**
     * <p>
     * The finding classification of Amazon RDS storage.
     * </p>
     * <p>
     * Findings for Amazon RDS instance include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough storage, an
     * Amazon RDS is considered under-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage, an Amazon
     * RDS is considered over-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance requirements of your
     * workload, the service is considered optimized.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageFinding}
     * will return {@link RDSStorageFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #storageFindingAsString}.
     * </p>
     * 
     * @return The finding classification of Amazon RDS storage. </p>
     *         <p>
     *         Findings for Amazon RDS instance include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough storage,
     *         an Amazon RDS is considered under-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage, an
     *         Amazon RDS is considered over-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance requirements
     *         of your workload, the service is considered optimized.
     *         </p>
     *         </li>
     * @see RDSStorageFinding
     */
    public final RDSStorageFinding storageFinding() {
        return RDSStorageFinding.fromValue(storageFinding);
    }

    /**
     * <p>
     * The finding classification of Amazon RDS storage.
     * </p>
     * <p>
     * Findings for Amazon RDS instance include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough storage, an
     * Amazon RDS is considered under-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage, an Amazon
     * RDS is considered over-provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance requirements of your
     * workload, the service is considered optimized.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageFinding}
     * will return {@link RDSStorageFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #storageFindingAsString}.
     * </p>
     * 
     * @return The finding classification of Amazon RDS storage. </p>
     *         <p>
     *         Findings for Amazon RDS instance include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough storage,
     *         an Amazon RDS is considered under-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage, an
     *         Amazon RDS is considered over-provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance requirements
     *         of your workload, the service is considered optimized.
     *         </p>
     *         </li>
     * @see RDSStorageFinding
     */
    public final String storageFindingAsString() {
        return storageFinding;
    }

    /**
     * <p>
     * The reason for the finding classification of an Amazon RDS instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of an Amazon RDS instance.
     */
    public final List<RDSInstanceFindingReasonCode> instanceFindingReasonCodes() {
        return RDSInstanceFindingReasonCodesCopier.copyStringToEnum(instanceFindingReasonCodes);
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceFindingReasonCodes property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInstanceFindingReasonCodes() {
        return instanceFindingReasonCodes != null && !(instanceFindingReasonCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason for the finding classification of an Amazon RDS instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of an Amazon RDS instance.
     */
    public final List<String> instanceFindingReasonCodesAsStrings() {
        return instanceFindingReasonCodes;
    }

    /**
     * <p>
     * The reason for the finding classification of Amazon RDS storage.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStorageFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of Amazon RDS storage.
     */
    public final List<RDSStorageFindingReasonCode> storageFindingReasonCodes() {
        return RDSStorageFindingReasonCodesCopier.copyStringToEnum(storageFindingReasonCodes);
    }

    /**
     * For responses, this returns true if the service returned a value for the StorageFindingReasonCodes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasStorageFindingReasonCodes() {
        return storageFindingReasonCodes != null && !(storageFindingReasonCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason for the finding classification of Amazon RDS storage.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStorageFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of Amazon RDS storage.
     */
    public final List<String> storageFindingReasonCodesAsStrings() {
        return storageFindingReasonCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceRecommendationOptions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInstanceRecommendationOptions() {
        return instanceRecommendationOptions != null && !(instanceRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the Amazon RDS instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceRecommendationOptions} method.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the Amazon RDS instance.
     */
    public final List<RDSDBInstanceRecommendationOption> instanceRecommendationOptions() {
        return instanceRecommendationOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the StorageRecommendationOptions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasStorageRecommendationOptions() {
        return storageRecommendationOptions != null && !(storageRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for Amazon RDS storage.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStorageRecommendationOptions} method.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for Amazon RDS storage.
     */
    public final List<RDSDBStorageRecommendationOption> storageRecommendationOptions() {
        return storageRecommendationOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationMetrics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the Amazon RDS.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the Amazon RDS.
     */
    public final List<RDSDBUtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * Describes the effective recommendation preferences for Amazon RDS.
     * </p>
     * 
     * @return Describes the effective recommendation preferences for Amazon RDS.
     */
    public final RDSEffectiveRecommendationPreferences effectiveRecommendationPreferences() {
        return effectiveRecommendationPreferences;
    }

    /**
     * <p>
     * The number of days the Amazon RDS utilization metrics were analyzed.
     * </p>
     * 
     * @return The number of days the Amazon RDS utilization metrics were analyzed.
     */
    public final Double lookbackPeriodInDays() {
        return lookbackPeriodInDays;
    }

    /**
     * <p>
     * The timestamp of when the Amazon RDS recommendation was last generated.
     * </p>
     * 
     * @return The timestamp of when the Amazon RDS recommendation was last generated.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags assigned to your Amazon RDS recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags assigned to your Amazon RDS recommendations.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(currentDBInstanceClass());
        hashCode = 31 * hashCode + Objects.hashCode(currentStorageConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(idleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceFindingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(storageFindingAsString());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasInstanceFindingReasonCodes() ? instanceFindingReasonCodesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStorageFindingReasonCodes() ? storageFindingReasonCodesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceRecommendationOptions() ? instanceRecommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStorageRecommendationOptions() ? storageRecommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(effectiveRecommendationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(lookbackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RDSDBRecommendation)) {
            return false;
        }
        RDSDBRecommendation other = (RDSDBRecommendation) obj;
        return Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(engine(), other.engine()) && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(currentDBInstanceClass(), other.currentDBInstanceClass())
                && Objects.equals(currentStorageConfiguration(), other.currentStorageConfiguration())
                && Objects.equals(idleAsString(), other.idleAsString())
                && Objects.equals(instanceFindingAsString(), other.instanceFindingAsString())
                && Objects.equals(storageFindingAsString(), other.storageFindingAsString())
                && hasInstanceFindingReasonCodes() == other.hasInstanceFindingReasonCodes()
                && Objects.equals(instanceFindingReasonCodesAsStrings(), other.instanceFindingReasonCodesAsStrings())
                && hasStorageFindingReasonCodes() == other.hasStorageFindingReasonCodes()
                && Objects.equals(storageFindingReasonCodesAsStrings(), other.storageFindingReasonCodesAsStrings())
                && hasInstanceRecommendationOptions() == other.hasInstanceRecommendationOptions()
                && Objects.equals(instanceRecommendationOptions(), other.instanceRecommendationOptions())
                && hasStorageRecommendationOptions() == other.hasStorageRecommendationOptions()
                && Objects.equals(storageRecommendationOptions(), other.storageRecommendationOptions())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(effectiveRecommendationPreferences(), other.effectiveRecommendationPreferences())
                && Objects.equals(lookbackPeriodInDays(), other.lookbackPeriodInDays())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("RDSDBRecommendation")
                .add("ResourceArn", resourceArn())
                .add("AccountId", accountId())
                .add("Engine", engine())
                .add("EngineVersion", engineVersion())
                .add("CurrentDBInstanceClass", currentDBInstanceClass())
                .add("CurrentStorageConfiguration", currentStorageConfiguration())
                .add("Idle", idleAsString())
                .add("InstanceFinding", instanceFindingAsString())
                .add("StorageFinding", storageFindingAsString())
                .add("InstanceFindingReasonCodes", hasInstanceFindingReasonCodes() ? instanceFindingReasonCodesAsStrings() : null)
                .add("StorageFindingReasonCodes", hasStorageFindingReasonCodes() ? storageFindingReasonCodesAsStrings() : null)
                .add("InstanceRecommendationOptions", hasInstanceRecommendationOptions() ? instanceRecommendationOptions() : null)
                .add("StorageRecommendationOptions", hasStorageRecommendationOptions() ? storageRecommendationOptions() : null)
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("EffectiveRecommendationPreferences", effectiveRecommendationPreferences())
                .add("LookbackPeriodInDays", lookbackPeriodInDays()).add("LastRefreshTimestamp", lastRefreshTimestamp())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "engineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "currentDBInstanceClass":
            return Optional.ofNullable(clazz.cast(currentDBInstanceClass()));
        case "currentStorageConfiguration":
            return Optional.ofNullable(clazz.cast(currentStorageConfiguration()));
        case "idle":
            return Optional.ofNullable(clazz.cast(idleAsString()));
        case "instanceFinding":
            return Optional.ofNullable(clazz.cast(instanceFindingAsString()));
        case "storageFinding":
            return Optional.ofNullable(clazz.cast(storageFindingAsString()));
        case "instanceFindingReasonCodes":
            return Optional.ofNullable(clazz.cast(instanceFindingReasonCodesAsStrings()));
        case "storageFindingReasonCodes":
            return Optional.ofNullable(clazz.cast(storageFindingReasonCodesAsStrings()));
        case "instanceRecommendationOptions":
            return Optional.ofNullable(clazz.cast(instanceRecommendationOptions()));
        case "storageRecommendationOptions":
            return Optional.ofNullable(clazz.cast(storageRecommendationOptions()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "effectiveRecommendationPreferences":
            return Optional.ofNullable(clazz.cast(effectiveRecommendationPreferences()));
        case "lookbackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookbackPeriodInDays()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RDSDBRecommendation, T> g) {
        return obj -> g.apply((RDSDBRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RDSDBRecommendation> {
        /**
         * <p>
         * The ARN of the current Amazon RDS.
         * </p>
         * <p>
         * The following is the format of the ARN:
         * </p>
         * <p>
         * <code>arn:aws:rds:{region}:{accountId}:db:{resourceName}</code>
         * </p>
         * 
         * @param resourceArn
         *        The ARN of the current Amazon RDS. </p>
         *        <p>
         *        The following is the format of the ARN:
         *        </p>
         *        <p>
         *        <code>arn:aws:rds:{region}:{accountId}:db:{resourceName}</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The Amazon Web Services account ID of the Amazon RDS.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the Amazon RDS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The engine of the RDS instance.
         * </p>
         * 
         * @param engine
         *        The engine of the RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The database engine version.
         * </p>
         * 
         * @param engineVersion
         *        The database engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The DB instance class of the current RDS instance.
         * </p>
         * 
         * @param currentDBInstanceClass
         *        The DB instance class of the current RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentDBInstanceClass(String currentDBInstanceClass);

        /**
         * <p>
         * The configuration of the current RDS storage.
         * </p>
         * 
         * @param currentStorageConfiguration
         *        The configuration of the current RDS storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentStorageConfiguration(DBStorageConfiguration currentStorageConfiguration);

        /**
         * <p>
         * The configuration of the current RDS storage.
         * </p>
         * This is a convenience method that creates an instance of the {@link DBStorageConfiguration.Builder} avoiding
         * the need to create one manually via {@link DBStorageConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DBStorageConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #currentStorageConfiguration(DBStorageConfiguration)}.
         * 
         * @param currentStorageConfiguration
         *        a consumer that will call methods on {@link DBStorageConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentStorageConfiguration(DBStorageConfiguration)
         */
        default Builder currentStorageConfiguration(Consumer<DBStorageConfiguration.Builder> currentStorageConfiguration) {
            return currentStorageConfiguration(DBStorageConfiguration.builder().applyMutation(currentStorageConfiguration)
                    .build());
        }

        /**
         * <p>
         * This indicates if the RDS instance is idle or not.
         * </p>
         * 
         * @param idle
         *        This indicates if the RDS instance is idle or not.
         * @see Idle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Idle
         */
        Builder idle(String idle);

        /**
         * <p>
         * This indicates if the RDS instance is idle or not.
         * </p>
         * 
         * @param idle
         *        This indicates if the RDS instance is idle or not.
         * @see Idle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Idle
         */
        Builder idle(Idle idle);

        /**
         * <p>
         * The finding classification of an Amazon RDS instance.
         * </p>
         * <p>
         * Findings for Amazon RDS instance include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough resource
         * specifications, an Amazon RDS is considered under-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
         * specifications, an Amazon RDS is considered over-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the performance
         * requirements of your workload, the service is considered optimized.
         * </p>
         * </li>
         * </ul>
         * 
         * @param instanceFinding
         *        The finding classification of an Amazon RDS instance. </p>
         *        <p>
         *        Findings for Amazon RDS instance include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough
         *        resource specifications, an Amazon RDS is considered under-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
         *        specifications, an Amazon RDS is considered over-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the
         *        performance requirements of your workload, the service is considered optimized.
         *        </p>
         *        </li>
         * @see RDSInstanceFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RDSInstanceFinding
         */
        Builder instanceFinding(String instanceFinding);

        /**
         * <p>
         * The finding classification of an Amazon RDS instance.
         * </p>
         * <p>
         * Findings for Amazon RDS instance include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough resource
         * specifications, an Amazon RDS is considered under-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
         * specifications, an Amazon RDS is considered over-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the performance
         * requirements of your workload, the service is considered optimized.
         * </p>
         * </li>
         * </ul>
         * 
         * @param instanceFinding
         *        The finding classification of an Amazon RDS instance. </p>
         *        <p>
         *        Findings for Amazon RDS instance include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough
         *        resource specifications, an Amazon RDS is considered under-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive resource
         *        specifications, an Amazon RDS is considered over-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — When the specifications of your Amazon RDS instance meet the
         *        performance requirements of your workload, the service is considered optimized.
         *        </p>
         *        </li>
         * @see RDSInstanceFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RDSInstanceFinding
         */
        Builder instanceFinding(RDSInstanceFinding instanceFinding);

        /**
         * <p>
         * The finding classification of Amazon RDS storage.
         * </p>
         * <p>
         * Findings for Amazon RDS instance include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough storage, an
         * Amazon RDS is considered under-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage, an
         * Amazon RDS is considered over-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance requirements of
         * your workload, the service is considered optimized.
         * </p>
         * </li>
         * </ul>
         * 
         * @param storageFinding
         *        The finding classification of Amazon RDS storage. </p>
         *        <p>
         *        Findings for Amazon RDS instance include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough
         *        storage, an Amazon RDS is considered under-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage,
         *        an Amazon RDS is considered over-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance
         *        requirements of your workload, the service is considered optimized.
         *        </p>
         *        </li>
         * @see RDSStorageFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RDSStorageFinding
         */
        Builder storageFinding(String storageFinding);

        /**
         * <p>
         * The finding classification of Amazon RDS storage.
         * </p>
         * <p>
         * Findings for Amazon RDS instance include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough storage, an
         * Amazon RDS is considered under-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage, an
         * Amazon RDS is considered over-provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance requirements of
         * your workload, the service is considered optimized.
         * </p>
         * </li>
         * </ul>
         * 
         * @param storageFinding
         *        The finding classification of Amazon RDS storage. </p>
         *        <p>
         *        Findings for Amazon RDS instance include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough
         *        storage, an Amazon RDS is considered under-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive storage,
         *        an Amazon RDS is considered over-provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — When the storage of your Amazon RDS meet the performance
         *        requirements of your workload, the service is considered optimized.
         *        </p>
         *        </li>
         * @see RDSStorageFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RDSStorageFinding
         */
        Builder storageFinding(RDSStorageFinding storageFinding);

        /**
         * <p>
         * The reason for the finding classification of an Amazon RDS instance.
         * </p>
         * 
         * @param instanceFindingReasonCodes
         *        The reason for the finding classification of an Amazon RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFindingReasonCodesWithStrings(Collection<String> instanceFindingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of an Amazon RDS instance.
         * </p>
         * 
         * @param instanceFindingReasonCodes
         *        The reason for the finding classification of an Amazon RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFindingReasonCodesWithStrings(String... instanceFindingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of an Amazon RDS instance.
         * </p>
         * 
         * @param instanceFindingReasonCodes
         *        The reason for the finding classification of an Amazon RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFindingReasonCodes(Collection<RDSInstanceFindingReasonCode> instanceFindingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of an Amazon RDS instance.
         * </p>
         * 
         * @param instanceFindingReasonCodes
         *        The reason for the finding classification of an Amazon RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFindingReasonCodes(RDSInstanceFindingReasonCode... instanceFindingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of Amazon RDS storage.
         * </p>
         * 
         * @param storageFindingReasonCodes
         *        The reason for the finding classification of Amazon RDS storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageFindingReasonCodesWithStrings(Collection<String> storageFindingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of Amazon RDS storage.
         * </p>
         * 
         * @param storageFindingReasonCodes
         *        The reason for the finding classification of Amazon RDS storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageFindingReasonCodesWithStrings(String... storageFindingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of Amazon RDS storage.
         * </p>
         * 
         * @param storageFindingReasonCodes
         *        The reason for the finding classification of Amazon RDS storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageFindingReasonCodes(Collection<RDSStorageFindingReasonCode> storageFindingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of Amazon RDS storage.
         * </p>
         * 
         * @param storageFindingReasonCodes
         *        The reason for the finding classification of Amazon RDS storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageFindingReasonCodes(RDSStorageFindingReasonCode... storageFindingReasonCodes);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Amazon RDS instance.
         * </p>
         * 
         * @param instanceRecommendationOptions
         *        An array of objects that describe the recommendation options for the Amazon RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRecommendationOptions(Collection<RDSDBInstanceRecommendationOption> instanceRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Amazon RDS instance.
         * </p>
         * 
         * @param instanceRecommendationOptions
         *        An array of objects that describe the recommendation options for the Amazon RDS instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRecommendationOptions(RDSDBInstanceRecommendationOption... instanceRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Amazon RDS instance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBInstanceRecommendationOption.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBInstanceRecommendationOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBInstanceRecommendationOption.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #instanceRecommendationOptions(List<RDSDBInstanceRecommendationOption>)}.
         * 
         * @param instanceRecommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBInstanceRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceRecommendationOptions(java.util.Collection<RDSDBInstanceRecommendationOption>)
         */
        Builder instanceRecommendationOptions(
                Consumer<RDSDBInstanceRecommendationOption.Builder>... instanceRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for Amazon RDS storage.
         * </p>
         * 
         * @param storageRecommendationOptions
         *        An array of objects that describe the recommendation options for Amazon RDS storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageRecommendationOptions(Collection<RDSDBStorageRecommendationOption> storageRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for Amazon RDS storage.
         * </p>
         * 
         * @param storageRecommendationOptions
         *        An array of objects that describe the recommendation options for Amazon RDS storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageRecommendationOptions(RDSDBStorageRecommendationOption... storageRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for Amazon RDS storage.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBStorageRecommendationOption.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBStorageRecommendationOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBStorageRecommendationOption.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #storageRecommendationOptions(List<RDSDBStorageRecommendationOption>)}.
         * 
         * @param storageRecommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBStorageRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storageRecommendationOptions(java.util.Collection<RDSDBStorageRecommendationOption>)
         */
        Builder storageRecommendationOptions(Consumer<RDSDBStorageRecommendationOption.Builder>... storageRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Amazon RDS.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Amazon RDS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<RDSDBUtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Amazon RDS.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Amazon RDS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(RDSDBUtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Amazon RDS.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBUtilizationMetric.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBUtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBUtilizationMetric.Builder#build()} is
         * called immediately and its result is passed to {@link #utilizationMetrics(List<RDSDBUtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.RDSDBUtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(java.util.Collection<RDSDBUtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<RDSDBUtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * Describes the effective recommendation preferences for Amazon RDS.
         * </p>
         * 
         * @param effectiveRecommendationPreferences
         *        Describes the effective recommendation preferences for Amazon RDS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder effectiveRecommendationPreferences(RDSEffectiveRecommendationPreferences effectiveRecommendationPreferences);

        /**
         * <p>
         * Describes the effective recommendation preferences for Amazon RDS.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link RDSEffectiveRecommendationPreferences.Builder} avoiding the need to create one manually via
         * {@link RDSEffectiveRecommendationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RDSEffectiveRecommendationPreferences.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #effectiveRecommendationPreferences(RDSEffectiveRecommendationPreferences)}.
         * 
         * @param effectiveRecommendationPreferences
         *        a consumer that will call methods on {@link RDSEffectiveRecommendationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #effectiveRecommendationPreferences(RDSEffectiveRecommendationPreferences)
         */
        default Builder effectiveRecommendationPreferences(
                Consumer<RDSEffectiveRecommendationPreferences.Builder> effectiveRecommendationPreferences) {
            return effectiveRecommendationPreferences(RDSEffectiveRecommendationPreferences.builder()
                    .applyMutation(effectiveRecommendationPreferences).build());
        }

        /**
         * <p>
         * The number of days the Amazon RDS utilization metrics were analyzed.
         * </p>
         * 
         * @param lookbackPeriodInDays
         *        The number of days the Amazon RDS utilization metrics were analyzed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookbackPeriodInDays(Double lookbackPeriodInDays);

        /**
         * <p>
         * The timestamp of when the Amazon RDS recommendation was last generated.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The timestamp of when the Amazon RDS recommendation was last generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * A list of tags assigned to your Amazon RDS recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon RDS recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon RDS recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon RDS recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon RDS recommendations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.computeoptimizer.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String resourceArn;

        private String accountId;

        private String engine;

        private String engineVersion;

        private String currentDBInstanceClass;

        private DBStorageConfiguration currentStorageConfiguration;

        private String idle;

        private String instanceFinding;

        private String storageFinding;

        private List<String> instanceFindingReasonCodes = DefaultSdkAutoConstructList.getInstance();

        private List<String> storageFindingReasonCodes = DefaultSdkAutoConstructList.getInstance();

        private List<RDSDBInstanceRecommendationOption> instanceRecommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private List<RDSDBStorageRecommendationOption> storageRecommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private List<RDSDBUtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private RDSEffectiveRecommendationPreferences effectiveRecommendationPreferences;

        private Double lookbackPeriodInDays;

        private Instant lastRefreshTimestamp;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RDSDBRecommendation model) {
            resourceArn(model.resourceArn);
            accountId(model.accountId);
            engine(model.engine);
            engineVersion(model.engineVersion);
            currentDBInstanceClass(model.currentDBInstanceClass);
            currentStorageConfiguration(model.currentStorageConfiguration);
            idle(model.idle);
            instanceFinding(model.instanceFinding);
            storageFinding(model.storageFinding);
            instanceFindingReasonCodesWithStrings(model.instanceFindingReasonCodes);
            storageFindingReasonCodesWithStrings(model.storageFindingReasonCodes);
            instanceRecommendationOptions(model.instanceRecommendationOptions);
            storageRecommendationOptions(model.storageRecommendationOptions);
            utilizationMetrics(model.utilizationMetrics);
            effectiveRecommendationPreferences(model.effectiveRecommendationPreferences);
            lookbackPeriodInDays(model.lookbackPeriodInDays);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            tags(model.tags);
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final String getCurrentDBInstanceClass() {
            return currentDBInstanceClass;
        }

        public final void setCurrentDBInstanceClass(String currentDBInstanceClass) {
            this.currentDBInstanceClass = currentDBInstanceClass;
        }

        @Override
        public final Builder currentDBInstanceClass(String currentDBInstanceClass) {
            this.currentDBInstanceClass = currentDBInstanceClass;
            return this;
        }

        public final DBStorageConfiguration.Builder getCurrentStorageConfiguration() {
            return currentStorageConfiguration != null ? currentStorageConfiguration.toBuilder() : null;
        }

        public final void setCurrentStorageConfiguration(DBStorageConfiguration.BuilderImpl currentStorageConfiguration) {
            this.currentStorageConfiguration = currentStorageConfiguration != null ? currentStorageConfiguration.build() : null;
        }

        @Override
        public final Builder currentStorageConfiguration(DBStorageConfiguration currentStorageConfiguration) {
            this.currentStorageConfiguration = currentStorageConfiguration;
            return this;
        }

        public final String getIdle() {
            return idle;
        }

        public final void setIdle(String idle) {
            this.idle = idle;
        }

        @Override
        public final Builder idle(String idle) {
            this.idle = idle;
            return this;
        }

        @Override
        public final Builder idle(Idle idle) {
            this.idle(idle == null ? null : idle.toString());
            return this;
        }

        public final String getInstanceFinding() {
            return instanceFinding;
        }

        public final void setInstanceFinding(String instanceFinding) {
            this.instanceFinding = instanceFinding;
        }

        @Override
        public final Builder instanceFinding(String instanceFinding) {
            this.instanceFinding = instanceFinding;
            return this;
        }

        @Override
        public final Builder instanceFinding(RDSInstanceFinding instanceFinding) {
            this.instanceFinding(instanceFinding == null ? null : instanceFinding.toString());
            return this;
        }

        public final String getStorageFinding() {
            return storageFinding;
        }

        public final void setStorageFinding(String storageFinding) {
            this.storageFinding = storageFinding;
        }

        @Override
        public final Builder storageFinding(String storageFinding) {
            this.storageFinding = storageFinding;
            return this;
        }

        @Override
        public final Builder storageFinding(RDSStorageFinding storageFinding) {
            this.storageFinding(storageFinding == null ? null : storageFinding.toString());
            return this;
        }

        public final Collection<String> getInstanceFindingReasonCodes() {
            if (instanceFindingReasonCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceFindingReasonCodes;
        }

        public final void setInstanceFindingReasonCodes(Collection<String> instanceFindingReasonCodes) {
            this.instanceFindingReasonCodes = RDSInstanceFindingReasonCodesCopier.copy(instanceFindingReasonCodes);
        }

        @Override
        public final Builder instanceFindingReasonCodesWithStrings(Collection<String> instanceFindingReasonCodes) {
            this.instanceFindingReasonCodes = RDSInstanceFindingReasonCodesCopier.copy(instanceFindingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceFindingReasonCodesWithStrings(String... instanceFindingReasonCodes) {
            instanceFindingReasonCodesWithStrings(Arrays.asList(instanceFindingReasonCodes));
            return this;
        }

        @Override
        public final Builder instanceFindingReasonCodes(Collection<RDSInstanceFindingReasonCode> instanceFindingReasonCodes) {
            this.instanceFindingReasonCodes = RDSInstanceFindingReasonCodesCopier.copyEnumToString(instanceFindingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceFindingReasonCodes(RDSInstanceFindingReasonCode... instanceFindingReasonCodes) {
            instanceFindingReasonCodes(Arrays.asList(instanceFindingReasonCodes));
            return this;
        }

        public final Collection<String> getStorageFindingReasonCodes() {
            if (storageFindingReasonCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return storageFindingReasonCodes;
        }

        public final void setStorageFindingReasonCodes(Collection<String> storageFindingReasonCodes) {
            this.storageFindingReasonCodes = RDSStorageFindingReasonCodesCopier.copy(storageFindingReasonCodes);
        }

        @Override
        public final Builder storageFindingReasonCodesWithStrings(Collection<String> storageFindingReasonCodes) {
            this.storageFindingReasonCodes = RDSStorageFindingReasonCodesCopier.copy(storageFindingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder storageFindingReasonCodesWithStrings(String... storageFindingReasonCodes) {
            storageFindingReasonCodesWithStrings(Arrays.asList(storageFindingReasonCodes));
            return this;
        }

        @Override
        public final Builder storageFindingReasonCodes(Collection<RDSStorageFindingReasonCode> storageFindingReasonCodes) {
            this.storageFindingReasonCodes = RDSStorageFindingReasonCodesCopier.copyEnumToString(storageFindingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder storageFindingReasonCodes(RDSStorageFindingReasonCode... storageFindingReasonCodes) {
            storageFindingReasonCodes(Arrays.asList(storageFindingReasonCodes));
            return this;
        }

        public final List<RDSDBInstanceRecommendationOption.Builder> getInstanceRecommendationOptions() {
            List<RDSDBInstanceRecommendationOption.Builder> result = RDSDBInstanceRecommendationOptionsCopier
                    .copyToBuilder(this.instanceRecommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceRecommendationOptions(
                Collection<RDSDBInstanceRecommendationOption.BuilderImpl> instanceRecommendationOptions) {
            this.instanceRecommendationOptions = RDSDBInstanceRecommendationOptionsCopier
                    .copyFromBuilder(instanceRecommendationOptions);
        }

        @Override
        public final Builder instanceRecommendationOptions(
                Collection<RDSDBInstanceRecommendationOption> instanceRecommendationOptions) {
            this.instanceRecommendationOptions = RDSDBInstanceRecommendationOptionsCopier.copy(instanceRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceRecommendationOptions(RDSDBInstanceRecommendationOption... instanceRecommendationOptions) {
            instanceRecommendationOptions(Arrays.asList(instanceRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceRecommendationOptions(
                Consumer<RDSDBInstanceRecommendationOption.Builder>... instanceRecommendationOptions) {
            instanceRecommendationOptions(Stream.of(instanceRecommendationOptions)
                    .map(c -> RDSDBInstanceRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<RDSDBStorageRecommendationOption.Builder> getStorageRecommendationOptions() {
            List<RDSDBStorageRecommendationOption.Builder> result = RDSDBStorageRecommendationOptionsCopier
                    .copyToBuilder(this.storageRecommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStorageRecommendationOptions(
                Collection<RDSDBStorageRecommendationOption.BuilderImpl> storageRecommendationOptions) {
            this.storageRecommendationOptions = RDSDBStorageRecommendationOptionsCopier
                    .copyFromBuilder(storageRecommendationOptions);
        }

        @Override
        public final Builder storageRecommendationOptions(
                Collection<RDSDBStorageRecommendationOption> storageRecommendationOptions) {
            this.storageRecommendationOptions = RDSDBStorageRecommendationOptionsCopier.copy(storageRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder storageRecommendationOptions(RDSDBStorageRecommendationOption... storageRecommendationOptions) {
            storageRecommendationOptions(Arrays.asList(storageRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder storageRecommendationOptions(
                Consumer<RDSDBStorageRecommendationOption.Builder>... storageRecommendationOptions) {
            storageRecommendationOptions(Stream.of(storageRecommendationOptions)
                    .map(c -> RDSDBStorageRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<RDSDBUtilizationMetric.Builder> getUtilizationMetrics() {
            List<RDSDBUtilizationMetric.Builder> result = RDSDBUtilizationMetricsCopier.copyToBuilder(this.utilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationMetrics(Collection<RDSDBUtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = RDSDBUtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        @Override
        public final Builder utilizationMetrics(Collection<RDSDBUtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = RDSDBUtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(RDSDBUtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<RDSDBUtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics).map(c -> RDSDBUtilizationMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final RDSEffectiveRecommendationPreferences.Builder getEffectiveRecommendationPreferences() {
            return effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences.toBuilder() : null;
        }

        public final void setEffectiveRecommendationPreferences(
                RDSEffectiveRecommendationPreferences.BuilderImpl effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences
                    .build() : null;
        }

        @Override
        public final Builder effectiveRecommendationPreferences(
                RDSEffectiveRecommendationPreferences effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences;
            return this;
        }

        public final Double getLookbackPeriodInDays() {
            return lookbackPeriodInDays;
        }

        public final void setLookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
        }

        @Override
        public final Builder lookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
            return this;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public RDSDBRecommendation build() {
            return new RDSDBRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
