/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the effective recommendation preferences for a resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EffectiveRecommendationPreferences implements SdkPojo, Serializable,
        ToCopyableBuilder<EffectiveRecommendationPreferences.Builder, EffectiveRecommendationPreferences> {
    private static final SdkField<List<String>> CPU_VENDOR_ARCHITECTURES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("cpuVendorArchitectures")
            .getter(getter(EffectiveRecommendationPreferences::cpuVendorArchitecturesAsStrings))
            .setter(setter(Builder::cpuVendorArchitecturesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpuVendorArchitectures").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ENHANCED_INFRASTRUCTURE_METRICS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("enhancedInfrastructureMetrics")
            .getter(getter(EffectiveRecommendationPreferences::enhancedInfrastructureMetricsAsString))
            .setter(setter(Builder::enhancedInfrastructureMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enhancedInfrastructureMetrics")
                    .build()).build();

    private static final SdkField<String> INFERRED_WORKLOAD_TYPES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inferredWorkloadTypes")
            .getter(getter(EffectiveRecommendationPreferences::inferredWorkloadTypesAsString))
            .setter(setter(Builder::inferredWorkloadTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferredWorkloadTypes").build())
            .build();

    private static final SdkField<ExternalMetricsPreference> EXTERNAL_METRICS_PREFERENCE_FIELD = SdkField
            .<ExternalMetricsPreference> builder(MarshallingType.SDK_POJO).memberName("externalMetricsPreference")
            .getter(getter(EffectiveRecommendationPreferences::externalMetricsPreference))
            .setter(setter(Builder::externalMetricsPreference)).constructor(ExternalMetricsPreference::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("externalMetricsPreference").build())
            .build();

    private static final SdkField<String> LOOK_BACK_PERIOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("lookBackPeriod").getter(getter(EffectiveRecommendationPreferences::lookBackPeriodAsString))
            .setter(setter(Builder::lookBackPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriod").build()).build();

    private static final SdkField<List<UtilizationPreference>> UTILIZATION_PREFERENCES_FIELD = SdkField
            .<List<UtilizationPreference>> builder(MarshallingType.LIST)
            .memberName("utilizationPreferences")
            .getter(getter(EffectiveRecommendationPreferences::utilizationPreferences))
            .setter(setter(Builder::utilizationPreferences))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationPreferences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationPreference> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationPreference::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<EffectivePreferredResource>> PREFERRED_RESOURCES_FIELD = SdkField
            .<List<EffectivePreferredResource>> builder(MarshallingType.LIST)
            .memberName("preferredResources")
            .getter(getter(EffectiveRecommendationPreferences::preferredResources))
            .setter(setter(Builder::preferredResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("preferredResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EffectivePreferredResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(EffectivePreferredResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<InstanceSavingsEstimationMode> SAVINGS_ESTIMATION_MODE_FIELD = SdkField
            .<InstanceSavingsEstimationMode> builder(MarshallingType.SDK_POJO).memberName("savingsEstimationMode")
            .getter(getter(EffectiveRecommendationPreferences::savingsEstimationMode))
            .setter(setter(Builder::savingsEstimationMode)).constructor(InstanceSavingsEstimationMode::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("savingsEstimationMode").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CPU_VENDOR_ARCHITECTURES_FIELD, ENHANCED_INFRASTRUCTURE_METRICS_FIELD, INFERRED_WORKLOAD_TYPES_FIELD,
            EXTERNAL_METRICS_PREFERENCE_FIELD, LOOK_BACK_PERIOD_FIELD, UTILIZATION_PREFERENCES_FIELD, PREFERRED_RESOURCES_FIELD,
            SAVINGS_ESTIMATION_MODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> cpuVendorArchitectures;

    private final String enhancedInfrastructureMetrics;

    private final String inferredWorkloadTypes;

    private final ExternalMetricsPreference externalMetricsPreference;

    private final String lookBackPeriod;

    private final List<UtilizationPreference> utilizationPreferences;

    private final List<EffectivePreferredResource> preferredResources;

    private final InstanceSavingsEstimationMode savingsEstimationMode;

    private EffectiveRecommendationPreferences(BuilderImpl builder) {
        this.cpuVendorArchitectures = builder.cpuVendorArchitectures;
        this.enhancedInfrastructureMetrics = builder.enhancedInfrastructureMetrics;
        this.inferredWorkloadTypes = builder.inferredWorkloadTypes;
        this.externalMetricsPreference = builder.externalMetricsPreference;
        this.lookBackPeriod = builder.lookBackPeriod;
        this.utilizationPreferences = builder.utilizationPreferences;
        this.preferredResources = builder.preferredResources;
        this.savingsEstimationMode = builder.savingsEstimationMode;
    }

    /**
     * <p>
     * Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.
     * </p>
     * <p>
     * For example, when you specify <code>AWS_ARM64</code> with:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute Optimizer
     * returns recommendations that consist of Graviton instance types only.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization metrics
     * for Graviton instance type recommendations only.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request, Compute
     * Optimizer exports recommendations that consist of Graviton instance types only.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCpuVendorArchitectures} method.
     * </p>
     * 
     * @return Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.</p>
     *         <p>
     *         For example, when you specify <code>AWS_ARM64</code> with:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
     *         Optimizer returns recommendations that consist of Graviton instance types only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
     *         metrics for Graviton instance type recommendations only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request,
     *         Compute Optimizer exports recommendations that consist of Graviton instance types only.
     *         </p>
     *         </li>
     */
    public final List<CpuVendorArchitecture> cpuVendorArchitectures() {
        return CpuVendorArchitecturesCopier.copyStringToEnum(cpuVendorArchitectures);
    }

    /**
     * For responses, this returns true if the service returned a value for the CpuVendorArchitectures property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCpuVendorArchitectures() {
        return cpuVendorArchitectures != null && !(cpuVendorArchitectures instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.
     * </p>
     * <p>
     * For example, when you specify <code>AWS_ARM64</code> with:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute Optimizer
     * returns recommendations that consist of Graviton instance types only.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization metrics
     * for Graviton instance type recommendations only.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request, Compute
     * Optimizer exports recommendations that consist of Graviton instance types only.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCpuVendorArchitectures} method.
     * </p>
     * 
     * @return Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.</p>
     *         <p>
     *         For example, when you specify <code>AWS_ARM64</code> with:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
     *         Optimizer returns recommendations that consist of Graviton instance types only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
     *         metrics for Graviton instance type recommendations only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request,
     *         Compute Optimizer exports recommendations that consist of Graviton instance types only.
     *         </p>
     *         </li>
     */
    public final List<String> cpuVendorArchitecturesAsStrings() {
        return cpuVendorArchitectures;
    }

    /**
     * <p>
     * Describes the activation status of the enhanced infrastructure metrics preference.
     * </p>
     * <p>
     * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh, and
     * a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
     * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enhancedInfrastructureMetrics} will return {@link EnhancedInfrastructureMetrics#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #enhancedInfrastructureMetricsAsString}.
     * </p>
     * 
     * @return Describes the activation status of the enhanced infrastructure metrics preference.</p>
     *         <p>
     *         A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
     *         refresh, and a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
     *         >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * @see EnhancedInfrastructureMetrics
     */
    public final EnhancedInfrastructureMetrics enhancedInfrastructureMetrics() {
        return EnhancedInfrastructureMetrics.fromValue(enhancedInfrastructureMetrics);
    }

    /**
     * <p>
     * Describes the activation status of the enhanced infrastructure metrics preference.
     * </p>
     * <p>
     * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh, and
     * a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
     * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enhancedInfrastructureMetrics} will return {@link EnhancedInfrastructureMetrics#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #enhancedInfrastructureMetricsAsString}.
     * </p>
     * 
     * @return Describes the activation status of the enhanced infrastructure metrics preference.</p>
     *         <p>
     *         A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
     *         refresh, and a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
     *         >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * @see EnhancedInfrastructureMetrics
     */
    public final String enhancedInfrastructureMetricsAsString() {
        return enhancedInfrastructureMetrics;
    }

    /**
     * <p>
     * Describes the activation status of the inferred workload types preference.
     * </p>
     * <p>
     * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh. A
     * status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inferredWorkloadTypes} will return {@link InferredWorkloadTypesPreference#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inferredWorkloadTypesAsString}.
     * </p>
     * 
     * @return Describes the activation status of the inferred workload types preference.</p>
     *         <p>
     *         A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
     *         refresh. A status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     * @see InferredWorkloadTypesPreference
     */
    public final InferredWorkloadTypesPreference inferredWorkloadTypes() {
        return InferredWorkloadTypesPreference.fromValue(inferredWorkloadTypes);
    }

    /**
     * <p>
     * Describes the activation status of the inferred workload types preference.
     * </p>
     * <p>
     * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh. A
     * status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inferredWorkloadTypes} will return {@link InferredWorkloadTypesPreference#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inferredWorkloadTypesAsString}.
     * </p>
     * 
     * @return Describes the activation status of the inferred workload types preference.</p>
     *         <p>
     *         A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
     *         refresh. A status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
     * @see InferredWorkloadTypesPreference
     */
    public final String inferredWorkloadTypesAsString() {
        return inferredWorkloadTypes;
    }

    /**
     * <p>
     * An object that describes the external metrics recommendation preference.
     * </p>
     * <p>
     * If the preference is applied in the latest recommendation refresh, an object with a valid <code>source</code>
     * value appears in the response. If the preference isn't applied to the recommendations already, then this object
     * doesn't appear in the response.
     * </p>
     * 
     * @return An object that describes the external metrics recommendation preference. </p>
     *         <p>
     *         If the preference is applied in the latest recommendation refresh, an object with a valid
     *         <code>source</code> value appears in the response. If the preference isn't applied to the recommendations
     *         already, then this object doesn't appear in the response.
     */
    public final ExternalMetricsPreference externalMetricsPreference() {
        return externalMetricsPreference;
    }

    /**
     * <p>
     * The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lookBackPeriod}
     * will return {@link LookBackPeriodPreference#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lookBackPeriodAsString}.
     * </p>
     * 
     * @return The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
     * @see LookBackPeriodPreference
     */
    public final LookBackPeriodPreference lookBackPeriod() {
        return LookBackPeriodPreference.fromValue(lookBackPeriod);
    }

    /**
     * <p>
     * The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lookBackPeriod}
     * will return {@link LookBackPeriodPreference#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lookBackPeriodAsString}.
     * </p>
     * 
     * @return The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
     * @see LookBackPeriodPreference
     */
    public final String lookBackPeriodAsString() {
        return lookBackPeriod;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationPreferences property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasUtilizationPreferences() {
        return utilizationPreferences != null && !(utilizationPreferences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource’s CPU and memory utilization preferences, such as threshold and headroom, that are used to generate
     * rightsizing recommendations.
     * </p>
     * <note>
     * <p>
     * This preference is only available for the Amazon EC2 instance resource type.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationPreferences} method.
     * </p>
     * 
     * @return The resource’s CPU and memory utilization preferences, such as threshold and headroom, that are used to
     *         generate rightsizing recommendations. </p> <note>
     *         <p>
     *         This preference is only available for the Amazon EC2 instance resource type.
     *         </p>
     */
    public final List<UtilizationPreference> utilizationPreferences() {
        return utilizationPreferences;
    }

    /**
     * For responses, this returns true if the service returned a value for the PreferredResources property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPreferredResources() {
        return preferredResources != null && !(preferredResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource type values that are considered as candidates when generating rightsizing recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPreferredResources} method.
     * </p>
     * 
     * @return The resource type values that are considered as candidates when generating rightsizing recommendations.
     */
    public final List<EffectivePreferredResource> preferredResources() {
        return preferredResources;
    }

    /**
     * <p>
     * Describes the savings estimation mode applied for calculating savings opportunity for a resource.
     * </p>
     * 
     * @return Describes the savings estimation mode applied for calculating savings opportunity for a resource.
     */
    public final InstanceSavingsEstimationMode savingsEstimationMode() {
        return savingsEstimationMode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasCpuVendorArchitectures() ? cpuVendorArchitecturesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(enhancedInfrastructureMetricsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inferredWorkloadTypesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(externalMetricsPreference());
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationPreferences() ? utilizationPreferences() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPreferredResources() ? preferredResources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(savingsEstimationMode());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EffectiveRecommendationPreferences)) {
            return false;
        }
        EffectiveRecommendationPreferences other = (EffectiveRecommendationPreferences) obj;
        return hasCpuVendorArchitectures() == other.hasCpuVendorArchitectures()
                && Objects.equals(cpuVendorArchitecturesAsStrings(), other.cpuVendorArchitecturesAsStrings())
                && Objects.equals(enhancedInfrastructureMetricsAsString(), other.enhancedInfrastructureMetricsAsString())
                && Objects.equals(inferredWorkloadTypesAsString(), other.inferredWorkloadTypesAsString())
                && Objects.equals(externalMetricsPreference(), other.externalMetricsPreference())
                && Objects.equals(lookBackPeriodAsString(), other.lookBackPeriodAsString())
                && hasUtilizationPreferences() == other.hasUtilizationPreferences()
                && Objects.equals(utilizationPreferences(), other.utilizationPreferences())
                && hasPreferredResources() == other.hasPreferredResources()
                && Objects.equals(preferredResources(), other.preferredResources())
                && Objects.equals(savingsEstimationMode(), other.savingsEstimationMode());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EffectiveRecommendationPreferences")
                .add("CpuVendorArchitectures", hasCpuVendorArchitectures() ? cpuVendorArchitecturesAsStrings() : null)
                .add("EnhancedInfrastructureMetrics", enhancedInfrastructureMetricsAsString())
                .add("InferredWorkloadTypes", inferredWorkloadTypesAsString())
                .add("ExternalMetricsPreference", externalMetricsPreference()).add("LookBackPeriod", lookBackPeriodAsString())
                .add("UtilizationPreferences", hasUtilizationPreferences() ? utilizationPreferences() : null)
                .add("PreferredResources", hasPreferredResources() ? preferredResources() : null)
                .add("SavingsEstimationMode", savingsEstimationMode()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cpuVendorArchitectures":
            return Optional.ofNullable(clazz.cast(cpuVendorArchitecturesAsStrings()));
        case "enhancedInfrastructureMetrics":
            return Optional.ofNullable(clazz.cast(enhancedInfrastructureMetricsAsString()));
        case "inferredWorkloadTypes":
            return Optional.ofNullable(clazz.cast(inferredWorkloadTypesAsString()));
        case "externalMetricsPreference":
            return Optional.ofNullable(clazz.cast(externalMetricsPreference()));
        case "lookBackPeriod":
            return Optional.ofNullable(clazz.cast(lookBackPeriodAsString()));
        case "utilizationPreferences":
            return Optional.ofNullable(clazz.cast(utilizationPreferences()));
        case "preferredResources":
            return Optional.ofNullable(clazz.cast(preferredResources()));
        case "savingsEstimationMode":
            return Optional.ofNullable(clazz.cast(savingsEstimationMode()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EffectiveRecommendationPreferences, T> g) {
        return obj -> g.apply((EffectiveRecommendationPreferences) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EffectiveRecommendationPreferences> {
        /**
         * <p>
         * Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.
         * </p>
         * <p>
         * For example, when you specify <code>AWS_ARM64</code> with:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer returns recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         * metrics for Graviton instance type recommendations only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer exports recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cpuVendorArchitectures
         *        Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.</p>
         *        <p>
         *        For example, when you specify <code>AWS_ARM64</code> with:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         *        Optimizer returns recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         *        metrics for Graviton instance type recommendations only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request,
         *        Compute Optimizer exports recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuVendorArchitecturesWithStrings(Collection<String> cpuVendorArchitectures);

        /**
         * <p>
         * Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.
         * </p>
         * <p>
         * For example, when you specify <code>AWS_ARM64</code> with:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer returns recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         * metrics for Graviton instance type recommendations only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer exports recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cpuVendorArchitectures
         *        Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.</p>
         *        <p>
         *        For example, when you specify <code>AWS_ARM64</code> with:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         *        Optimizer returns recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         *        metrics for Graviton instance type recommendations only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request,
         *        Compute Optimizer exports recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuVendorArchitecturesWithStrings(String... cpuVendorArchitectures);

        /**
         * <p>
         * Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.
         * </p>
         * <p>
         * For example, when you specify <code>AWS_ARM64</code> with:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer returns recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         * metrics for Graviton instance type recommendations only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer exports recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cpuVendorArchitectures
         *        Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.</p>
         *        <p>
         *        For example, when you specify <code>AWS_ARM64</code> with:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         *        Optimizer returns recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         *        metrics for Graviton instance type recommendations only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request,
         *        Compute Optimizer exports recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuVendorArchitectures(Collection<CpuVendorArchitecture> cpuVendorArchitectures);

        /**
         * <p>
         * Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.
         * </p>
         * <p>
         * For example, when you specify <code>AWS_ARM64</code> with:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer returns recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         * metrics for Graviton instance type recommendations only.
         * </p>
         * </li>
         * <li>
         * <p>
         * A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request, Compute
         * Optimizer exports recommendations that consist of Graviton instance types only.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cpuVendorArchitectures
         *        Describes the CPU vendor and architecture for an instance or Auto Scaling group recommendations.</p>
         *        <p>
         *        For example, when you specify <code>AWS_ARM64</code> with:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A <a>GetEC2InstanceRecommendations</a> or <a>GetAutoScalingGroupRecommendations</a> request, Compute
         *        Optimizer returns recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>GetEC2RecommendationProjectedMetrics</a> request, Compute Optimizer returns projected utilization
         *        metrics for Graviton instance type recommendations only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <a>ExportEC2InstanceRecommendations</a> or <a>ExportAutoScalingGroupRecommendations</a> request,
         *        Compute Optimizer exports recommendations that consist of Graviton instance types only.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuVendorArchitectures(CpuVendorArchitecture... cpuVendorArchitectures);

        /**
         * <p>
         * Describes the activation status of the enhanced infrastructure metrics preference.
         * </p>
         * <p>
         * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh,
         * and a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
         * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * </p>
         * 
         * @param enhancedInfrastructureMetrics
         *        Describes the activation status of the enhanced infrastructure metrics preference.</p>
         *        <p>
         *        A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
         *        refresh, and a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
         *        >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * @see EnhancedInfrastructureMetrics
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnhancedInfrastructureMetrics
         */
        Builder enhancedInfrastructureMetrics(String enhancedInfrastructureMetrics);

        /**
         * <p>
         * Describes the activation status of the enhanced infrastructure metrics preference.
         * </p>
         * <p>
         * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh,
         * and a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
         * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * </p>
         * 
         * @param enhancedInfrastructureMetrics
         *        Describes the activation status of the enhanced infrastructure metrics preference.</p>
         *        <p>
         *        A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
         *        refresh, and a status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
         *        >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * @see EnhancedInfrastructureMetrics
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnhancedInfrastructureMetrics
         */
        Builder enhancedInfrastructureMetrics(EnhancedInfrastructureMetrics enhancedInfrastructureMetrics);

        /**
         * <p>
         * Describes the activation status of the inferred workload types preference.
         * </p>
         * <p>
         * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh.
         * A status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         * </p>
         * 
         * @param inferredWorkloadTypes
         *        Describes the activation status of the inferred workload types preference.</p>
         *        <p>
         *        A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
         *        refresh. A status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         * @see InferredWorkloadTypesPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InferredWorkloadTypesPreference
         */
        Builder inferredWorkloadTypes(String inferredWorkloadTypes);

        /**
         * <p>
         * Describes the activation status of the inferred workload types preference.
         * </p>
         * <p>
         * A status of <code>Active</code> confirms that the preference is applied in the latest recommendation refresh.
         * A status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         * </p>
         * 
         * @param inferredWorkloadTypes
         *        Describes the activation status of the inferred workload types preference.</p>
         *        <p>
         *        A status of <code>Active</code> confirms that the preference is applied in the latest recommendation
         *        refresh. A status of <code>Inactive</code> confirms that it's not yet applied to recommendations.
         * @see InferredWorkloadTypesPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InferredWorkloadTypesPreference
         */
        Builder inferredWorkloadTypes(InferredWorkloadTypesPreference inferredWorkloadTypes);

        /**
         * <p>
         * An object that describes the external metrics recommendation preference.
         * </p>
         * <p>
         * If the preference is applied in the latest recommendation refresh, an object with a valid <code>source</code>
         * value appears in the response. If the preference isn't applied to the recommendations already, then this
         * object doesn't appear in the response.
         * </p>
         * 
         * @param externalMetricsPreference
         *        An object that describes the external metrics recommendation preference. </p>
         *        <p>
         *        If the preference is applied in the latest recommendation refresh, an object with a valid
         *        <code>source</code> value appears in the response. If the preference isn't applied to the
         *        recommendations already, then this object doesn't appear in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalMetricsPreference(ExternalMetricsPreference externalMetricsPreference);

        /**
         * <p>
         * An object that describes the external metrics recommendation preference.
         * </p>
         * <p>
         * If the preference is applied in the latest recommendation refresh, an object with a valid <code>source</code>
         * value appears in the response. If the preference isn't applied to the recommendations already, then this
         * object doesn't appear in the response.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExternalMetricsPreference.Builder}
         * avoiding the need to create one manually via {@link ExternalMetricsPreference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExternalMetricsPreference.Builder#build()} is called immediately
         * and its result is passed to {@link #externalMetricsPreference(ExternalMetricsPreference)}.
         * 
         * @param externalMetricsPreference
         *        a consumer that will call methods on {@link ExternalMetricsPreference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #externalMetricsPreference(ExternalMetricsPreference)
         */
        default Builder externalMetricsPreference(Consumer<ExternalMetricsPreference.Builder> externalMetricsPreference) {
            return externalMetricsPreference(ExternalMetricsPreference.builder().applyMutation(externalMetricsPreference).build());
        }

        /**
         * <p>
         * The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
         * </p>
         * 
         * @param lookBackPeriod
         *        The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
         * @see LookBackPeriodPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LookBackPeriodPreference
         */
        Builder lookBackPeriod(String lookBackPeriod);

        /**
         * <p>
         * The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
         * </p>
         * 
         * @param lookBackPeriod
         *        The number of days the utilization metrics of the Amazon Web Services resource are analyzed.
         * @see LookBackPeriodPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LookBackPeriodPreference
         */
        Builder lookBackPeriod(LookBackPeriodPreference lookBackPeriod);

        /**
         * <p>
         * The resource’s CPU and memory utilization preferences, such as threshold and headroom, that are used to
         * generate rightsizing recommendations.
         * </p>
         * <note>
         * <p>
         * This preference is only available for the Amazon EC2 instance resource type.
         * </p>
         * </note>
         * 
         * @param utilizationPreferences
         *        The resource’s CPU and memory utilization preferences, such as threshold and headroom, that are used
         *        to generate rightsizing recommendations. </p> <note>
         *        <p>
         *        This preference is only available for the Amazon EC2 instance resource type.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationPreferences(Collection<UtilizationPreference> utilizationPreferences);

        /**
         * <p>
         * The resource’s CPU and memory utilization preferences, such as threshold and headroom, that are used to
         * generate rightsizing recommendations.
         * </p>
         * <note>
         * <p>
         * This preference is only available for the Amazon EC2 instance resource type.
         * </p>
         * </note>
         * 
         * @param utilizationPreferences
         *        The resource’s CPU and memory utilization preferences, such as threshold and headroom, that are used
         *        to generate rightsizing recommendations. </p> <note>
         *        <p>
         *        This preference is only available for the Amazon EC2 instance resource type.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationPreferences(UtilizationPreference... utilizationPreferences);

        /**
         * <p>
         * The resource’s CPU and memory utilization preferences, such as threshold and headroom, that are used to
         * generate rightsizing recommendations.
         * </p>
         * <note>
         * <p>
         * This preference is only available for the Amazon EC2 instance resource type.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference.Builder#build()} is
         * called immediately and its result is passed to {@link #utilizationPreferences(List<UtilizationPreference>)}.
         * 
         * @param utilizationPreferences
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationPreferences(java.util.Collection<UtilizationPreference>)
         */
        Builder utilizationPreferences(Consumer<UtilizationPreference.Builder>... utilizationPreferences);

        /**
         * <p>
         * The resource type values that are considered as candidates when generating rightsizing recommendations.
         * </p>
         * 
         * @param preferredResources
         *        The resource type values that are considered as candidates when generating rightsizing
         *        recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredResources(Collection<EffectivePreferredResource> preferredResources);

        /**
         * <p>
         * The resource type values that are considered as candidates when generating rightsizing recommendations.
         * </p>
         * 
         * @param preferredResources
         *        The resource type values that are considered as candidates when generating rightsizing
         *        recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredResources(EffectivePreferredResource... preferredResources);

        /**
         * <p>
         * The resource type values that are considered as candidates when generating rightsizing recommendations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource.Builder#build()} is
         * called immediately and its result is passed to {@link #preferredResources(List<EffectivePreferredResource>)}.
         * 
         * @param preferredResources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #preferredResources(java.util.Collection<EffectivePreferredResource>)
         */
        Builder preferredResources(Consumer<EffectivePreferredResource.Builder>... preferredResources);

        /**
         * <p>
         * Describes the savings estimation mode applied for calculating savings opportunity for a resource.
         * </p>
         * 
         * @param savingsEstimationMode
         *        Describes the savings estimation mode applied for calculating savings opportunity for a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder savingsEstimationMode(InstanceSavingsEstimationMode savingsEstimationMode);

        /**
         * <p>
         * Describes the savings estimation mode applied for calculating savings opportunity for a resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link InstanceSavingsEstimationMode.Builder}
         * avoiding the need to create one manually via {@link InstanceSavingsEstimationMode#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceSavingsEstimationMode.Builder#build()} is called
         * immediately and its result is passed to {@link #savingsEstimationMode(InstanceSavingsEstimationMode)}.
         * 
         * @param savingsEstimationMode
         *        a consumer that will call methods on {@link InstanceSavingsEstimationMode.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #savingsEstimationMode(InstanceSavingsEstimationMode)
         */
        default Builder savingsEstimationMode(Consumer<InstanceSavingsEstimationMode.Builder> savingsEstimationMode) {
            return savingsEstimationMode(InstanceSavingsEstimationMode.builder().applyMutation(savingsEstimationMode).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<String> cpuVendorArchitectures = DefaultSdkAutoConstructList.getInstance();

        private String enhancedInfrastructureMetrics;

        private String inferredWorkloadTypes;

        private ExternalMetricsPreference externalMetricsPreference;

        private String lookBackPeriod;

        private List<UtilizationPreference> utilizationPreferences = DefaultSdkAutoConstructList.getInstance();

        private List<EffectivePreferredResource> preferredResources = DefaultSdkAutoConstructList.getInstance();

        private InstanceSavingsEstimationMode savingsEstimationMode;

        private BuilderImpl() {
        }

        private BuilderImpl(EffectiveRecommendationPreferences model) {
            cpuVendorArchitecturesWithStrings(model.cpuVendorArchitectures);
            enhancedInfrastructureMetrics(model.enhancedInfrastructureMetrics);
            inferredWorkloadTypes(model.inferredWorkloadTypes);
            externalMetricsPreference(model.externalMetricsPreference);
            lookBackPeriod(model.lookBackPeriod);
            utilizationPreferences(model.utilizationPreferences);
            preferredResources(model.preferredResources);
            savingsEstimationMode(model.savingsEstimationMode);
        }

        public final Collection<String> getCpuVendorArchitectures() {
            if (cpuVendorArchitectures instanceof SdkAutoConstructList) {
                return null;
            }
            return cpuVendorArchitectures;
        }

        public final void setCpuVendorArchitectures(Collection<String> cpuVendorArchitectures) {
            this.cpuVendorArchitectures = CpuVendorArchitecturesCopier.copy(cpuVendorArchitectures);
        }

        @Override
        public final Builder cpuVendorArchitecturesWithStrings(Collection<String> cpuVendorArchitectures) {
            this.cpuVendorArchitectures = CpuVendorArchitecturesCopier.copy(cpuVendorArchitectures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cpuVendorArchitecturesWithStrings(String... cpuVendorArchitectures) {
            cpuVendorArchitecturesWithStrings(Arrays.asList(cpuVendorArchitectures));
            return this;
        }

        @Override
        public final Builder cpuVendorArchitectures(Collection<CpuVendorArchitecture> cpuVendorArchitectures) {
            this.cpuVendorArchitectures = CpuVendorArchitecturesCopier.copyEnumToString(cpuVendorArchitectures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cpuVendorArchitectures(CpuVendorArchitecture... cpuVendorArchitectures) {
            cpuVendorArchitectures(Arrays.asList(cpuVendorArchitectures));
            return this;
        }

        public final String getEnhancedInfrastructureMetrics() {
            return enhancedInfrastructureMetrics;
        }

        public final void setEnhancedInfrastructureMetrics(String enhancedInfrastructureMetrics) {
            this.enhancedInfrastructureMetrics = enhancedInfrastructureMetrics;
        }

        @Override
        public final Builder enhancedInfrastructureMetrics(String enhancedInfrastructureMetrics) {
            this.enhancedInfrastructureMetrics = enhancedInfrastructureMetrics;
            return this;
        }

        @Override
        public final Builder enhancedInfrastructureMetrics(EnhancedInfrastructureMetrics enhancedInfrastructureMetrics) {
            this.enhancedInfrastructureMetrics(enhancedInfrastructureMetrics == null ? null : enhancedInfrastructureMetrics
                    .toString());
            return this;
        }

        public final String getInferredWorkloadTypes() {
            return inferredWorkloadTypes;
        }

        public final void setInferredWorkloadTypes(String inferredWorkloadTypes) {
            this.inferredWorkloadTypes = inferredWorkloadTypes;
        }

        @Override
        public final Builder inferredWorkloadTypes(String inferredWorkloadTypes) {
            this.inferredWorkloadTypes = inferredWorkloadTypes;
            return this;
        }

        @Override
        public final Builder inferredWorkloadTypes(InferredWorkloadTypesPreference inferredWorkloadTypes) {
            this.inferredWorkloadTypes(inferredWorkloadTypes == null ? null : inferredWorkloadTypes.toString());
            return this;
        }

        public final ExternalMetricsPreference.Builder getExternalMetricsPreference() {
            return externalMetricsPreference != null ? externalMetricsPreference.toBuilder() : null;
        }

        public final void setExternalMetricsPreference(ExternalMetricsPreference.BuilderImpl externalMetricsPreference) {
            this.externalMetricsPreference = externalMetricsPreference != null ? externalMetricsPreference.build() : null;
        }

        @Override
        public final Builder externalMetricsPreference(ExternalMetricsPreference externalMetricsPreference) {
            this.externalMetricsPreference = externalMetricsPreference;
            return this;
        }

        public final String getLookBackPeriod() {
            return lookBackPeriod;
        }

        public final void setLookBackPeriod(String lookBackPeriod) {
            this.lookBackPeriod = lookBackPeriod;
        }

        @Override
        public final Builder lookBackPeriod(String lookBackPeriod) {
            this.lookBackPeriod = lookBackPeriod;
            return this;
        }

        @Override
        public final Builder lookBackPeriod(LookBackPeriodPreference lookBackPeriod) {
            this.lookBackPeriod(lookBackPeriod == null ? null : lookBackPeriod.toString());
            return this;
        }

        public final List<UtilizationPreference.Builder> getUtilizationPreferences() {
            List<UtilizationPreference.Builder> result = UtilizationPreferencesCopier.copyToBuilder(this.utilizationPreferences);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationPreferences(Collection<UtilizationPreference.BuilderImpl> utilizationPreferences) {
            this.utilizationPreferences = UtilizationPreferencesCopier.copyFromBuilder(utilizationPreferences);
        }

        @Override
        public final Builder utilizationPreferences(Collection<UtilizationPreference> utilizationPreferences) {
            this.utilizationPreferences = UtilizationPreferencesCopier.copy(utilizationPreferences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationPreferences(UtilizationPreference... utilizationPreferences) {
            utilizationPreferences(Arrays.asList(utilizationPreferences));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationPreferences(Consumer<UtilizationPreference.Builder>... utilizationPreferences) {
            utilizationPreferences(Stream.of(utilizationPreferences)
                    .map(c -> UtilizationPreference.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<EffectivePreferredResource.Builder> getPreferredResources() {
            List<EffectivePreferredResource.Builder> result = EffectivePreferredResourcesCopier
                    .copyToBuilder(this.preferredResources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPreferredResources(Collection<EffectivePreferredResource.BuilderImpl> preferredResources) {
            this.preferredResources = EffectivePreferredResourcesCopier.copyFromBuilder(preferredResources);
        }

        @Override
        public final Builder preferredResources(Collection<EffectivePreferredResource> preferredResources) {
            this.preferredResources = EffectivePreferredResourcesCopier.copy(preferredResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredResources(EffectivePreferredResource... preferredResources) {
            preferredResources(Arrays.asList(preferredResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredResources(Consumer<EffectivePreferredResource.Builder>... preferredResources) {
            preferredResources(Stream.of(preferredResources)
                    .map(c -> EffectivePreferredResource.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final InstanceSavingsEstimationMode.Builder getSavingsEstimationMode() {
            return savingsEstimationMode != null ? savingsEstimationMode.toBuilder() : null;
        }

        public final void setSavingsEstimationMode(InstanceSavingsEstimationMode.BuilderImpl savingsEstimationMode) {
            this.savingsEstimationMode = savingsEstimationMode != null ? savingsEstimationMode.build() : null;
        }

        @Override
        public final Builder savingsEstimationMode(InstanceSavingsEstimationMode savingsEstimationMode) {
            this.savingsEstimationMode = savingsEstimationMode;
            return this;
        }

        @Override
        public EffectiveRecommendationPreferences build() {
            return new EffectiveRecommendationPreferences(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
