/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list that allows you to specify, or override, the source revision for a pipeline execution that's being started. A
 * source revision is the version with all the changes to your application code, or source artifact, for the pipeline
 * execution.
 * </p>
 * <note>
 * <p>
 * For the <code>S3_OBJECT_VERSION_ID</code> and <code>S3_OBJECT_KEY</code> types of source revisions, either of the
 * types can be used independently, or they can be used together to override the source with a specific ObjectKey and
 * VersionID.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceRevisionOverride implements SdkPojo, Serializable,
        ToCopyableBuilder<SourceRevisionOverride.Builder, SourceRevisionOverride> {
    private static final SdkField<String> ACTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionName").getter(getter(SourceRevisionOverride::actionName)).setter(setter(Builder::actionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionName").build()).build();

    private static final SdkField<String> REVISION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("revisionType").getter(getter(SourceRevisionOverride::revisionTypeAsString))
            .setter(setter(Builder::revisionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revisionType").build()).build();

    private static final SdkField<String> REVISION_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("revisionValue").getter(getter(SourceRevisionOverride::revisionValue))
            .setter(setter(Builder::revisionValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revisionValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_NAME_FIELD,
            REVISION_TYPE_FIELD, REVISION_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String actionName;

    private final String revisionType;

    private final String revisionValue;

    private SourceRevisionOverride(BuilderImpl builder) {
        this.actionName = builder.actionName;
        this.revisionType = builder.revisionType;
        this.revisionValue = builder.revisionValue;
    }

    /**
     * <p>
     * The name of the action where the override will be applied.
     * </p>
     * 
     * @return The name of the action where the override will be applied.
     */
    public final String actionName() {
        return actionName;
    }

    /**
     * <p>
     * The type of source revision, based on the source provider. For example, the revision type for the CodeCommit
     * action provider is the commit ID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #revisionType} will
     * return {@link SourceRevisionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #revisionTypeAsString}.
     * </p>
     * 
     * @return The type of source revision, based on the source provider. For example, the revision type for the
     *         CodeCommit action provider is the commit ID.
     * @see SourceRevisionType
     */
    public final SourceRevisionType revisionType() {
        return SourceRevisionType.fromValue(revisionType);
    }

    /**
     * <p>
     * The type of source revision, based on the source provider. For example, the revision type for the CodeCommit
     * action provider is the commit ID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #revisionType} will
     * return {@link SourceRevisionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #revisionTypeAsString}.
     * </p>
     * 
     * @return The type of source revision, based on the source provider. For example, the revision type for the
     *         CodeCommit action provider is the commit ID.
     * @see SourceRevisionType
     */
    public final String revisionTypeAsString() {
        return revisionType;
    }

    /**
     * <p>
     * The source revision, or version of your source artifact, with the changes that you want to run in the pipeline
     * execution.
     * </p>
     * 
     * @return The source revision, or version of your source artifact, with the changes that you want to run in the
     *         pipeline execution.
     */
    public final String revisionValue() {
        return revisionValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionName());
        hashCode = 31 * hashCode + Objects.hashCode(revisionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(revisionValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceRevisionOverride)) {
            return false;
        }
        SourceRevisionOverride other = (SourceRevisionOverride) obj;
        return Objects.equals(actionName(), other.actionName())
                && Objects.equals(revisionTypeAsString(), other.revisionTypeAsString())
                && Objects.equals(revisionValue(), other.revisionValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SourceRevisionOverride").add("ActionName", actionName())
                .add("RevisionType", revisionTypeAsString()).add("RevisionValue", revisionValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "actionName":
            return Optional.ofNullable(clazz.cast(actionName()));
        case "revisionType":
            return Optional.ofNullable(clazz.cast(revisionTypeAsString()));
        case "revisionValue":
            return Optional.ofNullable(clazz.cast(revisionValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SourceRevisionOverride, T> g) {
        return obj -> g.apply((SourceRevisionOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SourceRevisionOverride> {
        /**
         * <p>
         * The name of the action where the override will be applied.
         * </p>
         * 
         * @param actionName
         *        The name of the action where the override will be applied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionName(String actionName);

        /**
         * <p>
         * The type of source revision, based on the source provider. For example, the revision type for the CodeCommit
         * action provider is the commit ID.
         * </p>
         * 
         * @param revisionType
         *        The type of source revision, based on the source provider. For example, the revision type for the
         *        CodeCommit action provider is the commit ID.
         * @see SourceRevisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceRevisionType
         */
        Builder revisionType(String revisionType);

        /**
         * <p>
         * The type of source revision, based on the source provider. For example, the revision type for the CodeCommit
         * action provider is the commit ID.
         * </p>
         * 
         * @param revisionType
         *        The type of source revision, based on the source provider. For example, the revision type for the
         *        CodeCommit action provider is the commit ID.
         * @see SourceRevisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceRevisionType
         */
        Builder revisionType(SourceRevisionType revisionType);

        /**
         * <p>
         * The source revision, or version of your source artifact, with the changes that you want to run in the
         * pipeline execution.
         * </p>
         * 
         * @param revisionValue
         *        The source revision, or version of your source artifact, with the changes that you want to run in the
         *        pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisionValue(String revisionValue);
    }

    static final class BuilderImpl implements Builder {
        private String actionName;

        private String revisionType;

        private String revisionValue;

        private BuilderImpl() {
        }

        private BuilderImpl(SourceRevisionOverride model) {
            actionName(model.actionName);
            revisionType(model.revisionType);
            revisionValue(model.revisionValue);
        }

        public final String getActionName() {
            return actionName;
        }

        public final void setActionName(String actionName) {
            this.actionName = actionName;
        }

        @Override
        public final Builder actionName(String actionName) {
            this.actionName = actionName;
            return this;
        }

        public final String getRevisionType() {
            return revisionType;
        }

        public final void setRevisionType(String revisionType) {
            this.revisionType = revisionType;
        }

        @Override
        public final Builder revisionType(String revisionType) {
            this.revisionType = revisionType;
            return this;
        }

        @Override
        public final Builder revisionType(SourceRevisionType revisionType) {
            this.revisionType(revisionType == null ? null : revisionType.toString());
            return this;
        }

        public final String getRevisionValue() {
            return revisionValue;
        }

        public final void setRevisionValue(String revisionValue) {
            this.revisionValue = revisionValue;
        }

        @Override
        public final Builder revisionValue(String revisionValue) {
            this.revisionValue = revisionValue;
            return this;
        }

        @Override
        public SourceRevisionOverride build() {
            return new SourceRevisionOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
