/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the structure of actions and stages to be performed in the pipeline.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PipelineDeclaration implements SdkPojo, Serializable,
        ToCopyableBuilder<PipelineDeclaration.Builder, PipelineDeclaration> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(PipelineDeclaration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(PipelineDeclaration::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<ArtifactStore> ARTIFACT_STORE_FIELD = SdkField
            .<ArtifactStore> builder(MarshallingType.SDK_POJO).memberName("artifactStore")
            .getter(getter(PipelineDeclaration::artifactStore)).setter(setter(Builder::artifactStore))
            .constructor(ArtifactStore::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("artifactStore").build()).build();

    private static final SdkField<Map<String, ArtifactStore>> ARTIFACT_STORES_FIELD = SdkField
            .<Map<String, ArtifactStore>> builder(MarshallingType.MAP)
            .memberName("artifactStores")
            .getter(getter(PipelineDeclaration::artifactStores))
            .setter(setter(Builder::artifactStores))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("artifactStores").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ArtifactStore> builder(MarshallingType.SDK_POJO)
                                            .constructor(ArtifactStore::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<StageDeclaration>> STAGES_FIELD = SdkField
            .<List<StageDeclaration>> builder(MarshallingType.LIST)
            .memberName("stages")
            .getter(getter(PipelineDeclaration::stages))
            .setter(setter(Builder::stages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<StageDeclaration> builder(MarshallingType.SDK_POJO)
                                            .constructor(StageDeclaration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> VERSION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("version").getter(getter(PipelineDeclaration::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> EXECUTION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionMode").getter(getter(PipelineDeclaration::executionModeAsString))
            .setter(setter(Builder::executionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionMode").build()).build();

    private static final SdkField<String> PIPELINE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pipelineType").getter(getter(PipelineDeclaration::pipelineTypeAsString))
            .setter(setter(Builder::pipelineType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineType").build()).build();

    private static final SdkField<List<PipelineVariableDeclaration>> VARIABLES_FIELD = SdkField
            .<List<PipelineVariableDeclaration>> builder(MarshallingType.LIST)
            .memberName("variables")
            .getter(getter(PipelineDeclaration::variables))
            .setter(setter(Builder::variables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("variables").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PipelineVariableDeclaration> builder(MarshallingType.SDK_POJO)
                                            .constructor(PipelineVariableDeclaration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PipelineTriggerDeclaration>> TRIGGERS_FIELD = SdkField
            .<List<PipelineTriggerDeclaration>> builder(MarshallingType.LIST)
            .memberName("triggers")
            .getter(getter(PipelineDeclaration::triggers))
            .setter(setter(Builder::triggers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("triggers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PipelineTriggerDeclaration> builder(MarshallingType.SDK_POJO)
                                            .constructor(PipelineTriggerDeclaration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ROLE_ARN_FIELD,
            ARTIFACT_STORE_FIELD, ARTIFACT_STORES_FIELD, STAGES_FIELD, VERSION_FIELD, EXECUTION_MODE_FIELD, PIPELINE_TYPE_FIELD,
            VARIABLES_FIELD, TRIGGERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String roleArn;

    private final ArtifactStore artifactStore;

    private final Map<String, ArtifactStore> artifactStores;

    private final List<StageDeclaration> stages;

    private final Integer version;

    private final String executionMode;

    private final String pipelineType;

    private final List<PipelineVariableDeclaration> variables;

    private final List<PipelineTriggerDeclaration> triggers;

    private PipelineDeclaration(BuilderImpl builder) {
        this.name = builder.name;
        this.roleArn = builder.roleArn;
        this.artifactStore = builder.artifactStore;
        this.artifactStores = builder.artifactStores;
        this.stages = builder.stages;
        this.version = builder.version;
        this.executionMode = builder.executionMode;
        this.pipelineType = builder.pipelineType;
        this.variables = builder.variables;
        this.triggers = builder.triggers;
    }

    /**
     * <p>
     * The name of the pipeline.
     * </p>
     * 
     * @return The name of the pipeline.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for CodePipeline to use to either perform actions with no
     * <code>actionRoleArn</code>, or to use to assume roles for actions with an <code>actionRoleArn</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for CodePipeline to use to either perform actions with no
     *         <code>actionRoleArn</code>, or to use to assume roles for actions with an <code>actionRoleArn</code>.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * Represents information about the S3 bucket where artifacts are stored for the pipeline.
     * </p>
     * <note>
     * <p>
     * You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but you
     * cannot use both. If you create a cross-region action in your pipeline, you must use <code>artifactStores</code>.
     * </p>
     * </note>
     * 
     * @return Represents information about the S3 bucket where artifacts are stored for the pipeline.</p> <note>
     *         <p>
     *         You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but
     *         you cannot use both. If you create a cross-region action in your pipeline, you must use
     *         <code>artifactStores</code>.
     *         </p>
     */
    public final ArtifactStore artifactStore() {
        return artifactStore;
    }

    /**
     * For responses, this returns true if the service returned a value for the ArtifactStores property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasArtifactStores() {
        return artifactStores != null && !(artifactStores instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A mapping of <code>artifactStore</code> objects and their corresponding Amazon Web Services Regions. There must
     * be an artifact store for the pipeline Region and for each cross-region action in the pipeline.
     * </p>
     * <note>
     * <p>
     * You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but you
     * cannot use both. If you create a cross-region action in your pipeline, you must use <code>artifactStores</code>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArtifactStores} method.
     * </p>
     * 
     * @return A mapping of <code>artifactStore</code> objects and their corresponding Amazon Web Services Regions.
     *         There must be an artifact store for the pipeline Region and for each cross-region action in the
     *         pipeline.</p> <note>
     *         <p>
     *         You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but
     *         you cannot use both. If you create a cross-region action in your pipeline, you must use
     *         <code>artifactStores</code>.
     *         </p>
     */
    public final Map<String, ArtifactStore> artifactStores() {
        return artifactStores;
    }

    /**
     * For responses, this returns true if the service returned a value for the Stages property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasStages() {
        return stages != null && !(stages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The stage in which to perform the action.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStages} method.
     * </p>
     * 
     * @return The stage in which to perform the action.
     */
    public final List<StageDeclaration> stages() {
        return stages;
    }

    /**
     * <p>
     * The version number of the pipeline. A new pipeline always has a version number of 1. This number is incremented
     * when a pipeline is updated.
     * </p>
     * 
     * @return The version number of the pipeline. A new pipeline always has a version number of 1. This number is
     *         incremented when a pipeline is updated.
     */
    public final Integer version() {
        return version;
    }

    /**
     * <p>
     * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionMode}
     * will return {@link ExecutionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #executionModeAsString}.
     * </p>
     * 
     * @return The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * @see ExecutionMode
     */
    public final ExecutionMode executionMode() {
        return ExecutionMode.fromValue(executionMode);
    }

    /**
     * <p>
     * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionMode}
     * will return {@link ExecutionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #executionModeAsString}.
     * </p>
     * 
     * @return The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * @see ExecutionMode
     */
    public final String executionModeAsString() {
        return executionMode;
    }

    /**
     * <p>
     * CodePipeline provides the following pipeline types, which differ in characteristics and price, so that you can
     * tailor your pipeline features and cost to the needs of your applications.
     * </p>
     * <ul>
     * <li>
     * <p>
     * V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * V2 type pipelines have the same structure as a V1 type, along with additional parameters for release safety and
     * trigger configuration.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating a pipeline
     * will result in the pipeline having the V2 type of pipeline and the associated costs.
     * </p>
     * </important>
     * <p>
     * For information about pricing for CodePipeline, see <a
     * href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
     * </p>
     * <p>
     * For information about which type of pipeline to choose, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What type of
     * pipeline is right for me?</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pipelineType} will
     * return {@link PipelineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pipelineTypeAsString}.
     * </p>
     * 
     * @return CodePipeline provides the following pipeline types, which differ in characteristics and price, so that
     *         you can tailor your pipeline features and cost to the needs of your applications.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level
     *         parameters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         V2 type pipelines have the same structure as a V1 type, along with additional parameters for release
     *         safety and trigger configuration.
     *         </p>
     *         </li>
     *         </ul>
     *         <important>
     *         <p>
     *         Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating a
     *         pipeline will result in the pipeline having the V2 type of pipeline and the associated costs.
     *         </p>
     *         </important>
     *         <p>
     *         For information about pricing for CodePipeline, see <a
     *         href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
     *         </p>
     *         <p>
     *         For information about which type of pipeline to choose, see <a
     *         href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What type
     *         of pipeline is right for me?</a>.
     * @see PipelineType
     */
    public final PipelineType pipelineType() {
        return PipelineType.fromValue(pipelineType);
    }

    /**
     * <p>
     * CodePipeline provides the following pipeline types, which differ in characteristics and price, so that you can
     * tailor your pipeline features and cost to the needs of your applications.
     * </p>
     * <ul>
     * <li>
     * <p>
     * V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * V2 type pipelines have the same structure as a V1 type, along with additional parameters for release safety and
     * trigger configuration.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating a pipeline
     * will result in the pipeline having the V2 type of pipeline and the associated costs.
     * </p>
     * </important>
     * <p>
     * For information about pricing for CodePipeline, see <a
     * href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
     * </p>
     * <p>
     * For information about which type of pipeline to choose, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What type of
     * pipeline is right for me?</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pipelineType} will
     * return {@link PipelineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pipelineTypeAsString}.
     * </p>
     * 
     * @return CodePipeline provides the following pipeline types, which differ in characteristics and price, so that
     *         you can tailor your pipeline features and cost to the needs of your applications.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level
     *         parameters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         V2 type pipelines have the same structure as a V1 type, along with additional parameters for release
     *         safety and trigger configuration.
     *         </p>
     *         </li>
     *         </ul>
     *         <important>
     *         <p>
     *         Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating a
     *         pipeline will result in the pipeline having the V2 type of pipeline and the associated costs.
     *         </p>
     *         </important>
     *         <p>
     *         For information about pricing for CodePipeline, see <a
     *         href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
     *         </p>
     *         <p>
     *         For information about which type of pipeline to choose, see <a
     *         href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What type
     *         of pipeline is right for me?</a>.
     * @see PipelineType
     */
    public final String pipelineTypeAsString() {
        return pipelineType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Variables property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVariables() {
        return variables != null && !(variables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list that defines the pipeline variables for a pipeline resource. Variable names can have alphanumeric and
     * underscore characters, and the values must match <code>[A-Za-z0-9@\-_]+</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVariables} method.
     * </p>
     * 
     * @return A list that defines the pipeline variables for a pipeline resource. Variable names can have alphanumeric
     *         and underscore characters, and the values must match <code>[A-Za-z0-9@\-_]+</code>.
     */
    public final List<PipelineVariableDeclaration> variables() {
        return variables;
    }

    /**
     * For responses, this returns true if the service returned a value for the Triggers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTriggers() {
        return triggers != null && !(triggers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The trigger configuration specifying a type of event, such as Git tags, that starts the pipeline.
     * </p>
     * <note>
     * <p>
     * When a trigger configuration is specified, default change detection for repository and branch commits is
     * disabled.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTriggers} method.
     * </p>
     * 
     * @return The trigger configuration specifying a type of event, such as Git tags, that starts the pipeline.</p>
     *         <note>
     *         <p>
     *         When a trigger configuration is specified, default change detection for repository and branch commits is
     *         disabled.
     *         </p>
     */
    public final List<PipelineTriggerDeclaration> triggers() {
        return triggers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(artifactStore());
        hashCode = 31 * hashCode + Objects.hashCode(hasArtifactStores() ? artifactStores() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStages() ? stages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(executionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(pipelineTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasVariables() ? variables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTriggers() ? triggers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PipelineDeclaration)) {
            return false;
        }
        PipelineDeclaration other = (PipelineDeclaration) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(artifactStore(), other.artifactStore()) && hasArtifactStores() == other.hasArtifactStores()
                && Objects.equals(artifactStores(), other.artifactStores()) && hasStages() == other.hasStages()
                && Objects.equals(stages(), other.stages()) && Objects.equals(version(), other.version())
                && Objects.equals(executionModeAsString(), other.executionModeAsString())
                && Objects.equals(pipelineTypeAsString(), other.pipelineTypeAsString()) && hasVariables() == other.hasVariables()
                && Objects.equals(variables(), other.variables()) && hasTriggers() == other.hasTriggers()
                && Objects.equals(triggers(), other.triggers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PipelineDeclaration").add("Name", name()).add("RoleArn", roleArn())
                .add("ArtifactStore", artifactStore()).add("ArtifactStores", hasArtifactStores() ? artifactStores() : null)
                .add("Stages", hasStages() ? stages() : null).add("Version", version())
                .add("ExecutionMode", executionModeAsString()).add("PipelineType", pipelineTypeAsString())
                .add("Variables", hasVariables() ? variables() : null).add("Triggers", hasTriggers() ? triggers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "artifactStore":
            return Optional.ofNullable(clazz.cast(artifactStore()));
        case "artifactStores":
            return Optional.ofNullable(clazz.cast(artifactStores()));
        case "stages":
            return Optional.ofNullable(clazz.cast(stages()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "executionMode":
            return Optional.ofNullable(clazz.cast(executionModeAsString()));
        case "pipelineType":
            return Optional.ofNullable(clazz.cast(pipelineTypeAsString()));
        case "variables":
            return Optional.ofNullable(clazz.cast(variables()));
        case "triggers":
            return Optional.ofNullable(clazz.cast(triggers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PipelineDeclaration, T> g) {
        return obj -> g.apply((PipelineDeclaration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PipelineDeclaration> {
        /**
         * <p>
         * The name of the pipeline.
         * </p>
         * 
         * @param name
         *        The name of the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for CodePipeline to use to either perform actions with no
         * <code>actionRoleArn</code>, or to use to assume roles for actions with an <code>actionRoleArn</code>.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) for CodePipeline to use to either perform actions with no
         *        <code>actionRoleArn</code>, or to use to assume roles for actions with an <code>actionRoleArn</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * Represents information about the S3 bucket where artifacts are stored for the pipeline.
         * </p>
         * <note>
         * <p>
         * You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but you
         * cannot use both. If you create a cross-region action in your pipeline, you must use
         * <code>artifactStores</code>.
         * </p>
         * </note>
         * 
         * @param artifactStore
         *        Represents information about the S3 bucket where artifacts are stored for the pipeline.</p> <note>
         *        <p>
         *        You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline,
         *        but you cannot use both. If you create a cross-region action in your pipeline, you must use
         *        <code>artifactStores</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifactStore(ArtifactStore artifactStore);

        /**
         * <p>
         * Represents information about the S3 bucket where artifacts are stored for the pipeline.
         * </p>
         * <note>
         * <p>
         * You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but you
         * cannot use both. If you create a cross-region action in your pipeline, you must use
         * <code>artifactStores</code>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ArtifactStore.Builder} avoiding
         * the need to create one manually via {@link ArtifactStore#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ArtifactStore.Builder#build()} is called immediately and its
         * result is passed to {@link #artifactStore(ArtifactStore)}.
         * 
         * @param artifactStore
         *        a consumer that will call methods on {@link ArtifactStore.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #artifactStore(ArtifactStore)
         */
        default Builder artifactStore(Consumer<ArtifactStore.Builder> artifactStore) {
            return artifactStore(ArtifactStore.builder().applyMutation(artifactStore).build());
        }

        /**
         * <p>
         * A mapping of <code>artifactStore</code> objects and their corresponding Amazon Web Services Regions. There
         * must be an artifact store for the pipeline Region and for each cross-region action in the pipeline.
         * </p>
         * <note>
         * <p>
         * You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but you
         * cannot use both. If you create a cross-region action in your pipeline, you must use
         * <code>artifactStores</code>.
         * </p>
         * </note>
         * 
         * @param artifactStores
         *        A mapping of <code>artifactStore</code> objects and their corresponding Amazon Web Services Regions.
         *        There must be an artifact store for the pipeline Region and for each cross-region action in the
         *        pipeline.</p> <note>
         *        <p>
         *        You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline,
         *        but you cannot use both. If you create a cross-region action in your pipeline, you must use
         *        <code>artifactStores</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifactStores(Map<String, ArtifactStore> artifactStores);

        /**
         * <p>
         * The stage in which to perform the action.
         * </p>
         * 
         * @param stages
         *        The stage in which to perform the action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stages(Collection<StageDeclaration> stages);

        /**
         * <p>
         * The stage in which to perform the action.
         * </p>
         * 
         * @param stages
         *        The stage in which to perform the action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stages(StageDeclaration... stages);

        /**
         * <p>
         * The stage in which to perform the action.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.StageDeclaration.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.codepipeline.model.StageDeclaration#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.StageDeclaration.Builder#build()} is called
         * immediately and its result is passed to {@link #stages(List<StageDeclaration>)}.
         * 
         * @param stages
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.StageDeclaration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stages(java.util.Collection<StageDeclaration>)
         */
        Builder stages(Consumer<StageDeclaration.Builder>... stages);

        /**
         * <p>
         * The version number of the pipeline. A new pipeline always has a version number of 1. This number is
         * incremented when a pipeline is updated.
         * </p>
         * 
         * @param version
         *        The version number of the pipeline. A new pipeline always has a version number of 1. This number is
         *        incremented when a pipeline is updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Integer version);

        /**
         * <p>
         * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * </p>
         * 
         * @param executionMode
         *        The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * @see ExecutionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionMode
         */
        Builder executionMode(String executionMode);

        /**
         * <p>
         * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * </p>
         * 
         * @param executionMode
         *        The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * @see ExecutionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionMode
         */
        Builder executionMode(ExecutionMode executionMode);

        /**
         * <p>
         * CodePipeline provides the following pipeline types, which differ in characteristics and price, so that you
         * can tailor your pipeline features and cost to the needs of your applications.
         * </p>
         * <ul>
         * <li>
         * <p>
         * V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level parameters.
         * </p>
         * </li>
         * <li>
         * <p>
         * V2 type pipelines have the same structure as a V1 type, along with additional parameters for release safety
         * and trigger configuration.
         * </p>
         * </li>
         * </ul>
         * <important>
         * <p>
         * Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating a
         * pipeline will result in the pipeline having the V2 type of pipeline and the associated costs.
         * </p>
         * </important>
         * <p>
         * For information about pricing for CodePipeline, see <a
         * href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
         * </p>
         * <p>
         * For information about which type of pipeline to choose, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What type of
         * pipeline is right for me?</a>.
         * </p>
         * 
         * @param pipelineType
         *        CodePipeline provides the following pipeline types, which differ in characteristics and price, so that
         *        you can tailor your pipeline features and cost to the needs of your applications.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level
         *        parameters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        V2 type pipelines have the same structure as a V1 type, along with additional parameters for release
         *        safety and trigger configuration.
         *        </p>
         *        </li>
         *        </ul>
         *        <important>
         *        <p>
         *        Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating
         *        a pipeline will result in the pipeline having the V2 type of pipeline and the associated costs.
         *        </p>
         *        </important>
         *        <p>
         *        For information about pricing for CodePipeline, see <a
         *        href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
         *        </p>
         *        <p>
         *        For information about which type of pipeline to choose, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What
         *        type of pipeline is right for me?</a>.
         * @see PipelineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineType
         */
        Builder pipelineType(String pipelineType);

        /**
         * <p>
         * CodePipeline provides the following pipeline types, which differ in characteristics and price, so that you
         * can tailor your pipeline features and cost to the needs of your applications.
         * </p>
         * <ul>
         * <li>
         * <p>
         * V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level parameters.
         * </p>
         * </li>
         * <li>
         * <p>
         * V2 type pipelines have the same structure as a V1 type, along with additional parameters for release safety
         * and trigger configuration.
         * </p>
         * </li>
         * </ul>
         * <important>
         * <p>
         * Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating a
         * pipeline will result in the pipeline having the V2 type of pipeline and the associated costs.
         * </p>
         * </important>
         * <p>
         * For information about pricing for CodePipeline, see <a
         * href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
         * </p>
         * <p>
         * For information about which type of pipeline to choose, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What type of
         * pipeline is right for me?</a>.
         * </p>
         * 
         * @param pipelineType
         *        CodePipeline provides the following pipeline types, which differ in characteristics and price, so that
         *        you can tailor your pipeline features and cost to the needs of your applications.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        V1 type pipelines have a JSON structure that contains standard pipeline, stage, and action-level
         *        parameters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        V2 type pipelines have the same structure as a V1 type, along with additional parameters for release
         *        safety and trigger configuration.
         *        </p>
         *        </li>
         *        </ul>
         *        <important>
         *        <p>
         *        Including V2 parameters, such as triggers on Git tags, in the pipeline JSON when creating or updating
         *        a pipeline will result in the pipeline having the V2 type of pipeline and the associated costs.
         *        </p>
         *        </important>
         *        <p>
         *        For information about pricing for CodePipeline, see <a
         *        href="http://aws.amazon.com/codepipeline/pricing/">Pricing</a>.
         *        </p>
         *        <p>
         *        For information about which type of pipeline to choose, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipeline-types-planning.html">What
         *        type of pipeline is right for me?</a>.
         * @see PipelineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineType
         */
        Builder pipelineType(PipelineType pipelineType);

        /**
         * <p>
         * A list that defines the pipeline variables for a pipeline resource. Variable names can have alphanumeric and
         * underscore characters, and the values must match <code>[A-Za-z0-9@\-_]+</code>.
         * </p>
         * 
         * @param variables
         *        A list that defines the pipeline variables for a pipeline resource. Variable names can have
         *        alphanumeric and underscore characters, and the values must match <code>[A-Za-z0-9@\-_]+</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(Collection<PipelineVariableDeclaration> variables);

        /**
         * <p>
         * A list that defines the pipeline variables for a pipeline resource. Variable names can have alphanumeric and
         * underscore characters, and the values must match <code>[A-Za-z0-9@\-_]+</code>.
         * </p>
         * 
         * @param variables
         *        A list that defines the pipeline variables for a pipeline resource. Variable names can have
         *        alphanumeric and underscore characters, and the values must match <code>[A-Za-z0-9@\-_]+</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(PipelineVariableDeclaration... variables);

        /**
         * <p>
         * A list that defines the pipeline variables for a pipeline resource. Variable names can have alphanumeric and
         * underscore characters, and the values must match <code>[A-Za-z0-9@\-_]+</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.PipelineVariableDeclaration.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.codepipeline.model.PipelineVariableDeclaration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.PipelineVariableDeclaration.Builder#build()} is
         * called immediately and its result is passed to {@link #variables(List<PipelineVariableDeclaration>)}.
         * 
         * @param variables
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.PipelineVariableDeclaration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variables(java.util.Collection<PipelineVariableDeclaration>)
         */
        Builder variables(Consumer<PipelineVariableDeclaration.Builder>... variables);

        /**
         * <p>
         * The trigger configuration specifying a type of event, such as Git tags, that starts the pipeline.
         * </p>
         * <note>
         * <p>
         * When a trigger configuration is specified, default change detection for repository and branch commits is
         * disabled.
         * </p>
         * </note>
         * 
         * @param triggers
         *        The trigger configuration specifying a type of event, such as Git tags, that starts the pipeline.</p>
         *        <note>
         *        <p>
         *        When a trigger configuration is specified, default change detection for repository and branch commits
         *        is disabled.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggers(Collection<PipelineTriggerDeclaration> triggers);

        /**
         * <p>
         * The trigger configuration specifying a type of event, such as Git tags, that starts the pipeline.
         * </p>
         * <note>
         * <p>
         * When a trigger configuration is specified, default change detection for repository and branch commits is
         * disabled.
         * </p>
         * </note>
         * 
         * @param triggers
         *        The trigger configuration specifying a type of event, such as Git tags, that starts the pipeline.</p>
         *        <note>
         *        <p>
         *        When a trigger configuration is specified, default change detection for repository and branch commits
         *        is disabled.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggers(PipelineTriggerDeclaration... triggers);

        /**
         * <p>
         * The trigger configuration specifying a type of event, such as Git tags, that starts the pipeline.
         * </p>
         * <note>
         * <p>
         * When a trigger configuration is specified, default change detection for repository and branch commits is
         * disabled.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.PipelineTriggerDeclaration.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.codepipeline.model.PipelineTriggerDeclaration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.PipelineTriggerDeclaration.Builder#build()} is
         * called immediately and its result is passed to {@link #triggers(List<PipelineTriggerDeclaration>)}.
         * 
         * @param triggers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.PipelineTriggerDeclaration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #triggers(java.util.Collection<PipelineTriggerDeclaration>)
         */
        Builder triggers(Consumer<PipelineTriggerDeclaration.Builder>... triggers);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String roleArn;

        private ArtifactStore artifactStore;

        private Map<String, ArtifactStore> artifactStores = DefaultSdkAutoConstructMap.getInstance();

        private List<StageDeclaration> stages = DefaultSdkAutoConstructList.getInstance();

        private Integer version;

        private String executionMode;

        private String pipelineType;

        private List<PipelineVariableDeclaration> variables = DefaultSdkAutoConstructList.getInstance();

        private List<PipelineTriggerDeclaration> triggers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PipelineDeclaration model) {
            name(model.name);
            roleArn(model.roleArn);
            artifactStore(model.artifactStore);
            artifactStores(model.artifactStores);
            stages(model.stages);
            version(model.version);
            executionMode(model.executionMode);
            pipelineType(model.pipelineType);
            variables(model.variables);
            triggers(model.triggers);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final ArtifactStore.Builder getArtifactStore() {
            return artifactStore != null ? artifactStore.toBuilder() : null;
        }

        public final void setArtifactStore(ArtifactStore.BuilderImpl artifactStore) {
            this.artifactStore = artifactStore != null ? artifactStore.build() : null;
        }

        @Override
        public final Builder artifactStore(ArtifactStore artifactStore) {
            this.artifactStore = artifactStore;
            return this;
        }

        public final Map<String, ArtifactStore.Builder> getArtifactStores() {
            Map<String, ArtifactStore.Builder> result = ArtifactStoreMapCopier.copyToBuilder(this.artifactStores);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setArtifactStores(Map<String, ArtifactStore.BuilderImpl> artifactStores) {
            this.artifactStores = ArtifactStoreMapCopier.copyFromBuilder(artifactStores);
        }

        @Override
        public final Builder artifactStores(Map<String, ArtifactStore> artifactStores) {
            this.artifactStores = ArtifactStoreMapCopier.copy(artifactStores);
            return this;
        }

        public final List<StageDeclaration.Builder> getStages() {
            List<StageDeclaration.Builder> result = PipelineStageDeclarationListCopier.copyToBuilder(this.stages);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStages(Collection<StageDeclaration.BuilderImpl> stages) {
            this.stages = PipelineStageDeclarationListCopier.copyFromBuilder(stages);
        }

        @Override
        public final Builder stages(Collection<StageDeclaration> stages) {
            this.stages = PipelineStageDeclarationListCopier.copy(stages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stages(StageDeclaration... stages) {
            stages(Arrays.asList(stages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stages(Consumer<StageDeclaration.Builder>... stages) {
            stages(Stream.of(stages).map(c -> StageDeclaration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getVersion() {
            return version;
        }

        public final void setVersion(Integer version) {
            this.version = version;
        }

        @Override
        public final Builder version(Integer version) {
            this.version = version;
            return this;
        }

        public final String getExecutionMode() {
            return executionMode;
        }

        public final void setExecutionMode(String executionMode) {
            this.executionMode = executionMode;
        }

        @Override
        public final Builder executionMode(String executionMode) {
            this.executionMode = executionMode;
            return this;
        }

        @Override
        public final Builder executionMode(ExecutionMode executionMode) {
            this.executionMode(executionMode == null ? null : executionMode.toString());
            return this;
        }

        public final String getPipelineType() {
            return pipelineType;
        }

        public final void setPipelineType(String pipelineType) {
            this.pipelineType = pipelineType;
        }

        @Override
        public final Builder pipelineType(String pipelineType) {
            this.pipelineType = pipelineType;
            return this;
        }

        @Override
        public final Builder pipelineType(PipelineType pipelineType) {
            this.pipelineType(pipelineType == null ? null : pipelineType.toString());
            return this;
        }

        public final List<PipelineVariableDeclaration.Builder> getVariables() {
            List<PipelineVariableDeclaration.Builder> result = PipelineVariableDeclarationListCopier
                    .copyToBuilder(this.variables);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVariables(Collection<PipelineVariableDeclaration.BuilderImpl> variables) {
            this.variables = PipelineVariableDeclarationListCopier.copyFromBuilder(variables);
        }

        @Override
        public final Builder variables(Collection<PipelineVariableDeclaration> variables) {
            this.variables = PipelineVariableDeclarationListCopier.copy(variables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(PipelineVariableDeclaration... variables) {
            variables(Arrays.asList(variables));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(Consumer<PipelineVariableDeclaration.Builder>... variables) {
            variables(Stream.of(variables).map(c -> PipelineVariableDeclaration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<PipelineTriggerDeclaration.Builder> getTriggers() {
            List<PipelineTriggerDeclaration.Builder> result = PipelineTriggerDeclarationListCopier.copyToBuilder(this.triggers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTriggers(Collection<PipelineTriggerDeclaration.BuilderImpl> triggers) {
            this.triggers = PipelineTriggerDeclarationListCopier.copyFromBuilder(triggers);
        }

        @Override
        public final Builder triggers(Collection<PipelineTriggerDeclaration> triggers) {
            this.triggers = PipelineTriggerDeclarationListCopier.copy(triggers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder triggers(PipelineTriggerDeclaration... triggers) {
            triggers(Arrays.asList(triggers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder triggers(Consumer<PipelineTriggerDeclaration.Builder>... triggers) {
            triggers(Stream.of(triggers).map(c -> PipelineTriggerDeclaration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public PipelineDeclaration build() {
            return new PipelineDeclaration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
