/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.autoscaling.AutoScalingAsyncClient;
import software.amazon.awssdk.services.autoscaling.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.autoscaling.model.AutoScalingRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsRequest;
import software.amazon.awssdk.services.autoscaling.model.DescribeAutoScalingGroupsResponse;
import software.amazon.awssdk.services.autoscaling.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAutoScalingAsyncWaiter implements AutoScalingAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final AutoScalingAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeAutoScalingGroupsResponse> groupExistsWaiter;

    private final AsyncWaiter<DescribeAutoScalingGroupsResponse> groupInServiceWaiter;

    private final AsyncWaiter<DescribeAutoScalingGroupsResponse> groupNotExistsWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultAutoScalingAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AutoScalingAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.groupExistsWaiter = AsyncWaiter.builder(DescribeAutoScalingGroupsResponse.class)
                .acceptors(groupExistsWaiterAcceptors())
                .overrideConfiguration(groupExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.groupInServiceWaiter = AsyncWaiter.builder(DescribeAutoScalingGroupsResponse.class)
                .acceptors(groupInServiceWaiterAcceptors())
                .overrideConfiguration(groupInServiceWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.groupNotExistsWaiter = AsyncWaiter.builder(DescribeAutoScalingGroupsResponse.class)
                .acceptors(groupNotExistsWaiterAcceptors())
                .overrideConfiguration(groupNotExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAutoScalingGroupsResponse>> waitUntilGroupExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) {
        return groupExistsWaiter.runAsync(() -> client
                .describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAutoScalingGroupsResponse>> waitUntilGroupExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return groupExistsWaiter.runAsync(
                () -> client.describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)),
                groupExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAutoScalingGroupsResponse>> waitUntilGroupInService(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) {
        return groupInServiceWaiter.runAsync(() -> client
                .describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAutoScalingGroupsResponse>> waitUntilGroupInService(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return groupInServiceWaiter.runAsync(
                () -> client.describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)),
                groupInServiceWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAutoScalingGroupsResponse>> waitUntilGroupNotExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest) {
        return groupNotExistsWaiter.runAsync(() -> client
                .describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAutoScalingGroupsResponse>> waitUntilGroupNotExists(
            DescribeAutoScalingGroupsRequest describeAutoScalingGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return groupNotExistsWaiter.runAsync(
                () -> client.describeAutoScalingGroups(applyWaitersUserAgent(describeAutoScalingGroupsRequest)),
                groupNotExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> groupExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), true);
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), false);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> groupInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(
                    input.field("AutoScalingGroups")
                            .flatten()
                            .multiSelectList(
                                    x0 -> x0.field("Instances")
                                            .filter(x1 -> x1.field("LifecycleState").compare("==", x1.constant("InService")))
                                            .length().compare(">=", x0.field("MinSize"))).flatten()
                            .contains(input.constant(false)).value(), false);
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(
                    input.field("AutoScalingGroups")
                            .flatten()
                            .multiSelectList(
                                    x0 -> x0.field("Instances")
                                            .filter(x1 -> x1.field("LifecycleState").compare("==", x1.constant("InService")))
                                            .length().compare(">=", x0.field("MinSize"))).flatten()
                            .contains(input.constant(false)).value(), true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> groupNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAutoScalingGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), false);
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("AutoScalingGroups").length().compare(">", input.constant(0)).value(), true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration groupExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(10);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration groupInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration groupNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AutoScalingAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AutoScalingRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AutoScalingAsyncWaiter.Builder {
        private AutoScalingAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public AutoScalingAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public AutoScalingAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AutoScalingAsyncWaiter.Builder client(AutoScalingAsyncClient client) {
            this.client = client;
            return this;
        }

        public AutoScalingAsyncWaiter build() {
            return new DefaultAutoScalingAsyncWaiter(this);
        }
    }
}
