/*
 * Copyright 2002-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.http.client.reactive;

import org.eclipse.jetty.reactive.client.ReactiveResponse;
import reactor.core.publisher.Flux;

import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.ResponseCookie;
import org.springframework.http.support.JettyHeadersAdapter;
import org.springframework.util.MultiValueMap;

/**
 * {@link ClientHttpResponse} implementation for the Jetty ReactiveStreams HTTP client.
 *
 * @author Sebastien Deleuze
 * @since 5.1
 * @see <a href="https://github.com/jetty-project/jetty-reactive-httpclient">
 *     Jetty ReactiveStreams HttpClient</a>
 */
class JettyClientHttpResponse extends AbstractClientHttpResponse {

	public JettyClientHttpResponse(
			ReactiveResponse response, Flux<DataBuffer> content,
			MultiValueMap<String, ResponseCookie> cookies) {

		super(HttpStatusCode.valueOf(response.getStatus()), adaptHeaders(response), cookies, content);
	}

	private static HttpHeaders adaptHeaders(ReactiveResponse response) {
		MultiValueMap<String, String> headers = new JettyHeadersAdapter(response.getHeaders());
		return HttpHeaders.readOnlyHttpHeaders(headers);
	}

}
