/*
 * Copyright 2008-present MongoDB, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mongodb.kbson

import kotlinx.serialization.Serializable
import org.mongodb.kbson.serialization.BsonDoubleSerializer

/**
 * A representation of the BSON Double type.
 *
 * @constructor constructs a new instance with the given value
 * @property value the [Double] value
 */
@Serializable(with = BsonDoubleSerializer::class)
public class BsonDouble(public val value: Double) : BsonNumber(value), Comparable<BsonDouble> {

    override val bsonType: BsonType
        get() = BsonType.DOUBLE

    override fun compareTo(other: BsonDouble): Int {
        return value.compareTo(other.value)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || this::class != other::class) return false

        other as BsonDouble

        if (value != other.value) return false

        return true
    }

    override fun hashCode(): Int {
        return value.hashCode()
    }

    override fun toString(): String {
        return "BsonDouble(value=$value)"
    }
}
