/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.oak.cache.CacheStats;
import org.apache.jackrabbit.oak.commons.conditions.Validate;
import org.apache.jackrabbit.oak.commons.sort.StringSort;
import org.apache.jackrabbit.oak.commons.time.Stopwatch;
import org.apache.jackrabbit.oak.plugins.document.AbstractDocumentNodeState;
import org.apache.jackrabbit.oak.plugins.document.Branch;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.DiffCache;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.plugins.document.JournalEntry;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.Path;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionVector;
import org.apache.jackrabbit.oak.plugins.document.util.StringValue;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class JournalDiffLoader
implements DiffCache.Loader {
    private static final Logger LOG = LoggerFactory.getLogger(JournalDiffLoader.class);
    private final AbstractDocumentNodeState base;
    private final AbstractDocumentNodeState node;
    private final DocumentNodeStore ns;
    private Stats stats;

    JournalDiffLoader(@NotNull AbstractDocumentNodeState base, @NotNull AbstractDocumentNodeState node, @NotNull DocumentNodeStore ns) {
        this.base = Objects.requireNonNull(base);
        this.node = Objects.requireNonNull(node);
        this.ns = Objects.requireNonNull(ns);
        Validate.checkArgument((boolean)base.getPath().equals(node.getPath()), (String)"nodes must have matching paths: %s != %s", (Object[])new Object[]{base.getPath(), node.getPath()});
    }

    @Override
    public String call() {
        RevisionVector afterRev = this.node.getRootRevision();
        RevisionVector beforeRev = this.base.getRootRevision();
        this.stats = new Stats(this.node.getPath(), beforeRev, afterRev);
        StringSort changes = JournalEntry.newSorter();
        try {
            Path path = this.node.getPath();
            this.readTrunkChanges(path, beforeRev, afterRev, changes);
            this.readBranchChanges(path, beforeRev, changes);
            this.readBranchChanges(path, afterRev, changes);
            changes.sort();
            DiffCache df = this.ns.getDiffCache();
            WrappedDiffCache wrappedCache = new WrappedDiffCache(this.node.getPath(), df, this.stats);
            JournalEntry.applyTo((Iterable<String>)changes, wrappedCache, this.node.getPath(), beforeRev, afterRev);
            String string = wrappedCache.changes;
            return string;
        }
        catch (IOException e) {
            throw DocumentStoreException.convert(e);
        }
        finally {
            Utils.closeIfCloseable(changes);
            this.logStats();
        }
    }

    private void readBranchChanges(Path path, RevisionVector rv, StringSort changes) throws IOException {
        if (!rv.isBranch() || this.ns.isDisableBranches()) {
            return;
        }
        Branch b = this.ns.getBranches().getBranch(rv);
        if (b == null) {
            if (!this.ns.getBranches().isBranchBase(rv)) {
                JournalDiffLoader.missingBranch(rv);
            }
            return;
        }
        DocumentStore store = this.ns.getDocumentStore();
        for (Revision br : b.getCommits()) {
            Branch.BranchCommit bc = b.getCommit(br);
            if (bc.isRebase()) continue;
            JournalEntry entry = store.find(Collection.JOURNAL, JournalEntry.asId(br));
            if (entry != null) {
                entry.addTo(changes, path);
                ++this.stats.numJournalEntries;
                continue;
            }
            LOG.warn("Missing journal entry for {}", (Object)JournalEntry.asId(br));
        }
    }

    private void readTrunkChanges(Path path, RevisionVector beforeRev, RevisionVector afterRev, StringSort changes) throws IOException {
        JournalEntry localPending = this.ns.getCurrentJournalEntry();
        DocumentStore store = this.ns.getDocumentStore();
        NodeDocument root = Utils.getRootDocument(store);
        int clusterId = this.ns.getClusterId();
        Map<Integer, Revision> lastRevs = root.getLastRev();
        Revision localLastRev = clusterId == 0 ? afterRev.getRevision(clusterId) : lastRevs.get(clusterId);
        if (localLastRev == null) {
            throw new IllegalStateException("Root document does not have a lastRev entry for local clusterId " + clusterId);
        }
        if (this.ns.isDisableBranches()) {
            beforeRev = beforeRev.asTrunkRevision();
            afterRev = afterRev.asTrunkRevision();
        } else {
            beforeRev = this.getBaseRevision(beforeRev);
            afterRev = this.getBaseRevision(afterRev);
        }
        if (beforeRev.equals(afterRev)) {
            return;
        }
        RevisionVector max = beforeRev.pmax(afterRev);
        RevisionVector min = beforeRev.pmin(afterRev);
        if (!max.isRevisionNewer(localLastRev) && !localLastRev.equals(max.getRevision(clusterId))) {
            localPending.addTo(changes, path);
            ++this.stats.numJournalEntries;
        }
        for (Revision to : max) {
            Revision from = min.getRevision(to.getClusterId());
            if (from == null) {
                from = new Revision(0L, 0, to.getClusterId());
            }
            this.stats.numJournalEntries += (long)JournalEntry.fillExternalChanges(changes, null, path, from, to, this.ns.getDocumentStore(), entry -> {}, null, null);
        }
    }

    @NotNull
    private RevisionVector getBaseRevision(RevisionVector rv) {
        if (!rv.isBranch()) {
            return rv;
        }
        Branch b = this.ns.getBranches().getBranch(rv);
        if (b != null) {
            rv = b.getBase(rv.getBranchRevision());
        } else if (this.ns.getBranches().isBranchBase(rv)) {
            rv = rv.asTrunkRevision();
        } else {
            JournalDiffLoader.missingBranch(rv);
        }
        return rv;
    }

    private static void missingBranch(RevisionVector rv) {
        throw new IllegalStateException("Missing branch for revision " + String.valueOf(rv));
    }

    private void logStats() {
        this.stats.sw.stop();
        long timeInSec = this.stats.sw.elapsed(TimeUnit.SECONDS);
        if (timeInSec > 60L) {
            LOG.warn(this.stats.toString());
        } else if (timeInSec > 10L) {
            LOG.info(this.stats.toString());
        } else {
            LOG.debug(this.stats.toString());
        }
    }

    private static class WrappedDiffCache
    extends DiffCache {
        private final Path path;
        private String changes = "";
        private final DiffCache cache;
        private Stats stats;

        WrappedDiffCache(Path path, DiffCache cache, Stats stats) {
            this.path = path;
            this.cache = cache;
            this.stats = stats;
        }

        @Nullable
        String getChanges() {
            return this.changes;
        }

        @Override
        String getChanges(@NotNull RevisionVector from, @NotNull RevisionVector to, @NotNull Path path, @Nullable DiffCache.Loader loader) {
            return this.cache.getChanges(from, to, path, loader);
        }

        @Override
        @NotNull
        DiffCache.Entry newEntry(final @NotNull RevisionVector from, final @NotNull RevisionVector to, boolean local) {
            final DiffCache.Entry entry = this.cache.newEntry(from, to, local);
            return new DiffCache.Entry(){

                @Override
                public void append(@NotNull Path path, @NotNull String changes) {
                    this.trackStats(path, from, to, changes);
                    entry.append(path, changes);
                    if (path.equals(path)) {
                        changes = changes;
                    }
                }

                @Override
                public boolean done() {
                    return entry.done();
                }
            };
        }

        private void trackStats(Path path, RevisionVector from, RevisionVector to, String changes) {
            ++this.stats.numDiffEntries;
            this.stats.keyMemory += (long)path.getMemory();
            this.stats.keyMemory += (long)from.getMemory();
            this.stats.keyMemory += (long)to.getMemory();
            this.stats.valueMemory += (long)new StringValue(changes).getMemory();
        }

        @Override
        @NotNull
        Iterable<CacheStats> getStats() {
            return this.cache.getStats();
        }

        @Override
        public void invalidateAll() {
            this.cache.invalidateAll();
        }
    }

    private static class Stats {
        private final Stopwatch sw = Stopwatch.createStarted();
        private final Path path;
        private final RevisionVector from;
        private final RevisionVector to;
        private long numJournalEntries;
        private long numDiffEntries;
        private long keyMemory;
        private long valueMemory;

        Stats(Path path, RevisionVector from, RevisionVector to) {
            this.path = path;
            this.from = from;
            this.to = to;
        }

        public String toString() {
            String msg = "%d diffs for %s (%s/%s) loaded from %d journal entries in %s. Keys: %s, values: %s, total: %s";
            return String.format(msg, this.numDiffEntries, this.path, this.from, this.to, this.numJournalEntries, this.sw, FileUtils.byteCountToDisplaySize((long)this.keyMemory), FileUtils.byteCountToDisplaySize((long)this.valueMemory), FileUtils.byteCountToDisplaySize((long)(this.keyMemory + this.valueMemory)));
        }
    }
}

