/*
 * Decompiled with CFR 0.152.
 */
package com.artemis;

import com.artemis.Archetype;
import com.artemis.Component;
import com.artemis.ComponentPool;
import com.artemis.ComponentType;
import com.artemis.ComponentTypeFactory;
import com.artemis.Entity;
import com.artemis.InvalidComponentException;
import com.artemis.Manager;
import com.artemis.PackedComponent;
import com.artemis.PooledComponent;
import com.artemis.World;
import com.artemis.utils.Bag;
import com.artemis.utils.ImmutableBag;
import com.artemis.utils.IntBag;
import com.artemis.utils.reflect.ClassReflection;
import com.artemis.utils.reflect.Constructor;
import com.artemis.utils.reflect.ReflectionException;
import java.util.BitSet;

public class ComponentManager
extends Manager {
    private final Bag<Bag<Component>> componentsByType;
    private final Bag<PackedComponent> packedComponents;
    private final Bag<BitSet> packedComponentOwners;
    private final IntBag deleted;
    private final ComponentPool pooledComponents;
    private int highestSeenEntityId;
    protected final ComponentTypeFactory typeFactory;

    protected ComponentManager(int entityContainerSize) {
        this.highestSeenEntityId = entityContainerSize;
        this.componentsByType = new Bag();
        this.packedComponents = new Bag();
        this.packedComponentOwners = new Bag();
        this.pooledComponents = new ComponentPool();
        this.deleted = new IntBag();
        this.typeFactory = new ComponentTypeFactory();
    }

    protected <T extends Component> T create(Entity owner, Class<T> componentClass) {
        ComponentType type = this.typeFactory.getTypeFor(componentClass);
        T component = this.create(owner, type);
        return component;
    }

    <T extends Component> T create(Entity owner, ComponentType type) {
        Class<? extends Component> componentClass = type.getType();
        Component component = null;
        switch (type.getTaxonomy()) {
            case BASIC: {
                component = this.newInstance(componentClass, false);
                break;
            }
            case PACKED: {
                PackedComponent packedComponent = this.packedComponents.safeGet(type.getIndex());
                if (packedComponent == null) {
                    packedComponent = (PackedComponent)this.newInstance(componentClass, type.packedHasWorldConstructor);
                    this.packedComponents.set(type.getIndex(), packedComponent);
                }
                this.getPackedComponentOwners(type).set(owner.getId());
                this.ensurePackedComponentCapacity(owner.id);
                packedComponent.forEntity(owner.id);
                component = packedComponent;
                break;
            }
            case POOLED: {
                try {
                    this.reclaimPooled(owner, type);
                    component = this.pooledComponents.obtain(componentClass, type);
                    break;
                }
                catch (ReflectionException e) {
                    throw new InvalidComponentException(componentClass, "Unable to instantiate component.", e);
                }
            }
            default: {
                throw new InvalidComponentException(componentClass, " unknown component type: " + (Object)((Object)type.getTaxonomy()));
            }
        }
        this.addComponent(owner, type, component);
        return (T)component;
    }

    private void reclaimPooled(Entity owner, ComponentType type) {
        Bag<Component> components = this.componentsByType.safeGet(type.getIndex());
        if (components == null) {
            return;
        }
        Component old = components.safeGet(owner.id);
        if (old != null) {
            this.pooledComponents.free((PooledComponent)old, type);
        }
    }

    private void ensurePackedComponentCapacity(int entityId) {
        if (this.highestSeenEntityId - 1 < entityId) {
            this.highestSeenEntityId = entityId;
            int s = this.packedComponents.size();
            for (int i = 0; s > i; ++i) {
                PackedComponent component = this.packedComponents.get(i);
                if (component == null) continue;
                component.ensureCapacity(entityId + 1);
            }
        }
    }

    protected BitSet getPackedComponentOwners(ComponentType type) {
        BitSet owners = this.packedComponentOwners.safeGet(type.getIndex());
        if (owners == null) {
            owners = new BitSet();
            this.packedComponentOwners.set(type.getIndex(), owners);
        }
        return owners;
    }

    <T extends Component> T newInstance(Class<T> componentClass, boolean constructorHasWorldParameter) {
        try {
            if (constructorHasWorldParameter) {
                Constructor constructor = ClassReflection.getConstructor(componentClass, World.class);
                return (T)((Component)constructor.newInstance(this.world));
            }
            return (T)((Component)ClassReflection.newInstance(componentClass));
        }
        catch (ReflectionException e) {
            throw new InvalidComponentException(componentClass, "Unable to instantiate component.", e);
        }
    }

    private void removeComponents(int entityId) {
        BitSet componentBits = this.world.getEntityManager().componentBits(entityId);
        int i = componentBits.nextSetBit(0);
        while (i >= 0) {
            switch (this.typeFactory.getTaxonomy(i)) {
                case BASIC: {
                    this.componentsByType.get(i).set(entityId, null);
                    break;
                }
                case POOLED: {
                    Component pooled = this.componentsByType.get(i).get(entityId);
                    this.pooledComponents.free((PooledComponent)pooled, i);
                    this.componentsByType.get(i).set(entityId, null);
                    break;
                }
                case PACKED: {
                    PackedComponent pc = this.packedComponents.get(i);
                    pc.forEntity(entityId);
                    pc.reset();
                    break;
                }
                default: {
                    throw new InvalidComponentException(Component.class, " unknown component type: " + (Object)((Object)this.typeFactory.getTaxonomy(i)));
                }
            }
            i = componentBits.nextSetBit(i + 1);
        }
    }

    @Override
    protected void dispose() {
        int s = this.packedComponents.size();
        for (int i = 0; s > i; ++i) {
            PackedComponent component = this.packedComponents.get(i);
            if (component == null || !(component instanceof PackedComponent.DisposedWithWorld)) continue;
            ((PackedComponent.DisposedWithWorld)((Object)component)).free(this.world);
        }
    }

    protected void addComponent(Entity e, ComponentType type, Component component) {
        if (type.isPackedComponent()) {
            this.addPackedComponent(type, (PackedComponent)component);
        } else {
            this.addBasicComponent(e, type, component);
        }
    }

    protected void addComponents(Entity e, Archetype archetype) {
        ComponentType[] types = archetype.types;
        int s = types.length;
        for (int i = 0; s > i; ++i) {
            this.create(e, types[i]);
        }
    }

    private void addPackedComponent(ComponentType type, PackedComponent component) {
        PackedComponent packed = this.packedComponents.safeGet(type.getIndex());
        if (packed == null) {
            this.packedComponents.set(type.getIndex(), component);
        }
    }

    private void addBasicComponent(Entity e, ComponentType type, Component component) {
        Bag<Component> components = this.componentsByType.safeGet(type.getIndex());
        if (components == null) {
            components = new Bag(this.highestSeenEntityId);
            this.componentsByType.set(type.getIndex(), components);
        }
        components.set(e.id, component);
    }

    protected void removeComponent(Entity e, ComponentType type) {
        int index = type.getIndex();
        switch (type.getTaxonomy()) {
            case BASIC: {
                this.componentsByType.get(index).set(e.id, null);
                break;
            }
            case POOLED: {
                Component pooled = this.componentsByType.get(index).get(e.id);
                this.pooledComponents.free((PooledComponent)pooled, type);
                this.componentsByType.get(index).set(e.id, null);
                break;
            }
            case PACKED: {
                PackedComponent pc = this.packedComponents.get(index);
                pc.forEntity(e.id);
                pc.reset();
                this.getPackedComponentOwners(type).clear(e.id);
                break;
            }
            default: {
                throw new InvalidComponentException(type.getType(), " unknown component type: " + (Object)((Object)type.getTaxonomy()));
            }
        }
    }

    protected Bag<Component> getComponentsByType(ComponentType type) {
        if (type.isPackedComponent()) {
            throw new InvalidComponentException(type.getType(), "PackedComponent types aren't supported.");
        }
        Bag<Component> components = this.componentsByType.safeGet(type.getIndex());
        if (components == null) {
            components = new Bag();
            this.componentsByType.set(type.getIndex(), components);
        }
        return components;
    }

    public ImmutableBag<ComponentType> getComponentTypes() {
        return this.typeFactory.types;
    }

    protected Component getComponent(Entity e, ComponentType type) {
        if (type.isPackedComponent()) {
            PackedComponent component = this.packedComponents.safeGet(type.getIndex());
            if (component != null) {
                component.forEntity(e.id);
            }
            return component;
        }
        Bag<Component> components = this.componentsByType.safeGet(type.getIndex());
        if (components != null && components.isIndexWithinBounds(e.id)) {
            return components.get(e.id);
        }
        return null;
    }

    public Bag<Component> getComponentsFor(Entity e, Bag<Component> fillBag) {
        BitSet componentBits = e.getComponentBits();
        int i = componentBits.nextSetBit(0);
        while (i >= 0) {
            if (this.typeFactory.isPackedComponent(i)) {
                fillBag.add(this.packedComponents.get(i));
            } else {
                fillBag.add(this.componentsByType.get(i).get(e.id));
            }
            i = componentBits.nextSetBit(i + 1);
        }
        return fillBag;
    }

    @Override
    public void deleted(int entityId) {
        this.deleted.add(entityId);
    }

    @Override
    public void added(int entityId) {
        if (this.highestSeenEntityId - 1 < entityId) {
            this.ensurePackedComponentCapacity(entityId);
        }
    }

    protected void clean() {
        int s = this.deleted.size();
        if (s > 0) {
            int[] ids = this.deleted.getData();
            for (int i = 0; s > i; ++i) {
                this.removeComponents(ids[i]);
            }
            this.deleted.setSize(0);
        }
    }
}

