package io.agora.push;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;

import io.agora.util.EMLog;

import java.util.ArrayList;

/**
 * \~english
 * The push configuration class.
 */
public class PushConfig {
    private static final String TAG = "PushConfig";

    private String fcmSenderId;
    private String hwAppId;
    private String honorAppId;
    private String miAppId;
    private String miAppKey;
    private String mzAppId;
    private String mzAppKey;
    private String oppoAppKey;
    private String oppoAppSecret;
    private String vivoAppId;
    private String vivoAppKey;
    private ArrayList<PushType> enabledPushTypes;

    private PushConfig() {
    }

    /**
     * \~english
     * Gets the sender ID of FCM.
     * @return  The sender ID of FCM.
     */
    public String getFcmSenderId() {
        return fcmSenderId;
    }

    /**
     * \~english
     * Gets the app ID of Huawei push.
     * @return  The app ID of Huawei push.
     */
    public String getHwAppId() {
        return hwAppId;
    }

    /**
     * \~english
     * Gets the app ID of Honor push.
     * @return  The app ID of Honor push.
     */
    public String getHonorAppId() {
        return honorAppId;
    }

    /**
     * \~english
     * Gets the app ID of Mi push.
     * @return  The app ID of Mi push.
     */
    public String getMiAppId() {
        return this.miAppId;
    }

    /**
     * \~english
     * Gets the app key of Mi push.
     * @return  The app key of Mi push.
     */
    public String getMiAppKey() {
        return this.miAppKey;
    }

    /**
     * \~english
     * Gets the app ID of Meizu push.
     * @return The app ID of Meizu push.
     */
    public String getMzAppId() {
        return this.mzAppId;
    }

    /**
     * \~english
     * Gets the app key of Meizu push.
     * @return The app key of Meizu push.
     */
    public String getMzAppKey() {
        return this.mzAppKey;
    }

    /**
     * \~english
     * Gets the app key of Oppo push.
     * @return The app key of Oppo push.
     */
    public String getOppoAppKey() {
        return this.oppoAppKey;
    }

    /**
     * \~english
     * Gets the secret of Oppo push.
     * @return The secret of Oppo push.
     */
    public String getOppoAppSecret() {
        return this.oppoAppSecret;
    }

    /**
     * \~english
     * Gets the app ID of Vivo push.
     * @return The app ID of Vivo push.
     */
    public String getVivoAppId() {
        return vivoAppId;
    }

    /**
     * \~english
     * Gets the app key of Vivo push.
     * @return The app key of Vivo push.
     */
    public String getVivoAppKey() {
        return vivoAppKey;
    }

    /**
     * \~english
     * Gets the supported push types.
     * @return The list of supported push types.

     */
    public ArrayList<PushType> getEnabledPushTypes() {
        return this.enabledPushTypes;
    }

    @Override
    public String toString() {
        return "PushConfig{" +
                "fcmSenderId='" + fcmSenderId + '\'' +
                ", hwAppId='" + hwAppId + '\'' +
                ", honorAppId='" + honorAppId + '\'' +
                ", miAppId='" + miAppId + '\'' +
                ", miAppKey='" + miAppKey + '\'' +
                ", mzAppId='" + mzAppId + '\'' +
                ", mzAppKey='" + mzAppKey + '\'' +
                ", oppoAppKey='" + oppoAppKey + '\'' +
                ", oppoAppSecret='" + oppoAppSecret + '\'' +
                ", vivoAppId='" + vivoAppId + '\'' +
                ", vivoAppKey='" + vivoAppKey + '\'' +
                ", enabledPushTypes=" + enabledPushTypes +
                '}';
    }

    public static class Builder {
        private Context context;

        private String fcmSenderId;
        private String hwAppId;
        private String honorAppId;
        private String miAppId;
        private String miAppKey;
        private String mzAppId;
        private String mzAppKey;
        private String oppoAppKey;
        private String oppoAppSecret;
        private String vivoAppId;
        private String vivoAppKey;
        private ArrayList<PushType> enabledPushTypes = new ArrayList();

        /**
         * \~english
         * Creates an instance of the PushConfig.
         * @param context The context of Android Activity or Application.
         */
        public Builder(Context context) {
            this.context = context.getApplicationContext();
        }

        /**
         * \~english
         * Creates an instance of the PushConfig.
         * @param context The context of Android Activity or Application.
         * @param config    The instance of PushConfig.
         */
        public Builder(Context context, PushConfig config) {
            this(context);

            if (config == null) {
                return;
            }

            if (config.enabledPushTypes.contains(PushType.MIPUSH)) {
                enableMiPush(config.miAppId, config.miAppKey);
            }

            if (config.enabledPushTypes.contains(PushType.HMSPUSH)) {
                enableHWPush();
            }

            if (config.enabledPushTypes.contains(PushType.VIVOPUSH)) {
                enableVivoPush();
            }

            if (config.enabledPushTypes.contains(PushType.OPPOPUSH)) {
                enableOppoPush(config.oppoAppKey, config.oppoAppSecret);
            }

            if (config.enabledPushTypes.contains(PushType.MEIZUPUSH)) {
                enableMeiZuPush(config.mzAppId, config.mzAppKey);
            }

            if (config.enabledPushTypes.contains(PushType.FCM)) {
                enableFCM(config.fcmSenderId);
            }

            if (config.enabledPushTypes.contains(PushType.HONORPUSH)){
                enableHonorPush();
            }
        }

        /**
         * \~english
         * Passes the app ID and app key of Mi push to enable Mi push on Mi devices.
         * 
         * @return The instance of Builder.
         */
        public Builder enableMiPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.miAppId = appId;
                this.miAppKey = appKey;
                this.enabledPushTypes.add(PushType.MIPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MI push !");
                return this;
            }
        }

        /**
         * \~english
         * Enables Huawei push on Huawei devices.
         * Be sure to set app ID in AndroidManifest or to set agconnect-services.json.
         * @return  The instance of Builder.
         */
        public Builder enableHWPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                hwAppId = appInfo.metaData.getString("com.huawei.hms.client.appid");

                if(hwAppId != null && hwAppId.contains("=")) {
                    hwAppId = hwAppId.split("=")[1];
                } else {
                    // compatible for user set "value='xxxx'" directly
                    int id = appInfo.metaData.getInt("com.huawei.hms.client.appid");
                    hwAppId = String.valueOf(id);
                }
                this.enabledPushTypes.add(PushType.HMSPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                EMLog.e(TAG, "Huawei push must config meta-data: com.huawei.hms.client.appid in AndroidManifest.xml.");
            } catch (ArrayIndexOutOfBoundsException e) {
                EMLog.e(TAG, "Huawei push meta-data: com.huawei.hms.client.appid value must be like this 'appid=xxxxxx'.");
            }
            return this;
        }

        /**
         * \~english
         * Passes the sender ID of FCM push to enable FCM push on FCM devices.

         * @return The instance of Builder.
         */
        public Builder enableFCM(String senderId) {
            if (!TextUtils.isEmpty(senderId)) {
                this.fcmSenderId = senderId;
                this.enabledPushTypes.add(PushType.FCM);
                return this;
            } else {
                EMLog.e(TAG, "senderId can't be empty when enable FCM push !");
                return this;
            }
        }

        /**
         * \~english
         * Passes the app ID and app key of Meizu push to enable Meizu push on Meizu devices.
         * @return The instance of Builder.
         */
        public Builder enableMeiZuPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.mzAppId = appId;
                this.mzAppKey = appKey;
                this.enabledPushTypes.add(PushType.MEIZUPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MEIZU push !");
                return this;
            }
        }

        /**
         * \~english
         * Passes the app ID and app key of Oppo push to enable Oppo push on Oppo devices.
         * The app ID and app key of Oppo push are passed in to make Oppo push available on Oppo devices.
         * @return  The instance of Builder.
         */
        public Builder enableOppoPush(String appKey, String appSecret) {
            if (!TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(appSecret)) {
                this.oppoAppKey = appKey;
                this.oppoAppSecret = appSecret;
                this.enabledPushTypes.add(PushType.OPPOPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appKey or appSecret can't be empty when enable OPPO push !");
                return this;
            }
        }

        /**
         * \~english
         * Be sure to set the app ID and app key in AndroidManifest in order to make Vivo push available on Vivo devices.
         * @return The instance of Builder.
         */
        public Builder enableVivoPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                vivoAppId = appInfo.metaData.getInt("com.vivo.push.app_id") + "";
                vivoAppKey = appInfo.metaData.getString("com.vivo.push.api_key");
                this.enabledPushTypes.add(PushType.VIVOPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                EMLog.e(TAG, "NameNotFoundException: " + e.getMessage());
            }
            return this;
        }

        /**
         * \~english
         * Enables Honor push on Honor devices.
         * Be sure to set app ID in AndroidManifest or to set mcs-services.json.
         * @return  The instance of Builder.
         */
        public Builder enableHonorPush(){
            ApplicationInfo appInfo = null;
            try {
                appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                int id = appInfo.metaData.getInt("com.hihonor.push.app_id");
                honorAppId = String.valueOf(id);
                this.enabledPushTypes.add(PushType.HONORPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                EMLog.e(TAG, "Honor push must config meta-data: com.hihonor.push.app_id in AndroidManifest.xml.");
            }
            return this;
        }

        /**
         * \~english
         * Returns the instance of PushConfig.
         * @return  The PushConfig instance.
         */
        public PushConfig build() {
            PushConfig pushConfig = new PushConfig();
            pushConfig.fcmSenderId = this.fcmSenderId;
            pushConfig.hwAppId = this.hwAppId;
            pushConfig.honorAppId = this.honorAppId;
            pushConfig.miAppId = this.miAppId;
            pushConfig.miAppKey = this.miAppKey;
            pushConfig.mzAppId = this.mzAppId;
            pushConfig.mzAppKey = this.mzAppKey;
            pushConfig.oppoAppKey = this.oppoAppKey;
            pushConfig.oppoAppSecret = this.oppoAppSecret;
            pushConfig.vivoAppId = this.vivoAppId;
            pushConfig.vivoAppKey = this.vivoAppKey;
            pushConfig.enabledPushTypes = this.enabledPushTypes;
            return pushConfig;
        }
    }
}
