/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import io.agora.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import io.agora.chat.adapter.message.EMAMessageBody;
import io.agora.chat.adapter.message.EMAVideoMessageBody;
import io.agora.util.FileHelper;
import io.agora.util.EMLog;

/**
 * \~english
 * The video message body.
 * ```java
 *     VideoMessageBody body = new VideoMessageBody(videoFilePath, thumbPath, duration, filelength);
 * ```
 */
public class VideoMessageBody extends FileMessageBody implements Parcelable {
    private static final String TAG = VideoMessageBody.class.getSimpleName();

    public VideoMessageBody() {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
    }

    public VideoMessageBody(EMAVideoMessageBody body) {
        super(body);
    }

    /**
     * \~english
     * Creates a video message body.
     *
     * @param videoFilePath The path of the video file.
     * @param thumbPath     The thumbnail URI.
     * @param duration      The video duration in seconds.
     * @param filelength    The size of the video file.
     */
    public VideoMessageBody(String videoFilePath, String thumbPath, int duration, long filelength) {
        this(FileHelper.getInstance().formatInUri(videoFilePath), FileHelper.getInstance().formatInUri(thumbPath), duration, filelength);
        EMLog.d(ChatClient.TAG, "VideoMessageBody thumbPath = "+thumbPath);
    }

    /**
     * \~english
     * Creates a video message body.
     *
     * @param videoFilePath The URI of the video thumbnail.
     * @param thumbPath     The thumbnail URI.
     * @param duration      The video duration in seconds.
     * @param filelength    The size of the video file.
     */
    public VideoMessageBody(Uri videoFilePath, Uri thumbPath, int duration, long filelength) {
        super(videoFilePath, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(FileHelper.getInstance().formatInUriToString(thumbPath));
        ((EMAVideoMessageBody) emaObject).setDuration(duration);
        ((EMAVideoMessageBody) emaObject).setDisplayName(FileHelper.getInstance().getFilename(videoFilePath));
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
        ((EMAVideoMessageBody) emaObject).setThumbnailFileLength(FileHelper.getInstance().getFileLength(thumbPath));
        EMLog.d("videomsg", "create video, message body for:" + videoFilePath + " filename = "+FileHelper.getInstance().getFilename(videoFilePath));
        EMLog.d(ChatClient.TAG, "VideoMessageBody thumbPath = "+thumbPath);
    }

    /**
     * create video message body from incoming xmpp message json
     */
    VideoMessageBody(String fileName, String remoteUrl, String thumbnailUrl, int length) {
        super(fileName, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setLocalPath(fileName);
        ((EMAVideoMessageBody) emaObject).setRemotePath(remoteUrl);
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setFileLength(length);
        ((EMAVideoMessageBody) emaObject).setThumbnailFileLength(FileHelper.getInstance().getFileLength(thumbnailUrl));
    }

    /**
     * \~english
     * Gets the size of the video file in bytes.
     *
     * @return The size of the video file.
     */
    public long getVideoFileLength() {
        return ((EMAVideoMessageBody) emaObject).fileLength();
    }

    /**
     * \~english
     * Sets the size of the video file in bytes.
     * This method may be used during database migration.
     * @param filelength The size of the video file.
     */
    public void setVideoFileLength(long filelength) {
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
    }

    /**
     *  \~english
     *  Gets the URL of the thumbnail on the server.
     * @return The URL of the thumbnail on the server.
     */
    public String getThumbnailUrl() {
        return ((EMAVideoMessageBody) emaObject).thumbnailRemotePath();
    }

    /**
     *  \~english
     *  Sets the URL of the thumbnail on the server.
     *  @param thumbnailUrl The URL of the thumbnail on the server.
     */
    public void setThumbnailUrl (String thumbnailUrl) {
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    /**
     * \~english
     * Sets the size of the video thumbnail.
     * @param width     The thumbnail's width.
     * @param height    The thumbnail's height.
     */
    public void setThumbnailSize(int width, int height) {
        ((EMAVideoMessageBody)emaObject).setSize(width, height);
    }
    /**
     * \~english
     * Gets the width of the video thumbnail.
     * @return  The width of video thumbnail.
     */
    public int getThumbnailWidth() {
        return ((EMAVideoMessageBody)emaObject).width();
    }

    /**
     * \~english
     * Gets the height of the video thumbnail.
     * @return  The height of the video thumbnail.
     */
    public int getThumbnailHeight() {
        return ((EMAVideoMessageBody)emaObject).height();
    }

    /**
     *  \~english
     *  Gets the local absolute path of the video thumbnail or its URI in string format.
     *
     *  @return The local path of the video thumbnail.
     */
    public String getLocalThumb() {
        return FileHelper.getInstance().formatOutLocalUrl(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
    }

    /**
     *  \~english
     *  Gets the local URI of the video thumbnail.
     *
     *  @return The local URI of the video thumbnail.
     */
    public Uri getLocalThumbUri() {
        return FileHelper.getInstance().formatOutUri(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
    }

    /**
     * \~english
     * Sets the path of the video thumbnail.
     * @param localThumbPath The path of the video thumbnail, which can be the absolute path or URI of the video thumbnail.
     */
    public void setLocalThumb(String localThumbPath) {
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(FileHelper.getInstance().formatInUriToString(localThumbPath));
        ((EMAVideoMessageBody) emaObject).setThumbnailFileLength(FileHelper.getInstance().getFileLength(localThumbPath));
    }

    /**
     * \~english
     * Sets the path of the video thumbnail.
     * @param localThumbPath    The URI of video thumbnail.
     */
    public void setLocalThumb(Uri localThumbPath) {
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(FileHelper.getInstance().formatInUriToString(localThumbPath));
        ((EMAVideoMessageBody) emaObject).setThumbnailFileLength(FileHelper.getInstance().getFileLength(localThumbPath));
    }

    /**
     * \~english
     * Gets the video duration in seconds.
     *
     * @return  The video duration in seconds.
     */
    public int getDuration() {
        return (int) ((EMAVideoMessageBody) emaObject).duration();
    }

    public String toString() {
        return "video: " + ((EMAVideoMessageBody) emaObject).displayName() +
                ", localUrl: " + ((EMAVideoMessageBody) emaObject).getLocalUrl() +
                ", remoteUrl: " + ((EMAVideoMessageBody) emaObject).getRemoteUrl() +
                ", thumbnailUrl: " + ((EMAVideoMessageBody) emaObject).thumbnailLocalPath() +
                ", length: " + ((EMAVideoMessageBody) emaObject).fileLength();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        // members from FileMessageBody
        dest.writeString(((EMAVideoMessageBody) emaObject).displayName());
        dest.writeString(((EMAVideoMessageBody) emaObject).getLocalUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).getRemoteUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeInt(((EMAVideoMessageBody) emaObject).duration());
        dest.writeLong(((EMAVideoMessageBody) emaObject).fileLength());
        dest.writeInt(((EMAVideoMessageBody) emaObject).width());
        dest.writeInt(((EMAVideoMessageBody) emaObject).height());
        dest.writeLong(((EMAVideoMessageBody) emaObject).thumbnailFileLength());
    }

    public static final Parcelable.Creator<VideoMessageBody> CREATOR = new Creator<VideoMessageBody>() {

        @Override
        public VideoMessageBody[] newArray(int size) {
            return new VideoMessageBody[size];
        }

        @Override
        public VideoMessageBody createFromParcel(Parcel in) {

            return new VideoMessageBody(in);
        }
    };

    private VideoMessageBody(Parcel in) {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
        // members from FileMessageBody
        ((EMAVideoMessageBody) emaObject).setDisplayName(in.readString());
        ((EMAVideoMessageBody) emaObject).setLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setDuration(in.readInt());
        ((EMAVideoMessageBody) emaObject).setFileLength(in.readLong());
        int width = in.readInt();
        int height = in.readInt();
        ((EMAVideoMessageBody) emaObject).setSize(width, height);
        ((EMAVideoMessageBody) emaObject).setThumbnailFileLength(in.readLong());
    }

    /**
     *  \~english
     *  Gets the secret key of the video thumbnail.
     *  The secret key is used for validation during download of the video thumbnail.
     * @return The secret key of the video thumbnail.
     */
    public String getThumbnailSecret() {
        return ((EMAVideoMessageBody) emaObject).thumbnailSecretKey();
    }

    /**
     * \~english
     * Sets the secret key of the video thumbnail.
     * @param secret The secret key of the video thumbnail.
     */
    public void setThumbnailSecret(String secret) {
        ((EMAVideoMessageBody) emaObject).setThumbnailSecretKey(secret);
    }

    /**
     * \~english
     * Gets the download status of the video thumbnail.
     *
     * @return  The download status of the video thumbnail.
     */
    public EMDownloadStatus thumbnailDownloadStatus() {
        EMADownloadStatus _status = ((EMAVideoMessageBody) emaObject).thumbnailDownloadStatus();
        switch (_status) {
            case DOWNLOADING:
                return EMDownloadStatus.DOWNLOADING;
            case SUCCESSED:
                return EMDownloadStatus.SUCCESSED;
            case FAILED:
                return EMDownloadStatus.FAILED;
            case PENDING:
                return EMDownloadStatus.PENDING;
        }
        return EMDownloadStatus.SUCCESSED;
    }

    /**
     * \~english
     * Sets the download status of the video thumbnail.
     *
     * @param status  The download status of the video thumbnail.
     */
    public void setThumbnailDownloadStatus(EMDownloadStatus status) {
        ((EMAVideoMessageBody) emaObject).setThumbnailDownloadStatus(EMADownloadStatus.valueOf(status.name()));
    }
}
