package io.agora.chat;

import android.os.AsyncTask;

import io.agora.Error;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMATranslateManager;
import io.agora.chat.adapter.EMATranslateResult;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * \~english
 * The translation information management class, which is responsible for searching for, adding, modifying, and deleting translation information that is saved in the SDK local database.
 * @deprecated use {@link ChatManager#fetchSupportLanguages(ValueCallBack)} {@link ChatManager#translateMessage(ChatMessage, List, ValueCallBack)} instead.
 */
@Deprecated
public class TranslationManager {
    private static final String TAG = TranslationManager.class.getSimpleName();

    public final static int MaxTranslationTextSize = 5000;

    static private TranslationManager instance = null;

    private Translator mTranslator;

    private AtomicBoolean mInitializing = new AtomicBoolean(false);
    private AtomicBoolean mInitialized = new AtomicBoolean(false);

    private List<Language> mLanguageList;

    EMATranslateManager emaObject;
    static final int MaxCacheSize = 10000;
    private CacheManager mCacheManager;

    TranslationManager(EMATranslateManager translateManager){
        emaObject = translateManager;
        mLanguageList = new ArrayList<>();
        mCacheManager = new CacheManager(MaxCacheSize);
    }

    /**
     * \~english
     * Initializes translation parameters.
     *
     * @param params The parameter objects.
     */
    public void init(TranslateParams params) {
        if( !ChatClient.getInstance().isLoggedInBefore())
            return;

        if(mInitializing.compareAndSet(false, true)) {
            AsyncTask.execute(new Runnable() {
                @Override
                public void run() {
                    cleanCache();
                    loadIds(params.LoadCount);
                    mInitialized.set(true);
                    mTranslator = new Translator(params);
                    mLanguageList = mTranslator.getSupportedLanguages();
                }
            });
        }
    }

    /**
     * \~english
     * Whether the initialization is completed.
     *
     * @return boolean - `true`: Yes; 
     * - `false`: No.
     */
    public boolean isInitialized() {
        return mInitialized.get();
    }

    /**
     * \~english
     * Gets the list of supported languages for translation.
     *
     * @return The list of supported languages for translation.
     */
    public List<Language> getSupportedLanguages() {
        if(!mInitialized.get())
            return new ArrayList<>();

        return mLanguageList;
    }

    /**
     * \~english
     * Translates the text.
     *
     * @param messageId The message ID.
     * @param conversationId The conversation ID.
     * @param messageText The text to be translated.
     * @param targetLanguageCode The code of the target language in Microsoft Translation Service.
     * 
     * @return The translation record.
     */
    public void translate(String messageId, String conversationId, String messageText, String targetLanguageCode, ValueCallBack<TranslationResult> callback) {
        TranslationResult result;

        if(! mInitializing.get()) {
            callback.onError(Error.TRANSLATE_NOT_INIT, "TranslationManager is not initialized");
            return;
        }

        if(messageText.length() > MaxTranslationTextSize) {
            callback.onError(Error.TRANSLATE_INVALID_PARAMS, "Text exceeds limit");
            return;
        }

        if(isTranslationResultForMessage(messageId)) {
            result = getTranslationResult(messageId);
        }else{
            result = new TranslationResult(messageId);
            result.setConversationId(conversationId);
        }

        mTranslator.translate(messageText, targetLanguageCode, (TranslationText, ErrorText) -> {
            if (!TranslationText.isEmpty()) {
                result.setShowTranslation(true);
                result.setTranslatedText(TranslationText);
                result.setTranslateCount(result.translateCount() + 1);
                updateTranslationResult(result);
                callback.onSuccess(result);
            }else{
                callback.onError(Error.TRANSLATE_FAIL, ErrorText);
            }
        });
    }

    /**
     * \~english
     * Deletes translation records by the message ID.
     *
     * @param messageId The message ID.
     */
    public void removeTranslationResult(String messageId) {
        removeTranslationResults(Arrays.asList(messageId));
    }

    /**
     * \~english
     * Batch delete translation records by the message ID.
     *
     * @param messageIds The message ID array.
     */
    public void removeTranslationResults(List<String> messageIds) {
        if(! mInitializing.get())
            return;

        deleteTranslationResults(messageIds);
    }

    /**
     * \~english
     * Deletes all translation records.
     *
     */
    public void clearTranslations() {
        if(! mInitializing.get())
            return;

        deleteAll();
    }

    /**
     * \~english
     * Deletes translation records by the conversation ID.
     *
     * @param conversationId The conversation ID.
     */
    public void removeResultsByConversationId(String conversationId) {
        if(! mInitializing.get())
            return;
        removeTranslationsByConversationId(conversationId);
        mCacheManager.removeByConversationId(conversationId);
    }

    /**
     * \~english
     * Gets translation records by message ID.
     *
     * @param messageId The message ID.
     */
    public TranslationResult getTranslationResult(String messageId) {
        if(! mInitializing.get())
            return null;

        if(!mCacheManager.check(messageId))
            return null;

        if(mCacheManager.get(messageId) != null)
            return mCacheManager.get(messageId);

        TranslationResult result =  getTranslationResultByMsgId(messageId);

        return result;
    }

    /**
     * \~english
     * Checks whether there are translation records in the cache.
     *
     * @param messageId The message ID.
     */
    public boolean isTranslationResultForMessage(String messageId) {
        if(!mInitialized.get())
            return false;

        return isMessageResult(messageId);
    }

    /**
     * \~english
     * Resets parameters to clear the cache.
     */
    public void logout() {
        if(!mInitialized.get())
            return;

        mInitializing.set(false);
        mInitialized.set(false);

        cleanCache();
        mLanguageList.clear();
    }

    private void loadIds(int loadCount) {
        List<TranslationResult> results = loadTranslateResults(loadCount);
        if(results.size() > 0) {
            for(TranslationResult result : results) {
                mCacheManager.add(result);
            }
        }
    }

    /**
     * \~english
     * Updates the translation records.
     *
     * @param result The translation record object.
     */
    public void updateTranslationResult(TranslationResult result) {
        String messageId = result.msgId();

        if(mCacheManager.check(messageId)) {
            removeTranslationResult(messageId);
        }

        mCacheManager.add(result);
        updateTranslate(result);
    }

    private void deleteTranslationResults(List<String> messageIds) {
        List<String> toBeDeletedMsgIds = new ArrayList<>();
        for(String messageId : messageIds) {
            if(mCacheManager.check(messageId))
                toBeDeletedMsgIds.add(messageId);
        }

        removeTranslationsByMsgId(toBeDeletedMsgIds);
        mCacheManager.removeByMsgIds(toBeDeletedMsgIds);
    }

    private void deleteAll() {
        removeAllTranslations();
        mCacheManager.clear();
    }

    private boolean isMessageResult(String messageId) {
        return mCacheManager.check(messageId);
    }

    private void cleanCache() {
        mCacheManager.clear();
    }

    // 调用底层实现
    private boolean updateTranslate(TranslationResult result){
        return emaObject.updateTranslation(result.emaObject);
    }

    private TranslationResult getTranslationResultByMsgId(String msgId){

        EMATranslateResult aResult = emaObject.getTranslationResultByMsgId(msgId);
        if(aResult == null){
            return new TranslationResult(msgId);
        }
        return new TranslationResult(aResult);
    }

    private List<TranslationResult> loadTranslateResults(int count){
        List<EMATranslateResult> list = emaObject.loadTranslateResults(count);
        List<TranslationResult> results = new ArrayList<>();
        if(list.size() > 0){
            for(int i = 0; i < list.size(); i++){
                EMATranslateResult aResult = list.get(i);
                TranslationResult result = new TranslationResult(aResult);
                results.add(result);
            }
            return results;
        }else {
            return results;
        }
    }

    private boolean removeTranslationsByMsgId(List<String> msgIds){
        return emaObject.removeTranslationsByMsgId(msgIds);
    }

    private boolean removeTranslationsByConversationId(String conversationId){
        return emaObject.removeTranslationsByConversationId(conversationId);
    }

    private boolean removeAllTranslations(){
        return emaObject.removeAllTranslations();
    }
}

@Deprecated
class CacheManager {
    private TranslationCache mTranslationCache;
    private Set<Integer> mIdCache;

    CacheManager(int maxCacheSize) {
        mTranslationCache =  new TranslationCache(maxCacheSize);
        mIdCache = new HashSet<Integer>();
    }

    void add(TranslationResult result) {
        synchronized (CacheManager.class) {
            String id = result.msgId();
            mTranslationCache.put(id, result);

            int value = id2Value(id);
            mIdCache.add(value);
        }
    }

    void removeByMsgIds(List<String> messageIds) {
        synchronized (CacheManager.class) {
            for(String messageId : messageIds) {
                mTranslationCache.remove(messageId);

                int value = id2Value(messageId);
                mIdCache.remove(value);
            }
        }
    }

    void removeByConversationId(String conversationId) {
        mTranslationCache.removeByConversationId(conversationId);
    }

    TranslationResult get(String messageId) {
        TranslationResult result;
        synchronized (CacheManager.class) {
            result = mTranslationCache.get(messageId);
        }

        return result;
    }

    boolean check(String messageId) {
        boolean isExist = false;
        synchronized (CacheManager.class) {
            int value = id2Value(messageId);
            isExist = mIdCache.contains(value);
        }

        return isExist;
    }

    void clear() {
        synchronized (CacheManager.class) {
            mTranslationCache.clear();
            mIdCache.clear();
        }
    }

    private int id2Value(String messageId) {
        //get message timestamp part and return as 32 bit int
        long value = Long.parseLong(messageId);

        value >>= 22;
        value &= 0xffffffff;

        return (int)value;
    }
}

@Deprecated
class TranslationCache extends LRUCache<String, TranslationResult> {
    private HashMap<String, List<String>> mConversationMap;


    public TranslationCache(int maxSize) {
        super(maxSize);
        mConversationMap = new HashMap<>();
    }

    @Override
    public void clear() {
        super.clear();
        mConversationMap.clear();
    }

    @Override
    public void remove(String key) {
        TranslationResult result = get(key);
        List<String> messageList = mConversationMap.get(result.conversationId());
        if(messageList != null)
            messageList.remove(key);

        super.remove(key);

    }

    @Override
    public void put(String key, TranslationResult value) {
        super.put(key, value);

        if(mConversationMap.get(value.conversationId()) == null)
            mConversationMap.put(value.conversationId(), new ArrayList<>());

        List<String> messageIdList = mConversationMap.get(value.conversationId());
        assert messageIdList != null;
        messageIdList.add(key);
    }

    public void removeByConversationId(String conversationId) {
        List<String> messageList = mConversationMap.get(conversationId);

        if(messageList != null) {
            for(String messageId : messageList) {
                super.remove(messageId);
            }
        }
    }
}

@Deprecated
class LRUCache<K, V>{

    // Define Node with pointers to the previous and next items and a key, value pair
    class Node<T, U> {
        Node<T, U> previous;
        Node<T, U> next;
        T key;
        U value;

        public Node(Node<T, U> previous, Node<T, U> next, T key, U value){
            this.previous = previous;
            this.next = next;
            this.key = key;
            this.value = value;
        }
    }

    private HashMap<K, Node<K, V>> cache;
    private Node<K, V> leastRecentlyUsed;
    private Node<K, V> mostRecentlyUsed;
    private int maxSize;
    private int currentSize;

    public LRUCache(int maxSize){
        this.maxSize = maxSize > 1 ? maxSize : 2;;
        this.currentSize = 0;
        leastRecentlyUsed = new Node<K, V>(null, null, null, null);
        mostRecentlyUsed = leastRecentlyUsed;
        cache = new HashMap<K, Node<K, V>>();
    }

    public void clear() {
        this.currentSize = 0;
        leastRecentlyUsed = new Node<K, V>(null, null, null, null);
        mostRecentlyUsed = leastRecentlyUsed;
        cache.clear();
    }

    public void remove(K key) {
        Node<K, V> tempNode = cache.get(key);
        if(tempNode == null)
            return;

        cache.remove(key);
        currentSize--;

        if(tempNode.key == mostRecentlyUsed.key) {
            mostRecentlyUsed = mostRecentlyUsed.previous;
            if(mostRecentlyUsed != null)
                mostRecentlyUsed.next = null;
            else
                leastRecentlyUsed = null;
        } else if (tempNode.key == leastRecentlyUsed.key) {
            leastRecentlyUsed = leastRecentlyUsed.next;
            if(leastRecentlyUsed != null)
                leastRecentlyUsed.previous = null;
            else
                mostRecentlyUsed = null;
        } else {
            tempNode.next.previous = tempNode.previous;
            tempNode.previous.next = tempNode.next;
        }
    }

    public boolean contains(K key) {
        return cache.containsKey(key);
    }

    public V get(K key){
        Node<K, V> tempNode = cache.get(key);
        if (tempNode == null){
            return null;
        } else if (tempNode.key == mostRecentlyUsed.key){
            // If MRU leave the list as it is
            return mostRecentlyUsed.value;
        }

        // Get the next and previous nodes
        Node<K, V> nextNode = tempNode.next;
        Node<K, V> previousNode = tempNode.previous;

        if (tempNode.key == leastRecentlyUsed.key){
            // If at the left-most, we update LRU
            nextNode.previous = null;
            leastRecentlyUsed = nextNode;
        } else if (tempNode.key != mostRecentlyUsed.key){
            // If we are in the middle, we need to update the items before and after our item
            previousNode.next = nextNode;
            nextNode.previous = previousNode;
        }

        // Finally move our item to the MRU
        tempNode.previous = mostRecentlyUsed;
        mostRecentlyUsed.next = tempNode;
        mostRecentlyUsed = tempNode;
        mostRecentlyUsed.next = null;

        return tempNode.value;

    }

    public void put(K key, V value){
        if (cache.containsKey(key)){
            return;
        }

        // Put the new node at the right-most end of the linked-list
        Node<K, V> myNode = new Node<K, V>(mostRecentlyUsed, null, key, value);
        mostRecentlyUsed.next = myNode;
        cache.put(key, myNode);
        mostRecentlyUsed = myNode;

        // Delete the left-most entry and update the LRU pointer
        if (currentSize == maxSize){
            cache.remove(leastRecentlyUsed.key);
            leastRecentlyUsed = leastRecentlyUsed.next;
            leastRecentlyUsed.previous = null;
            return;
        }

        currentSize++;
    }
}

