/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import android.text.TextUtils;
import android.util.Pair;

import io.agora.CallBack;
import io.agora.Error;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAError;
import io.agora.chat.adapter.EMAPushConfigs;
import io.agora.chat.adapter.EMAPushManager;
import io.agora.chat.adapter.EMASilentModeItem;
import io.agora.chat.core.EMPreferenceUtils;
import io.agora.cloud.EMHttpClient;
import io.agora.exceptions.ChatException;
import io.agora.util.DeviceUuidFactory;
import io.agora.util.EMLog;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * \~english
 * The message push configuration options.
 */
public class PushManager {
    private static final String TAG = PushManager.class.getSimpleName();

    /**
     * \~english
     * The push message presentation style: SimpleBanner represents the presentation of a simple message,
     * and MessageSummary represents the presentation of message content.
     */
    public enum DisplayStyle {
        SimpleBanner, MessageSummary
    }

    ChatClient mClient;
    EMAPushManager emaObject;
    PushManager(ChatClient client, EMAPushManager pushManager) {
        emaObject = pushManager;
        mClient = client;
    }


    /**
     * \~english
     * Turns on the push notification.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @throws ChatException
     * @deprecated Use {@link PushManager#setSilentModeForAll(SilentModeParam, ValueCallBack)} instead.
     */
    @Deprecated
    public void enableOfflinePush() throws ChatException {
        EMAError error = new EMAError();
        emaObject.enableOfflineNotification(error);
        handleError(error);
    }

    /**
     * \~english
     * Do not push the offline messages within the specified time period (24-hour clock).

     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param start The start hour.
     * @param end The end hour.
     * @throws ChatException A description of the cause of the exception.
     * @deprecated Use {@link PushManager#setSilentModeForAll(SilentModeParam, ValueCallBack)} instead.
     */
    @Deprecated
    public void disableOfflinePush(int start, int end) throws ChatException {
        EMAError error = new EMAError();
        emaObject.disableOfflineNotification(start, end, error);
        handleError(error);
    }

    /**
     * \~english
     * Get the push configs from cache.
     * @return The push configs.
     */
    public PushConfigs getPushConfigs(){
        EMAPushConfigs pushConfigs = emaObject.getPushConfigs();
        if(pushConfigs == null){
            return null;
        }
        return new PushConfigs(pushConfigs);
    }

    /**
     * \~english
     * Get the push configs from the server.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return The push configs.
     * @throws ChatException A description of the cause of the exception.
     */
    public PushConfigs getPushConfigsFromServer() throws ChatException{
        EMAError error = new EMAError();
        EMAPushConfigs pushConfigs = emaObject.getPushConfigsFromServer(error);
        handleError(error);

        return new PushConfigs(pushConfigs);
    }


    /**
     * \~english
     * Sets whether to turn on or turn off the push notification for the the specified groups.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param groupIds The list of groups to be set.
     * @param noPush - `true`: Turns off the notification;
     *               - `false`: Turns on the notification.
     * @throws ChatException A description of the cause of the exception.
     * @deprecated Use {@link PushManager#setSilentModeForConversation(String, Conversation.ConversationType, SilentModeParam, ValueCallBack)} instead,set DND Settings for each session.
     */
    @Deprecated
    public void updatePushServiceForGroup(List<String> groupIds, boolean noPush) throws ChatException {
        EMAError error = new EMAError();
        emaObject.updatePushServiceForGroup(groupIds, noPush, error);
        handleError(error);
    }
    /**
     * \~english
     * Sets whether the specified group accepts the offline message notification.
     * @param userIds The list of users to be set.
     * @param noPush - `true`：turn off the notification;
     *               - `false`：turn on the notification.
     * @throws ChatException A description of the cause of the exception.
     * @deprecated Use {@link PushManager#setSilentModeForConversation(String, Conversation.ConversationType, SilentModeParam, ValueCallBack)} instead,set DND Settings for each session
     */
    @Deprecated
    public void updatePushServiceForUsers(List<String> userIds, boolean noPush) throws ChatException {
        EMAError error = new EMAError();
        emaObject.updatePushServiceForUsers(userIds, noPush, error);
        handleError(error);
    }

    /**
     * \~english
     * Gets the list of groups which have blocked the push notification.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return The list of groups that blocked the push notification.
     * @deprecated Use {@link PushManager#getSilentModeForConversation(String, Conversation.ConversationType, ValueCallBack)} instead,get the DND Settings for each conversation to determine
     */
    @Deprecated
    public List<String> getNoPushGroups(){
        return emaObject.getNoPushGroups();
    }
    /**
     * \~english
     * Gets the list of user ID which have blocked the push notification from the cache.
     * 
     * Note:
     * If you needs to get the latest data, call {@Link EmpushManager# getPushConfigsFromServer()} before calling this method.
     * @return  The list of users who have blocked the push notification.
     * @deprecated Use {@link PushManager#getSilentModeForConversation(String, Conversation.ConversationType, ValueCallBack)} instead,get the DND Settings for each conversation to determine.
     */
    @Deprecated
    public List<String> getNoPushUsers(){
        return emaObject.getNoPushUsers();
    }

    /**
	 * \~english
     * Updates the push display nickname of the current user.
	 * This method can be used to set a push display nickname, the push display nickname will be used to show for offline push notification.
     * When the app user changes the nickname in the user profile(use {@link UserInfoManager#updateOwnInfo(UserInfo, ValueCallBack)
     * or {@link UserInfoManager#updateOwnInfoByAttribute(UserInfo.UserInfoType, String, ValueCallBack)} to set},
     * be sure to also call this method to update to prevent the display differences.
     *
     * Reference:
     * The asynchronous method see {@link #asyncUpdatePushNickname(String, CallBack)}
	 *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param nickname The push display nickname, which is different from the nickname in the user profile.
	 */
    public boolean updatePushNickname(String nickname) throws IllegalArgumentException, ChatException {
        if (TextUtils.isEmpty(nickname)) {
            throw new IllegalArgumentException("nick name is null or empty");
        }
        String currentUser = ChatClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = ChatClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushNickname(nickname, error);
        handleError(error);
        return true;
    }

    /**
	 * \~english
	 * Update the push display nickname of the current user.
     * 
     * This is an asynchronous method.
     * 
     * This method can be used to set a push nickname, the push nickname will be used for offline push notification.
     * When the user changes the nickname in the user profile(use {@link UserInfoManager#updateOwnInfo(UserInfo, ValueCallBack)
     * or {@link UserInfoManager#updateOwnInfoByAttribute(UserInfo.UserInfoType, String, ValueCallBack)} to set),
     * be sure to also call this method to update the display nickname to prevent the display differences.
     *
     * Reference:
     * The synchronous method see {@link #updatePushNickname(String)}.
     *
	 * @param nickname  The push nickname, which is different from the nickname in user profiles.
	 */
    public void asyncUpdatePushNickname(String nickname, CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushNickname(nickname);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(Error.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
	 * \~english
	 * Update the push message style. The default value is {@link DisplayStyle#SimpleBanner}.
     * 
     * Reference:
     * The asynchronous method see {@link #asyncUpdatePushDisplayStyle(DisplayStyle, CallBack)}
     *
     * This is a synchronous method and blocks the current thread.
	 *
	 * @param style The push message display style.
	 */
    public void updatePushDisplayStyle(DisplayStyle style) throws IllegalArgumentException, ChatException
    {
        String currentUser = ChatClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = ChatClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushDisplayStyle(style.ordinal(), error);
        handleError(error);
    }

    /**
	 * \~english
	 * Update the push message style. The default value is {@link DisplayStyle#SimpleBanner}.
     *   
     * This is an asynchronous method.
     * 
     * Reference:
     * The synchronous method see {@link #updatePushDisplayStyle(DisplayStyle)}
	 *
	 * @param style The push message style.
	 */
    public void asyncUpdatePushDisplayStyle(DisplayStyle style, CallBack callback)
    {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushDisplayStyle(style);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(Error.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    private void reportPushAction(String parameters) throws IllegalArgumentException, ChatException
    {
        EMAError error = new EMAError();
        emaObject.reportPushAction(parameters, error);
        handleError(error);
    }

    private void handleError(EMAError error) throws ChatException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new ChatException(error);
        }
    }

    /**
     * \~english
     * The push actions.
     */
    public enum EMPushAction{
        ARRIVE("arrive"),/** \~chinese 送达事件。  \~english Arrived event. */
        CLICK("click");/** \~chinese 点击事件。  \~english Clicked event. */

        private String name;

        EMPushAction(String name){
            this.name = name;
        }
    }

    /**
     * \~english
     * Offline push notification type enumeration class.
     */
    public enum PushRemindType {
        ALL, /** \～chinese 收取全部离线推送。 \~english  Collect all offline push.  */
        MENTION_ONLY, /** \～chinese 只收取@我的离线推送。 \~english  Only receive @me offline push.  */
        NONE /** \～chinese 不收取离线推送。 \~english  Offline push is not collected.  */
    }

    /**
     * \~english
     * Set the DND of the conversation.
     * @param conversationId The conversation id.
     * @param type The conversation type.
     * @param param Push DND parameters offline.
     * @param callBack Complete the callback.
     */
    public void setSilentModeForConversation(String conversationId, Conversation.ConversationType type, SilentModeParam param, ValueCallBack<SilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.setSilentModeForConversation(conversationId, type.ordinal(), param.emaObject, error);
                    handleError(error);
                    callBack.onSuccess(new SilentModeResult(item));
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Clear the setting of offline push notification type for the conversation.
     * After clearing, the session follows the Settings of the current logged-in user  {@link PushManager#setSilentModeForAll(SilentModeParam, ValueCallBack)}.
     * @param conversationId The conversation id.
     * @param type The conversation type.
     * @param callBack Complete the callback.
     */
    public void clearRemindTypeForConversation(String conversationId, Conversation.ConversationType type, CallBack callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    emaObject.clearRemindTypeForConversation(conversationId, type.ordinal(), error);
                    handleError(error);
                    callBack.onSuccess();
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Gets the DND setting of the conversation.
     * @param conversationId The conversation id.
     * @param type The conversation type.
     * @param callBack Complete the callback.
     */
    public void getSilentModeForConversation(String conversationId, Conversation.ConversationType type, ValueCallBack<SilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.getSilentModeForConversation(conversationId, type.ordinal(), error);
                    handleError(error);
                    callBack.onSuccess(new SilentModeResult(item));
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Example Set the DND Settings for the current login user.
     * @param param Push DND parameters offline.
     * @param callBack Complete the callback.
     */
    public void setSilentModeForAll(SilentModeParam param, ValueCallBack<SilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.setSilentModeForAll(param.emaObject, error);
                    handleError(error);
                    callBack.onSuccess(new SilentModeResult(item));
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Gets the DND Settings of the current login user.
     * @param callBack Complete the callback.
     */
    public void getSilentModeForAll(ValueCallBack<SilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.getSilentModeForAll(error);
                    handleError(error);
                    callBack.onSuccess(new SilentModeResult(item));
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Obtain the DND Settings of specified conversations in batches.
     * @param conversationList The conversation list.
     * @param callBack Complete the callback.
     */
    public void getSilentModeForConversations(List<Conversation> conversationList, ValueCallBack<Map<String, SilentModeResult>> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    StringBuilder userIdStr = new StringBuilder();
                    StringBuilder groupIdStr = new StringBuilder();
                    for(Conversation conversation : conversationList){
                        if(conversation.getType() == Conversation.ConversationType.Chat){
                            if(!userIdStr.toString().isEmpty()){
                                userIdStr.append(",");
                            }
                            userIdStr.append(conversation.conversationId());
                        } else {
                            if(!groupIdStr.toString().isEmpty()){
                                groupIdStr.append(",");
                            }
                            groupIdStr.append(conversation.conversationId());
                        }
                    }
                    Map<String, String> map = new HashMap<>();
                    map.put("user", userIdStr.toString());
                    map.put("group", groupIdStr.toString());
                    EMAError error = new EMAError();
                    List<EMASilentModeItem> items = emaObject.getSilentModeForConversations(map,  error);
                    handleError(error);
                    Map<String, SilentModeResult> itemMap = new HashMap<>();
                    for(EMASilentModeItem item : items){
                        itemMap.put(item.getConversationId(), new SilentModeResult(item));
                    }
                    callBack.onSuccess(itemMap);
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });

    }

    /**
     * \~english
     * Set user push translation language.
     * @param languageCode language code.
     * @param callBack Complete the callback.
     */
    public void setPreferredNotificationLanguage(String languageCode, CallBack callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    emaObject.setPushPerformLanguage(languageCode, error);
                    handleError(error);
                    callBack.onSuccess();
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Gets the push translation language set by the user.
     * @param callBack Complete the callback.
     */
    public void getPreferredNotificationLanguage(ValueCallBack<String> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    String languageCode = emaObject.getPushPerformLanguage(error);
                    handleError(error);
                    callBack.onSuccess(languageCode);
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Binds device token to chat server.
     * If device token is null or "", means that unbinding device token from chat server.
     * @param notifierName Means current device ID, follow:
     *                     FCM - Sender ID;
     *                     Hawei - App ID;
     *                     Xiaomi - App ID;
     *                     Meizu - App ID;
     *                     OPPO - App Key;
     *                     Vivo - App ID + "#" + App Key;
     * @param deviceToken  Device token generated by the device manufacture
     * @param callBack
     */
    public synchronized void bindDeviceToken(String notifierName, String deviceToken, CallBack callBack) {
        if(mClient == null || !mClient.isSdkInited()) {
            if(callBack != null) {
                callBack.onError(Error.GENERAL_ERROR, "SDK should init first!");
            }
            return;
        }
        mClient.execute(()-> {
            try {
                bindDeviceToken(notifierName, deviceToken);
                if(callBack != null) {
                    callBack.onSuccess();
                }
            } catch (ChatException e) {
                if(callBack != null) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    void unBindDeviceToken() throws ChatException {
        bindDeviceToken(EMPreferenceUtils.getInstance().getPushNotifierName(), "");
    }

    synchronized void bindDeviceToken(String notifierName, String deviceToken) throws ChatException {
        if(mClient == null || !mClient.isSdkInited()) {
            throw new ChatException(Error.GENERAL_ERROR, "SDK should init first!");
        }
        if(!mClient.isLoggedInBefore()) {
            throw new ChatException(Error.GENERAL_ERROR, "You need to log in first!");
        }
        if(TextUtils.isEmpty(notifierName)) {
            throw new ChatException(Error.INVALID_PARAM, "Notifier name should not be empty!");
        }
        EMPreferenceUtils preferenceUtils = EMPreferenceUtils.getInstance();
        String savedToken = preferenceUtils.getPushToken();
        if(!TextUtils.isEmpty(savedToken) && TextUtils.equals(savedToken, deviceToken)) {
            if (!ChatClient.getInstance().getChatConfigPrivate().isNewLoginOnDevice()) {
                EMLog.e(TAG, TAG + " not first login, ignore token upload action.");
                return;
            }
            EMLog.d(TAG, "push token not change, but last login is not on this device, upload to server");
        }
        String remoteUrl = ChatClient.getInstance().getChatConfigPrivate().getBaseUrl(true, false) + "/users/"
                + ChatClient.getInstance().getCurrentUser() + "/push/binding";
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(ChatClient.getInstance().getContext());
        JSONObject json = new JSONObject();
        try {
            json.put("device_token", TextUtils.isEmpty(notifierName)? "" : deviceToken);
            json.put("notifier_name", notifierName);
            json.put("device_id", deviceFactory.getDeviceUuid().toString());
        } catch (Exception e) {
            EMLog.e(TAG, "uploadTokenInternal put json exception: " + e.toString());
            throw new ChatException(Error.GENERAL_ERROR, "uploadTokenInternal put json exception: " + e.getMessage());
        }
        int retry_times = 2;
        int statusCode = Error.GENERAL_ERROR;
        String content = "";
        do{
            try {
                EMLog.e(TAG, "uploadTokenInternal, token=" + deviceToken + ", url=" + remoteUrl
                        + ", notifier name=" + notifierName);
                Pair<Integer, String> response = EMHttpClient.getInstance().sendRequestWithToken(remoteUrl,
                        json.toString(), EMHttpClient.PUT);
                statusCode = response.first;
                content = response.second;
                if (statusCode == HttpURLConnection.HTTP_OK) {
                    // Save token info after requested successfully
                    EMPreferenceUtils.getInstance().setPushToken(deviceToken);
                    EMPreferenceUtils.getInstance().setPushNotifierName(notifierName);
                    EMLog.e(TAG, "uploadTokenInternal success.");
                    return;
                }
                EMLog.e(TAG, "uploadTokenInternal failed: " + content);
                remoteUrl = ChatClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + ChatClient.getInstance().getCurrentUser();
            } catch (ChatException e) {
                EMLog.e(TAG, "uploadTokenInternal failed: " + e.getDescription());
                remoteUrl = ChatClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + ChatClient.getInstance().getCurrentUser();
            }
        }while (--retry_times > 0);

        throw new ChatException(statusCode, content);
    }

    /**
     * \~english
     * Set the push template for offline push.
     * @param templateName template name.
     * @param callBack Complete the callback.
     */
    public void setPushTemplate(String templateName, CallBack callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    emaObject.setPushTemplate(templateName, error);
                    handleError(error);
                    callBack.onSuccess();
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Gets the offline push template for Settings.
     * @param callBack Complete the callback.
     */
    public void getPushTemplate(ValueCallBack<String> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    String templateName = emaObject.getPushTemplate(error);
                    handleError(error);
                    callBack.onSuccess(templateName);
                }catch (ChatException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Reports the push events.
     *
     * This is an asynchronous method.
     *
     * @param json Value corresponding to the EPush field carried in the push data.
     * @param action Push event
     * @param callBack Complete the callback.
     */
    public void reportPushAction(JSONObject json, EMPushAction action, CallBack callBack)
    {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    if(json != null){
                        json.put("action", action.name);
                        reportPushAction(json.toString());
                    }
                    callBack.onSuccess();
                } catch (IllegalArgumentException e) {
                    callBack.onError(Error.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (ChatException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        });
    }

}