/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import io.agora.CallBack;
import io.agora.Error;
import io.agora.GroupChangeListener;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMACallback;
import io.agora.chat.adapter.EMAError;
import io.agora.chat.adapter.EMAGroup;
import io.agora.chat.adapter.EMAGroup.GroupLeaveReason;
import io.agora.chat.adapter.EMAGroupManager;
import io.agora.chat.adapter.EMAGroupManagerListener;
import io.agora.chat.adapter.EMAGroupSetting;
import io.agora.chat.adapter.EMAMucShareFile;
import io.agora.exceptions.ChatException;
import io.agora.util.EMLog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;


/**
 * \~english
 * The group manager for management of group creation and deletion and member management.
 */
public class GroupManager {

    /**
	 * \~english
	 * The group styles.
     */
	public enum GroupStyle {

		/**
		 * \~english
		 * Private groups. Only the group owner can invite users to join.
		 */
		GroupStylePrivateOnlyOwnerInvite,

		/**
		 * \~english
		 * Private groups. Both the group owner and group members can invite users to join.
		 */
		GroupStylePrivateMemberCanInvite,

		/**
		 * \~english
		 * Public groups. Only the owner can invite users to join. 
		 * A user can join a group only after the owner approves the user's group request;
		 */
		GroupStylePublicJoinNeedApproval,

		/**
		 * \~english
		 * Public groups. A user can join a group without the group owner approving user's group request.
		 */
		GroupStylePublicOpenJoin
	}

	EMAGroupManager emaObject;

	private static String TAG = "group";

	List<GroupChangeListener> groupChangeListeners;

	EMAGroupManagerListener listenerImpl = new EMAGroupManagerListener() {

		@Override
		public void onReceiveInviteFromGroup(String groupId, String groupName, String inviter, String inviteMessage) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationReceived(groupId, groupName, inviter, inviteMessage);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		/**
		 * \~english
		 * Occurs when the user accepts the invitation to join the group.
		 * @param  invitee The user invited to join the group.
		 * @return NA
		 */
		@Override
		public void onReceiveInviteAcceptionFromGroup(EMAGroup group, String invitee) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationAccepted(group.groupId(), invitee, "");
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		/**
		 * \~english
		 * Occurs when the user declines to join the group.
		 * @param  invitee The user invited to join the group.
		 * @param  reason  The reason of declining.
		 * @return NA
		 */
		@Override
		public void onReceiveInviteDeclineFromGroup(EMAGroup group, String invitee, String reason) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationDeclined(group.groupId(), invitee, "");
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAutoAcceptInvitationFromGroup(EMAGroup group, String inviter, String inviteMessage) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onAutoAcceptInvitationFromGroup(group.groupId(), inviter, inviteMessage);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onLeaveGroup(EMAGroup group, int reason) {
			ChatClient.getInstance().chatManager().caches.remove(group.groupId());
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						if (reason == GroupLeaveReason.BE_KICKED.ordinal()) {
							listener.onUserRemoved(group.groupId(), group.groupSubject());
						} else {
							listener.onGroupDestroyed(group.groupId(), group.groupSubject());
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveJoinGroupApplication(EMAGroup group, String from, String message) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinReceived(group.groupId(), group.groupSubject(), from, message);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveAcceptionFromGroup(EMAGroup group) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinAccepted(group.groupId(), group.groupSubject(), group.getOwner());
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveRejectionFromGroup(String groupId, String reason) {
			Group group = GroupManager.this.getGroup(groupId);
			String groupName = group == null ? "" : group.groupSubject();
			String decliner = group == null ? "" : group.getOwner();

			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinDeclined(groupId, groupName, decliner, reason);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateMyGroupList(List<EMAGroup> groups) {

		}

		@Override
		public void onAddMutesFromGroup(EMAGroup group, final List<String> muteMembers, long muteExpire) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListAdded(group.groupId(), muteMembers, muteExpire);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveMutesFromGroup(EMAGroup group, final List<String> banPostList) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListRemoved(group.groupId(), banPostList);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

        public void onWhiteListAdded(EMAGroup group, List<String> members) {
            synchronized (groupChangeListeners) {
                try {
                    for(GroupChangeListener listener : groupChangeListeners) {
                        listener.onWhiteListAdded(group.groupId(), members);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        public void onWhiteListRemoved(EMAGroup group, List<String> members) {
            synchronized (groupChangeListeners) {
                try {
                    for(GroupChangeListener listener : groupChangeListeners) {
                        listener.onWhiteListRemoved(group.groupId(), members);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        public void onAllMemberMuteStateChanged(EMAGroup group, boolean isMuted) {
            synchronized (groupChangeListeners) {
                try {
                    for(GroupChangeListener listener : groupChangeListeners) {
                        listener.onAllMemberMuteStateChanged(group.groupId(), isMuted);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

		@Override
		public void onAddAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onAdminAdded(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onAdminRemoved(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAssignOwnerFromGroup(EMAGroup group, String newOwner, String oldOwner) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onOwnerChanged(group.groupId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onMemberJoined(EMAGroup group, String member) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onMemberJoined(group.groupId(), member);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onMemberExited(EMAGroup group, String member) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onMemberExited(group.groupId(), member);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAGroup group, String announcement) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onAnnouncementChanged(group.groupId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUploadShareFileFromGroup(EMAGroup group, EMAMucShareFile shareFile) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onSharedFileAdded(group.groupId(), new MucSharedFile(shareFile));
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onDeleteShareFileFromGroup(EMAGroup group, String fileId) {
			synchronized (groupChangeListeners) {
				try {
					for (GroupChangeListener listener : groupChangeListeners) {
						listener.onSharedFileDeleted(group.groupId(), fileId);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateSpecificationFromGroup(EMAGroup group) {
			synchronized (groupChangeListeners) {
				for (GroupChangeListener listener : groupChangeListeners) {
					try {
						if (null != group) {
							listener.onSpecificationChanged(new Group(group));
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onStateChangedFromGroup(EMAGroup group, boolean isDisabled) {
			synchronized (groupChangeListeners) {
				for (GroupChangeListener listener : groupChangeListeners) {
					try {
						listener.onStateChanged(new Group(group), isDisabled);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onGroupMemberAttributeChanged(String groupId,String userId, Map<String,String> attributeMap, String from) {
			synchronized (groupChangeListeners) {
				if (null != attributeMap && attributeMap.size() > 0){
					for (GroupChangeListener listener : groupChangeListeners) {
						try {
							listener.onGroupMemberAttributeChanged(groupId,userId,attributeMap,from);
							} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			}
		}
	};

	ChatClient mClient;
	GroupManager(ChatClient client, EMAGroupManager groupManager) {
		emaObject = groupManager;
		mClient = client;
		groupChangeListeners = Collections.synchronizedList(new ArrayList<GroupChangeListener>());
		emaObject.addListener(listenerImpl);
		ChatClient.getInstance().chatManager();
	}

	/**
	 * \~english
	 * Gets all groups of the current user (from the cache).
	 * Before a call to this method, call {@link #loadAllGroups()} to load data from the group to the cache.
	 * If {@link #loadAllGroups()} is not called, this method will load data from the database before
	 * loading from the cache.
	 *
	 * @return The group list.
	 */
	public List<Group> getAllGroups() {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		List<Group> result = new ArrayList<Group>();
		for (EMAGroup group : groups) {
			result.add(new Group(group));
		}
		return Collections.unmodifiableList(result);
	}

	/**
	 * \~english
	 * Gets the group instance from the cache by group ID.
	 *
	 * @param groupId	The group ID.
	 * @return 			The group instance. Returns null if the group does not exist.
	 */
	public Group getGroup(String groupId) {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		for (EMAGroup group : groups) {
			if (group.groupId().equals(groupId)) {
				return new Group(group);
			}
		}
		return null;
	}

	/**
	 * \~english
	 * Creates a group instance.
	 * After the group is created, the data in the cache and database will be updated and multiple devices will receive the notification event and
	 * update the group to the cache and database. 
	 * You can set {@link io.agora.MultiDeviceListener} to listen on the event. The event callback function
	 * is {@link io.agora.MultiDeviceListener#onGroupEvent(int, String, List)}, where the first parameter is the event,
	 * for example, {@link io.agora.MultiDeviceListener#GROUP_CREATE} for the group creation event.
	 * For the asynchronous method, see {@link #asyncCreateGroup(String, String, String[], String, GroupOptions, ValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *

	 * @param groupName     		The group name. It is optional. Pass null if you do not want to set this parameter.
     * @param desc          		The group description. It is optional. Pass null if you do not want to set this parameter.
	 * @param allMembers    		The group member array. The group owner ID is optional. This parameter can not be null.
	 * @param reason        		The group joining invitation. It is optional. Pass null if you do not want to set this parameter.
	 * @param option        		The options for creating a group. They are optional and can not be null. See {@link GroupOptions}.
	 *                      		The options are as follows:
	 *                      		- The maximum number of group members. The default value is 200.
	 *                      		- The group style. See {@link GroupStyle}. The default value is {@link GroupStyle#GroupStylePrivateOnlyOwnerInvite}.
	 *                      		- Whether to ask for permission when inviting a user to joing the group. The default value is false, indicating that invitees are automaticall added to the group without their permission.
	 *                      		- The group detail extensions.
	 *
	 * @return 						The created group instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group createGroup(String groupName, String desc, String[] allMembers, String reason, GroupOptions option) throws ChatException{
		int style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;

		switch(option.style) {
			case GroupStylePrivateOnlyOwnerInvite:
				style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;
				break;

			case GroupStylePrivateMemberCanInvite:
				style = EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE;
				break;

			case GroupStylePublicJoinNeedApproval:
				style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL;
				break;

			case GroupStylePublicOpenJoin:
				style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN;
				break;
		}

		return createGroup(style, groupName, desc, allMembers, option.maxUsers, reason, option.inviteNeedConfirm, option.extField);
	}

	/**
	 * \~english
	 * Creates a group instance.
     * After the group is created, the data in the cache and database will be updated and multiple devices will receive the notification event and
	 * update the group to the cache and database. You can set {@link io.agora.MultiDeviceListener} to listen on the event. The event callback function
	 * is {@link io.agora.MultiDeviceListener#onGroupEvent(int, String, List)}, where the first parameter is the event, for example,
	 * {@link io.agora.MultiDeviceListener#GROUP_CREATE} for the group creation event.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupName     The group name. It is optional. Pass null if you do not want to set this parameter.
     * @param desc          The group description. It is optional. Pass null if you do not want to set this parameter.
	 * @param allMembers    The group member array. The group owner ID is optional. This parameter can not be null.
	 * @param reason        The group joining invitation. It is optional. Pass null if you do not want to set this parameter.
	 * @param option        The options for creating a group. They are optional and can not be null. See {@link GroupOptions}.
	 *                      The options are as follows:
	 *                      - The maximum number of group members. The default value is 200.
	 *                      - The group style. See {@link GroupStyle}. The default value is {@link GroupStyle#GroupStylePrivateOnlyOwnerInvite}.
	 *                      - Whether to ask for permission when inviting a user to joining the group. The default value is false, indicating that invitees are automaticall added to the group without their permission.
	 *                      - The group detail extensions.
	 *
     * @param callback		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the created group object;
	 * 						If this call fails, calls {@link ValueCallBack#onError(int, String)}.
     */
	public void asyncCreateGroup(final String groupName,
								 final String desc,
								 final String[] allMembers,
								 final String reason,
								 final GroupOptions option,
								 final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					Group group = createGroup(groupName, desc, allMembers, reason, option);
					callback.onSuccess(group);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

    /// @cond
	// Private method.
    /**
     *  Creates a group.
     *
     *  @param style                The public or private group.
     *  @param groupName            The group name.
     *  @param desc                 The group description.
     *  @param allMembers           The group members, excluding the owner and creator.
     *  @param maxUsers             The maximum group member capacity.
     *  @param reason               The invitation message.
     *  @param inviteNeedConfirm    Whether to need invitation confirmation.
     *  @param extension            The group extension information.
     *  @return                     The created group.
     */
	 /// @endcond
    private Group createGroup(int style,
                                String groupName,
                                String desc,
                                String[] allMembers,
                                int maxUsers,
                                String reason,
                                boolean inviteNeedConfirm,
                                String extension) throws ChatException {
        EMAGroupSetting setting = new EMAGroupSetting(style, maxUsers, inviteNeedConfirm, extension);
        List<String> newMembers = new ArrayList<String>();
        Collections.addAll(newMembers, allMembers);
        EMAError error = new EMAError();
        EMAGroup group = emaObject.createGroup(groupName, desc, reason, setting, newMembers, inviteNeedConfirm, error);
        handleError(error);
        return new Group(group);
    }

	/**
	 * \~english
	 * Loads all local groups from the database.
	 */
	public synchronized void loadAllGroups() {
		emaObject.loadAllMyGroupsFromDB();
	}

	/**
	 * \~english
	 * Destroys the group instance.
	 * Only the group owner can call this method.
	 * For the asynchronous method, see {@link #asyncDestroyGroup(String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void destroyGroup(final String groupId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.destroyGroup(groupId, error);
		ChatClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}

	/**
	 * \~english
	 * Destroys the group instance.
	 * Only the group owner can call this method.
	 * For the synchronous method, see {@link #destroyGroup(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
     * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					if this call fails, calls {@link CallBack#onError(int, String)}.
     */
	public void asyncDestroyGroup(final String groupId, final CallBack callback){
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyGroup(groupId);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Adds users to the group.
	 * Only the group creator or admin can call this method.
	 * For the asynchronous method, see {@link #asyncAddUsersToGroup(String, String[], CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param newmembers			The array of new members to add.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void addUsersToGroup(String groupId, String[] newmembers) throws ChatException {
		addUsersToGroup(groupId, newmembers, "welcome");
	}

	/**
	 * \~english
	 * Adds users to the group.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncAddUsersToGroup(String, String[], CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId                The group ID.
	 * @param newmembers             The array of new members to add.
	 * @param welcomeMessage		 The welcome message.
	 * @throws ChatException    A description of the exception. See {@link Error}.
	 */
	public void addUsersToGroup(String groupId, String[] newmembers, String welcomeMessage) throws ChatException {
		EMAError error = new EMAError();
		List<String> newmembers2 = new ArrayList<String>();
		Collections.addAll(newmembers2, newmembers);
		emaObject.addGroupMembers(groupId, newmembers2, welcomeMessage, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Adds users to the group.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #addUsersToGroup(String, String[])}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId		The group ID.
	 * @param newmembers	The array of new members to add.
	 * @param callback		The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 						If this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncAddUsersToGroup(final String groupId,
									 final String[] newmembers,
									 final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					addUsersToGroup(groupId, newmembers);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes a member from the group.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncRemoveUserFromGroup(String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param username   			The user ID of the member to be removed.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void removeUserFromGroup(final String groupId, final String username) throws ChatException {
		List<String> members = new ArrayList<String>();
		EMAError error = new EMAError();
		members.add(username);
		emaObject.removeGroupMembers(groupId, members, error);
		handleError(error);
		emaObject.fetchGroupSpecification(groupId, error, true);
		handleError(error);
	}

	/**
	 * \~english
	 * Removes a member from the group.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #removeUserFromGroup(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId   The group ID.
	 * @param username  The user ID of the member to be removed.
     * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					if this call fails, calls {@link CallBack#onError(int, String)}.
     */
	public void asyncRemoveUserFromGroup(final String groupId,
										 final String username,
										 final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					removeUserFromGroup(groupId, username);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes members from the group.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #removeUsersFromGroup(String, List)}}.
	 *
	 * Only the group owner or admin can call this method.
	 *
	 * @param groupId				The group ID.
	 * @param members   			The user IDs of members to be removed.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void removeUsersFromGroup(final String groupId, final List<String> members) throws ChatException {
		EMAError error = new EMAError();
		emaObject.removeGroupMembers(groupId, members, error);
		handleError(error);
		emaObject.fetchGroupSpecification(groupId, error, true);
		handleError(error);
	}

	/**
	 * \~english
	 * Removes members from the group.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #removeUsersFromGroup(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param members   The user IDs of members to be removed.
     * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					if this call fails, calls {@link CallBack#onError(int, String)}.
     */
	public void asyncRemoveUsersFromGroup(final String groupId,
										 final List<String> members,
										 final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					removeUsersFromGroup(groupId, members);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Leaves a group.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void leaveGroup(String groupId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.leaveGroup(groupId, error);
		ChatClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}

	/**
	 * \~english
	 * Leaves a group.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
     * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					If this call fails, calls {@link CallBack#onError(int, String)}.
     */
	public void asyncLeaveGroup(final String groupId, final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					leaveGroup(groupId);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets group information from the server.
	 * This method does not get member information. If member information is required, call {@link #fetchGroupMembers(String, String, int)}.
	 * For the asynchronous method, see {@link #asyncGetGroupFromServer(String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @return Group				The group instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group getGroupFromServer(String groupId) throws ChatException {
		if(groupId == null || groupId.isEmpty())
			throw new ChatException(Error.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
	 * Gets group information from the server.
	 * If fetchMembers is set to true, a member list is also fetched. If more members need to be fetched, call {@link #fetchGroupMembers(String, String, int)}.
	 * Only members in the group can get the member list of the group.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param fetchMembers 			Whether to get group members. By default, a list of 200 members is fetched.
	 * @return Group				The updated group instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group getGroupFromServer(String groupId, boolean fetchMembers) throws ChatException {
		if(groupId == null || groupId.isEmpty())
			throw new ChatException(Error.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, fetchMembers);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
	 * Gets group information from the server.
	 * For the synchronous method, see {@link #getGroupFromServer(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
     * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					if this call fails, calls {@link CallBack#onError(int, String)}.
     */
	public void asyncGetGroupFromServer(final String groupId,
										final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					Group group = getGroupFromServer(groupId);
					callback.onSuccess(group);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets all groups from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * @return The group list. 
	 * @throws ChatException A description of the exception. See {@link Error}.
	 */
	synchronized List<Group> getGroupsFromServer() throws ChatException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroups(error);
		handleError(error);
		List<Group> result = new ArrayList<Group>();
		for (EMAGroup group : groups) {
			result.add(new Group(group));
		}
		return result;
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server with pagination。
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId).}. 
	 *
	 * @param pageIndex  The page number, starting from 1.
	 * @param pageSize   The number of groups per page.
	 * @return           The group list.  
	 * @throws ChatException A description of the exception. See {@link Error}.
	 */
	synchronized List<Group> getGroupsFromServer(int pageIndex, int pageSize) throws ChatException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroupsWithPage(pageIndex, pageSize, error);
		handleError(error);
		List<Group> result = new ArrayList<Group>();
		for (EMAGroup group : groups) {
			result.add(new Group(group));
		}
		return result;
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server with pagination。
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId).}.
	 *
	 * @param pageIndex  The page number, starting from 0.
	 * @param pageSize   The number of groups per page. pageSize max 20
	 * @param needMemberCount  Whether the number of group members is required.
	 * @param needRole  Whether the user's role in the group is required.
	 * @return           The group list.
	 * @throws ChatException A description of the exception. See {@link Error}.
	 */
	synchronized List<Group> getGroupsFromServer(int pageIndex, int pageSize, boolean needMemberCount, boolean needRole) throws ChatException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroupsWithPage(pageIndex, pageSize, needMemberCount, needRole, error);
		handleError(error);
		List<Group> result = new ArrayList<Group>();
		for (EMAGroup group : groups) {
			result.add(new Group(group));
		}
		return result;
	}


	/**
	 * \~english
	 * Gets all groups of the current user from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 *
	 * @param callback  Contains the group list.
	 */
	void asyncGetGroupsFromServer(final ValueCallBack<List<Group>> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<Group> groups = getGroupsFromServer();
					callback.onSuccess(groups);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the asynchronous method, see {@link #asyncGetJoinedGroupsFromServer(ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @return 						The list of groups that the current user joins.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public synchronized List<Group> getJoinedGroupsFromServer() throws ChatException{
		return getGroupsFromServer();
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server with pagination.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the asynchronous method, see {@link #asyncGetJoinedGroupsFromServer(int, int, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageIndex 			The page number, starting from 1.
	 * @param pageSize				The number of groups per page.
	 * @return 						The group list on the next page.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 *
	 * @deprecated Deprecated. Please use {@link #getJoinedGroupsFromServer(int, int, boolean, boolean)} instead.
	 */
	@Deprecated
	public synchronized List<Group> getJoinedGroupsFromServer(int pageIndex, int pageSize) throws ChatException{
		return getGroupsFromServer(pageIndex, pageSize);
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server with pagination.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the asynchronous method, see {@link #asyncGetJoinedGroupsFromServer(int, int, boolean, boolean, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageIndex 			The page number, starting from 0.
	 * @param pageSize				The number of groups per page. pageSize max 20
	 * @param needMemberCount		Whether the number of group members is required.
	 * @param needRole				Whether the user's role in the group is required.
	 * @return 						The group list on the next page.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public synchronized List<Group> getJoinedGroupsFromServer(int pageIndex, int pageSize, boolean needMemberCount, boolean needRole) throws ChatException{
		return getGroupsFromServer(pageIndex, pageSize , needMemberCount, needRole);
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For a synchronous method, see {@link #getJoinedGroupsFromServer()}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param callback 	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and 
	 *                  returns the list of groups that the user has joined;
	 * 					if the call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncGetJoinedGroupsFromServer(final ValueCallBack<List<Group>> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<Group> groups = getJoinedGroupsFromServer();
					callback.onSuccess(groups);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server with pagination.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the synchronous method, see {@link #getJoinedGroupsFromServer(int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param pageIndex	The page number, starting from 1.
	 * @param pageSize	The number of groups per page.
     * @param callback 	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the group list on the next page;
	 *                  if this call fails, calls {@link ValueCallBack#onError(int, String)};
	 *
	 * @deprecated Deprecated. Please use {@link #asyncGetJoinedGroupsFromServer(int, int, boolean, boolean, ValueCallBack)} instead.
	 */
	@Deprecated
	public void asyncGetJoinedGroupsFromServer(final int pageIndex,
											   final int pageSize,
											   final ValueCallBack<List<Group>> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<Group> groups = getJoinedGroupsFromServer(pageIndex, pageSize);
					callback.onSuccess(groups);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets all groups of the current user from the server with pagination.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the synchronous method, see {@link #getJoinedGroupsFromServer(int, int, boolean, boolean)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param pageIndex			The page number, starting from 0.
	 * @param pageSize			The number of groups per page.pageSize max 20
	 * @param needMemberCount  Whether the number of group members is required.
	 * @param needRole  		Whether the user's role in the group is required.
	 * @param callback 			The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the group list on the next page;
	 * 							if this call fails, calls {@link ValueCallBack#onError(int, String)};
	 *
	 */
	public void asyncGetJoinedGroupsFromServer(final int pageIndex,
											   final int pageSize,
											   boolean needMemberCount,
											   boolean needRole,
											   final ValueCallBack<List<Group>> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<Group> groups = getJoinedGroupsFromServer(pageIndex, pageSize, needMemberCount, needRole);
					callback.onSuccess(groups);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}


	/**
	 * \~english
	 * Gets public groups from the server with pagination.
	 * For the asynchronous method, see {@link #asyncGetPublicGroupsFromServer(int, String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageSize  			The number of public groups per page.
	 * @param cursor    			The cursor position from which to start to get data next time. Sets the parameter as null for the first time.
	 * @return  		    		The result of {@link CursorResult}, including the cursor for getting data next time and the group list.
	 * 								For the last page, the return value of cursor is an empty string.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public CursorResult<GroupInfo> getPublicGroupsFromServer(int pageSize, String cursor) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<GroupInfo> result = emaObject.fetchPublicGroupsWithCursor(cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~english
	 * Gets public groups from the server with pagination.
	 * For the synchronous method, see {@link #getPublicGroupsFromServer(int, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param pageSize  	The number of public groups per page.
	 * @param cursor    	The cursor position from which to start getting data next time. Sets the parameter as null for the first time.
	 * @param callback      The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} 
	 *                      and returns the result of {@link CursorResult}), including the cursor for getting data next time and the group list.
	 * 						For the last page, the return value of cursor is an empty string.
	 * 						If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncGetPublicGroupsFromServer(final int pageSize,
											   final String cursor,
											   final ValueCallBack<CursorResult<GroupInfo>> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					CursorResult<GroupInfo> result = getPublicGroupsFromServer(pageSize, cursor);
					callback.onSuccess(result);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Joins a public group.
	 * For a group that requires no authentication，users can join it freely without the need of having permission. 
	 * For a group that requires authentication, users need to wait for the owner to agree before joing the group. For details, see {@link GroupStyle}.
	 * For the asynchronous method, see {@link #asyncJoinGroup(String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void joinGroup(String groupId) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup _group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		if (_group.groupSetting() == null) {
			throw new ChatException();
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN) {
			emaObject.joinPublicGroup(groupId, error);
			handleError(error);
			return;
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL) {
			emaObject.applyJoinPublicGroup(groupId, mClient.getCurrentUser(), "hello", error);
			handleError(error);
			return;
		}
	}

	/**
	 * \~english
	 * Joins a public group.
	 * For a group that requires no authentication，users can join it freely without the need of having permission.
	 * For a group that requires authentication, users need to wait for the owner to agree before joing the group. For details, see {@link GroupStyle}.
	 * For the synchronous method, see {@link #joinGroup(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncJoinGroup(final String groupId,
							   final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					joinGroup(groupId);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Changes the group name.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncChangeGroupName(String, String, CallBack)}.
	 * 
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The ID of group whose name is to be changed.
	 * @param changedGroupName 		The new group name.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void changeGroupName(String groupId, String changedGroupName) throws ChatException{
		EMAError error = new EMAError();
		emaObject.changeGroupSubject(groupId, changedGroupName, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Changes the group name.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #changeGroupName(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 			The ID of group whose name is to be changed.
	 * @param changedGroupName 	The new group name.
	 * @param callback			The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 							if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncChangeGroupName(final String groupId,
									 final String changedGroupName,
									 final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					changeGroupName(groupId, changedGroupName);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Changes the group description.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncChangeGroupDescription(String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param changedDescription 	The new group description.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void changeGroupDescription(String groupId, String changedDescription)
			throws ChatException {
		EMAError error = new EMAError();
		emaObject.changeGroupDescription(groupId, changedDescription, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Changes the group description.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #changeGroupDescription(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 				The group ID.
	 * @param changedDescription 	The new group description.
     * @param callBack				The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *								if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncChangeGroupDescription(final String groupId,
											final String changedDescription,
											final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override public void run() {
				try {
					changeGroupDescription(groupId, changedDescription);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Accepts a group invitation.
	 * For the asynchronous method, see {@link #asyncAcceptInvitation(String, String, ValueCallBack)}。
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param  groupId 				The group ID.
	 * @param  inviter 				The user who initiates the invitation.
	 * @return 						The group instance which the user has accepted the invitation to join.
	 * @throws ChatException	A description of the exception. see {@link Error}.
	 */
	public Group acceptInvitation(String groupId, String inviter) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.acceptInvitationFromGroup(groupId, inviter == null ? "" : inviter, error);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
	 * Accepts a group invitation.
	 * For the synchronous method, see {@link #acceptInvitation(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param inviter 	The inviter ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)}
	 *                  and returns the updated group instance;
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncAcceptInvitation(final String groupId,
									  final String inviter,
									  final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					Group group = acceptInvitation(groupId, inviter);
					callback.onSuccess(group);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Declines a group invitation.
	 * For the asynchronous method, see {@link #asyncDeclineInvitation(String, String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param inviter 				The inviter.
	 * @param reason				The reason for declining.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void declineInvitation(
			String groupId,
			String inviter,
			String reason) throws ChatException {
		EMAError error = new EMAError();
		emaObject.declineInvitationFromGroup(groupId, inviter == null ? "" : inviter, reason == null ? "" : reason, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Declines a group invitation.
	 * For the synchronous method, see {@link #declineInvitation(String, String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param inviter 	The inviter ID.
	 * @param reason	The reason for declining.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncDeclineInvitation(final String groupId,
									   final String inviter,
									   final String reason,
									   final CallBack callback) {

		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					declineInvitation(groupId, inviter, reason);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Approves a group request.
     * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncAcceptApplication(String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param username 				The ID of the user who sends a request to join the group.
	 * @param groupId  				The group ID.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void acceptApplication(String username, String groupId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.acceptJoinGroupApplication(groupId, username, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Approves a group request.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #acceptApplication(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param username 	The ID of the user who sends the request to join the group.
	 * @param groupId  	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)} if it does not.
	 */
	public void asyncAcceptApplication(final String username,
									   final String groupId,
									   final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					acceptApplication(username, groupId);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Declines a group request.
     * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncDeclineApplication(String, String, String, CallBack)}.
	 *
	 * Only the group owner or admin can call this method.
	 *
	 * @param username 				The ID of the user who sends the request for join the group.
	 * @param groupId  				The group ID.
	 * @param reason   				The reason of declining.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void declineApplication(String username, String groupId, String reason) throws ChatException {
		EMAError error = new EMAError();
		emaObject.declineJoinGroupApplication(groupId, username, reason, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Declines a group request.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #declineApplication(String, String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param username 	The ID of the user who sends the request for join the group.
	 * @param groupId  	The group ID.
	 * @param reason   	The reason of declining.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncDeclineApplication(final String username,
                                        final String groupId,
                                        final String reason,
                                        final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					declineApplication(username, groupId, reason);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Invites users to join a group.
	 * Note: The group style is {@link GroupStyle#GroupStylePrivateMemberCanInvite}, which allows group
	 * members to invite users.
	 * For the asynchronous method, see {@link #asyncInviteUser(String, String[], String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param beInvitedUsernames   	The array of users to be invited.
	 * @param reason               	The reason for invitation.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void inviteUser(String groupId, String[] beInvitedUsernames, String reason) throws ChatException {
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		Collections.addAll(members, beInvitedUsernames);
		emaObject.addGroupMembers(groupId, members, reason, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Invites users to join a group.
	 * Note: The group style is {@link GroupStyle#GroupStylePrivateMemberCanInvite}, which allows group
	 * members to invite users.
	 * For the synchronous method, see {@link #inviteUser(String, String[], String)}。
	 *
	 * The is an asynchronous method.
     *
	 * @param groupId				The group ID.
	 * @param beInvitedUsernames   	The array of users to be invited.
	 * @param reason               	The reason for invitation.
	 * @param callback				The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *								if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncInviteUser(final String groupId,
                                final String[] beInvitedUsernames,
                                final String reason,
                                final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					inviteUser(groupId, beInvitedUsernames, reason);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Requests to join a group.
     * Note: The group style is {@link GroupStyle#GroupStylePublicJoinNeedApproval}, which is a public group 
	 * requiring authentication.
	 * For an asynchronous method, see {@link #asyncApplyJoinToGroup(String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param reason    			The reason for requesting to join the group.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void applyJoinToGroup(String groupId, String reason) throws ChatException {
		String userName = mClient.getCurrentUser();
		EMAError error = new EMAError();
		emaObject.applyJoinPublicGroup(groupId, userName, reason, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Requests to join a group.
     * Note: The group style is {@link GroupStyle#GroupStylePublicJoinNeedApproval}, which is a public group requiring authentication.
	 * For the synchronous method, see {@link #applyJoinToGroup(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param reason    The reason for requesting to joining the group.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncApplyJoinToGroup(final String groupId,
                                      final String reason,
                                      final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					applyJoinToGroup(groupId, reason);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 *\~english
	 * Blocks group messages.
     * The user that blocks group messages is still a group member, but can't receive group messages.
	 * For the asynchronous method, see {@link #asyncBlockGroupMessage(String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void blockGroupMessage(String groupId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.blockGroupMessage(groupId, error);
		handleError(error);
	}


	/**
	 *\~english
	 * Blocks group messages.
     * The user that blocks group messages is still a group member, but can't receive group messages.
	 * For the synchronous method, see {@link #blockGroupMessage(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncBlockGroupMessage(final String groupId, final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockGroupMessage(groupId);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Unblocks group messages.
	 * For the asynchronous method, see {@link #asyncUnblockGroupMessage(String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void unblockGroupMessage(String groupId) throws ChatException {
		EMLog.d(TAG, "try to unblock group msg:" + groupId);
		EMAError error = new EMAError();
		emaObject.unblockGroupMessage(groupId, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Unblocks group messages.
     * For the synchronous method, see {@link #unblockGroupMessage(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncUnblockGroupMessage(final String groupId,
                                         final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockGroupMessage(groupId);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 *\~english
	 * Adds the user to the group blocklist. 
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again. 
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncBlockUser(String, String, CallBack)}.
	 *
	 *
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param username 				The user to be added to the blocklist.
     * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void blockUser(String groupId, String username) throws ChatException {
		EMLog.d(TAG, "block user for groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		String reason = "";
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.blockGroupMembers(groupId, members, error, reason);
		handleError(error);
	}

	/**
	 *\~english
	 * Adds the user to the group blocklist.
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again. 
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #blockUser(String, String)}.
     *
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param username 	The user to be added to the blocklist.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					If this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncBlockUser(final String groupId,
                               final String username,
                               final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockUser(groupId, username);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
     * Adds the user to the group blocklist.
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncBlockUsers(String, List, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param members 				The list of users to be added to the blocklist.
     * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void blockUsers(String groupId, List<String> members) throws ChatException {
		EMLog.d(TAG, "block user for groupid:" + groupId + " members:" + members == null ? "":members.toString());
		EMAError error = new EMAError();
		String reason = "";
		emaObject.blockGroupMembers(groupId, members, error, reason);
		handleError(error);
	}

	/**
	 * \~english
     * Adds the user to the group blocklist.
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #blockUsers(String, List)}.
	 *
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param members 	The list of users to be blocked.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncBlockUsers(final String groupId,
                               final List<String> members,
                               final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockUsers(groupId, members);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes a user from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUnblockUser(String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param username				The user to be removed from the group blocklist.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void unblockUser(String groupId, String username) throws ChatException{
		EMLog.d(TAG, "unblock user groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.unblockGroupMembers(groupId, members, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Removes a user from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #unblockUser(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param username	The user to be removed from the group blocklist.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncUnblockUser(final String groupId,
                                 final String username,
                                 final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockUser(groupId, username);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes users from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncBlockUsers(String, List, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param members				The users to be removed from the group blocklist.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void unblockUsers(String groupId, List<String> members) throws ChatException{
		EMLog.d(TAG, "unblock user groupid:" + groupId + " members:" + members == null ? "":members.toString());
		EMAError error = new EMAError();
		emaObject.unblockGroupMembers(groupId, members, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Removes users from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #unblockUsers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param members	The users to be removed from the bloclist.
	 * @param callback	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncUnblockUsers(final String groupId,
                                 final List<String> members,
                                 final CallBack callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockUsers(groupId, members);
					callback.onSuccess();
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the group blocklist that can contain a maximum of 200 users by default.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncGetBlockedUsers(String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @return List					The group blocklist.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public List<String> getBlockedUsers(String groupId) throws ChatException {
		return getBlockedUsers(groupId, 0, 200);
	}

	/**
	 * \~english
	 * Gets the group blocklist from server with pagination.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncGetBlockedUsers(String, int, int, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId               The group ID.
	 * @param pageIndex            	The page number, starting from 1.
	 * @param pageSize              The number of groups per page.
	 * @return List					The group blocklist.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public List<String> getBlockedUsers(String groupId, int pageIndex, int pageSize) throws ChatException{
		EMLog.d(TAG, "get blocked users of the group: " + groupId);
		EMAError error = new EMAError();
		List<String> members = emaObject.fetchGroupBlackList(groupId, pageIndex, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~english
	 * Gets the group blocklist that can contain a maximum of 200 users by default.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #getBlockedUsers(String)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                  if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 *					
	 */
	public void asyncGetBlockedUsers(final String groupId,
                                     final ValueCallBack<List<String>> callback) {
		asyncGetBlockedUsers(groupId, 0, 200, callback);
	}

	/**
	 * \~english
	 * Gets the group blocklist with pagination.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #getBlockedUsers(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId     The group ID.
	 * @param pageIndex   The page number, starting from 1.
	 * @param pageSize    The number of groups per page.
	 * @param callback	  The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the group blocklist;
	 *                    if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 *					  
	 */
	public void asyncGetBlockedUsers(final String groupId,
                                     final int pageIndex,
                                     final int pageSize,
                                     final ValueCallBack<List<String>> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<String> users = getBlockedUsers(groupId, pageIndex, pageSize);
					callback.onSuccess(users);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Registers a group change listener.
	 * The registered listener needs to be used together with {@link #removeGroupChangeListener(GroupChangeListener)}.
	 *
	 * @param listener	The group event listener to be registered.
	 */
	public void addGroupChangeListener(GroupChangeListener listener) {
		EMLog.d(TAG, "add group change listener:" + listener.getClass().getName());
		if (!groupChangeListeners.contains(listener)) {
			groupChangeListeners.add(listener);
		}
	}

	/**
	 * \~english
	 * Removes a group change listener.
     * This method removes a group change listener registered with {@link #addGroupChangeListener(GroupChangeListener)}.
	 *
	 * @param listener 	The group event listener to be removed.
	 */
	public void removeGroupChangeListener(GroupChangeListener listener) {
		if (listener != null) {
			EMLog.d(TAG, "remove group change listener:" + listener.getClass().getName());
			groupChangeListeners.remove(listener);
		}
	}


	// ============================= group_reform new add api begin =============================
	/**
	 * \~english
     * Gets a group's member list with pagination.
	 * When CursorResult.getCursor() is an empty string ("") in the result, there is no more data.
     * For the asynchronous method, see {@link #asyncFetchGroupMembers(String, String, int, ValueCallBack)}.
	 *
     * Synchronization method will block the current thread.
	 * 
	 * For example:
	 * ```java
	 *     CursorResult result = fetchGroupMembers(groupId, cursor, pageSize);    // search 1
	 * 	   result = fetchGroupMembers(groupId, result.getCursor(), pageSize);       // search 2
	 * ```
     *
	 * @param groupId				The group ID.
	 * @param cursor				The cursor position from which to start to get data next time. Sets the parameter as null for the first time.
	 * @param pageSize				The number of group members per page.
	 * @return 						The result of {@link CursorResult}, 
	 *                              including the cursor for getting data next time and the group member list.
	 *                              For the last page, the return value of cursor is an empty string.
	 * @throws ChatException	A description of the exception. See {@link Error}
	 */
	public CursorResult<String> fetchGroupMembers(String groupId, String cursor, int pageSize) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<String> result = emaObject.fetchGroupMembers(groupId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~english
	 * Gets a group's member list with pagination.
	 * For the synchronous method, see {@link #fetchGroupMembers(String, String, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param cursor	The number of group members per page.
	 * @param pageSize	The cursor position from which to start to get data next time. Sets the parameter as null for the first time.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and 
	 *                  returns the result of {@link CursorResult}, including the cursor for getting data next time and the group member list.
	 *               	For the last page, the return value of cursor is an empty string.
	 *					If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupMembers(final String groupId,
                                       final String cursor,
                                       final int pageSize,
                                       final ValueCallBack<CursorResult<String>> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					CursorResult<String> result = fetchGroupMembers(groupId, cursor, pageSize);
					callback.onSuccess(result);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
     * Transfers the group ownership.
	 * Only the group owner can call this method.
	 * For the asynchronous method, see {@link #asyncChangeOwner(String, String, ValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param newOwner				The new owner ID.
	 * @return						The updated group instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group changeOwner(String groupId, String newOwner) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.transferGroupOwner(groupId, newOwner, error);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
	 * Transfers the group ownership.
	 * Only the group owner can call this method.
	 * For the synchronous method, see {@link #changeOwner(String, String)}。
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param newOwner	The new owner ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeOwner(final String groupId,
                                 final String newOwner,
                                 final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					Group group = changeOwner(groupId, newOwner);
					callback.onSuccess(group);
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Adds a group admin.
	 * Only the group owner can call this method and admin can not.
	 * For the asynchronous method, see {@link #asyncAddGroupAdmin(String, String, ValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param admin					The admin ID to add.
	 * @return						The updated group instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group addGroupAdmin(final String groupId, final String admin) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.addGroupAdmin(groupId, admin, error);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
     * Adds a group admin.
	 * Only the group owner can call this method and admin can not.
	 * For the synchronous method, see {@link #addGroupAdmin(String, String)}
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param admin		The new owner ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and 
	 *                  returns the updated group instance;
	 *					If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncAddGroupAdmin(final String groupId,
                                   final String admin,
                                   final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(addGroupAdmin(groupId, admin));
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes a group admin.
	 * Only the group owner can call this method.
	 * For the asynchronous method, see {@link #asyncRemoveGroupAdmin(String, String, ValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param admin					The admin ID to remove.
	 * @return						The updated group instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group removeGroupAdmin(String groupId, String admin) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.removeGroupAdmin(groupId, admin, error);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
	 * Removes a group admin.
	 * Only the group owner can call this method.
	 * For the synchronous method, see {@link #removeGroupAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param admin		The admin ID to remove.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                  returns the updated group instance;
	 *					if this call succeeds, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveGroupAdmin(final String groupId,
                                      final String admin,
                                      final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(removeGroupAdmin(groupId, admin));
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
     * Mutes group members.
     * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncMuteGroupMembers(String, List, long, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param muteMembers 			The list of members to be muted.
	 * @param duration 				The mute duration in milliseconds. The value `-1` indicates that the members are muted permanently.
	 * @return						The updated group instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group muteGroupMembers(String groupId, List<String> muteMembers, long duration) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.muteGroupMembers(groupId, muteMembers, duration, error);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
     * Mutes group members.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #muteGroupMembers(String, List, long)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId		The group ID.
	 * @param muteMembers 	The list of members to be muted.
	 * @param duration 		The mute duration in milliseconds. The value `-1` indicates that the members are muted permanently.
	 * @param callback		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and 
	 *                      returns the updated group instance;
	 *						If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncMuteGroupMembers(final String groupId,
                                      final List<String> muteMembers,
                                      final long duration, final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(muteGroupMembers(groupId, muteMembers, duration));
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Unmutes group members.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUnMuteGroupMembers(String, List, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
     * @param groupId				The group ID.
     * @param members				The list of members to be muted.
     * @return 						The updated group instance.
     * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group unMuteGroupMembers(String groupId, List<String> members) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.unMuteGroupMembers(groupId, members, error);
		handleError(error);
		return new Group(group);
	}

	/**
	 * \~english
	 * Unmutes group members.
	 * Only the group owner or admin can call this method.
	 * For the synchronious method, see {@link #unMuteGroupMembers(String, List)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param members	The list of members to be unmuted.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and 
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncUnMuteGroupMembers(final String groupId,
                                        final List<String> members,
                                        final ValueCallBack<Group> callback) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(unMuteGroupMembers(groupId, members));
				} catch (ChatException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the mutelist of the group from the server.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncFetchGroupMuteList(String, int, int, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The cursor position from which to start to get data next time.
	 * @return 						A map which contains the muted member ID and mute duration, where the key of each entry
	 * 								is the ID of a muted member and the value is the mute duration in milliseconds.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Map<String, Long> fetchGroupMuteList(String groupId, int pageNum, int pageSize) throws ChatException {
		EMAError error = new EMAError();
		Map<String, Long> muteMembers = emaObject.fetchGroupMutes(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~english
	 * Gets the mutelist of the group from the server.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #fetchGroupMuteList(String, int, int)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param pageNum	The page number, starting from 1.
	 * @param pageSize	The number of muted members per page.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                  The map which contains the muted member ID and mute duration, where the key of each entry
	 * 					is the ID of a muted member and the value is the mute duration in milliseconds.
	 *                  If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 *
	 */
	public void asyncFetchGroupMuteList(final String groupId,
                                        final int pageNum,
                                        final int pageSize,
                                        final ValueCallBack<Map<String, Long>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupMuteList(groupId, pageNum, pageSize));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Checks whether the current member is muted or not.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	    The group ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} to show whether the member is be muted or not;
	 * 						if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	void asyncCheckIfInMuteList(final String groupId, final ValueCallBack<Boolean> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfBeMuted(groupId, error);
					handleError(error);
					callBack.onSuccess(re);
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}


	/**
	 * \~english
	 * Gets the blocklist of group from the server with pagination.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncFetchGroupBlackList(String, int, int, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId       		The group ID.
	 * @param pageNum       		The page number, starting from 1.
	 * @param pageSize      		The number of users on the blocklist per page.
	 * @return						The group blocklist on the next page.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public List<String> fetchGroupBlackList(String groupId, int pageNum, int pageSize) throws ChatException {
		EMAError error = new EMAError();
		List<String> muteMembers = emaObject.fetchGroupBlackList(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~english
	 * Gets the blocklist of group from the server with pagination.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #fetchGroupBlackList(String, int, int)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId       The group ID.
	 * @param pageNum       The page number, starting from 1.
	 * @param pageSize      The number of users on the blocklist per page.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                      returns the group blocklist on the next page;
	 *						if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupBlackList(final String groupId,
                                         final int pageNum,
                                         final int pageSize,
                                         final ValueCallBack<List<String>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupBlackList(groupId, pageNum, pageSize));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Adds members to the allowlist.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param members 	The members to be added to the allowlist.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void addToGroupWhiteList(final String groupId, final List<String> members, final CallBack callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					emaObject.addToWhiteList(groupId, members, error);
					handleError(error);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes members from the allowlist.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param members 	The members to be removed from the allowlist.
	 * @param callBack  The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void removeFromGroupWhiteList(final String groupId, final List<String> members, final CallBack callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					emaObject.removeFromWhiteList(groupId, members, error);
					handleError(error);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets whether the member is on the allowlist.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                  returns a Boolean value to indicate whether the current user is on the group blocklist;
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void checkIfInGroupWhiteList(final String groupId, ValueCallBack<Boolean> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(groupId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~english
	 * Gets the allowlist of group from the server.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                  return the group blocklist;
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void fetchGroupWhiteList(final String groupId, final ValueCallBack<List<String>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchGroupWhiteList(groupId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}


    /**
     * \~english
     * Mutes all members.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
     * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and 
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
     */
    public void muteAllMembers(final String groupId, final ValueCallBack<Group> callBack) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMAGroup group = emaObject.muteAllMembers(groupId, error);
                    handleError(error);
                    callBack.onSuccess(new Group(group));
                } catch (ChatException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Unmutes all members.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
     * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
     */
    public void unmuteAllMembers(final String groupId, final ValueCallBack<Group> callBack) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMAGroup group = emaObject.unmuteAllMembers(groupId, error);
                    handleError(error);
                    callBack.onSuccess(new Group(group));
                } catch (ChatException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~english
	 * Updates the group announcement.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUpdateGroupAnnouncement(String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param announcement 			The group announcement.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void updateGroupAnnouncement(String groupId, String announcement) throws ChatException {
		EMAError error = new EMAError();
		emaObject.updateGroupAnnouncement(groupId, announcement, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Updates the group announcement.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #updateGroupAnnouncement(String, String)}
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 		The group ID.
	 * @param announcement 	The group announcement.
	 * @param callBack 		The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 						if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncUpdateGroupAnnouncement(final String groupId,
                                             final String announcement,
                                             final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateGroupAnnouncement(groupId, announcement);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the group announcement from the server.
	 * Group members can call this method.
	 * For the asynchronous method, see {@link #asyncFetchGroupAnnouncement(String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @return						The group announcement.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public String fetchGroupAnnouncement(String groupId) throws ChatException {
		EMAError error = new EMAError();
		String result = emaObject.fetchGroupAnnouncement(groupId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~english
	 * Gets the group announcement from the server.
	 * Group members can call this method.
	 * For the synchronous method, see {@link #fetchGroupAnnouncement(String)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupAnnouncement(final String groupId,
                                            final ValueCallBack<String> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupAnnouncement(groupId));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Uploads the shared file to the group.
	 * Note: The callback is only used for progress callback.
	 * For the asynchronous method, see {@link #asyncUploadGroupSharedFile(String, String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param filePath 				The local file path.
	 * @param callBack 				The file upload progress callback.
	 * @return						The attribute object of the shared file.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public MucSharedFile uploadGroupSharedFile(String groupId, String filePath, CallBack callBack) throws ChatException {
		EMAError error = new EMAError();
		EMACallback aCallback = new EMACallback(callBack);
		EMAMucShareFile aFile = emaObject.uploadGroupShareFile(groupId, filePath, aCallback, error);
		handleError(error);
		callBack.onSuccess();
		return new MucSharedFile(aFile);
	}

	/**
	 * \~english
	 * Uploads the shared file to the group.
	 * 
	 * For the synchronous method, see {@link #uploadGroupSharedFile(String, String, CallBack)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param filePath 	The local file path.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 *                  Calls {@link CallBack#onSuccess()} to view the upload progress.
	 * @Deprecated  Deprecated. Please use {@link #asyncUploadGroupSharedFile(String, String, ValueCallBack)} instead.
	 */
	@Deprecated
	public void asyncUploadGroupSharedFile(final String groupId,
										   final String filePath,
										   final CallBack callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					MucSharedFile file = uploadGroupSharedFile(groupId, filePath, callBack);
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Uploads the shared file to the group.
	 * 
     * For the synchronous method, see {@link #uploadGroupSharedFile(String, String, CallBack)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param filePath 	The local file path.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *					if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 *                  Calls {@link ValueCallBack#onSuccess(Object)} to view the upload progress.
	 */
	public void asyncUploadGroupSharedFile(final String groupId,
                                           final String filePath,
                                           final ValueCallBack<MucSharedFile> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					MucSharedFile aShareFile = uploadGroupSharedFile(groupId, filePath, new CallBack() {
						@Override
						public void onSuccess() {

						}

						@Override
						public void onError(int code, String error) {

						}

						@Override
						public void onProgress(int progress, String status) {
							callBack.onProgress(progress,status);
						}
					});
					if (aShareFile != null){
						callBack.onSuccess(aShareFile);
					}
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(),e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the shared files of group from the server.
     * For the asynchronous method, see {@link #asyncFetchGroupSharedFileList(String, int, int, ValueCallBack)}.
	 *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId 				The group ID.
	 * @param pageNum 				The page number, starting from 1.
	 * @param pageSize 				The number of shared files per page.
	 * @return						The shared files.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public List<MucSharedFile> fetchGroupSharedFileList(String groupId, int pageNum, int pageSize) throws ChatException {
		EMAError error = new EMAError();
		List<EMAMucShareFile> files = emaObject.fetchGroupShareFiles(groupId, pageNum, pageSize, error);
		handleError(error);
		List<MucSharedFile> result = new ArrayList<MucSharedFile>();
		for(EMAMucShareFile file : files){
			result.add(new MucSharedFile(file));
		}
		return result;
	}

	/**
	 * \~english
	 * Gets the shared file list from the server.
     * For the synchronous method, see {@link #fetchGroupSharedFileList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param pageNum 	The page number, starting from 1.
	 * @param pageSize 	The number of shared files per page.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and
	 *                  returns the shared files; if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 *
	 *  Note the following:
     * - pageSize specifies the number of shared files expected to return for this call. 
	 *   For the last page, the actual number of returned shared files is less than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data as specified by pageSize and pageNum.
				
	 */
	public void asyncFetchGroupSharedFileList(final String groupId,
                                              final int pageNum,
                                              final int pageSize,
                                              final ValueCallBack<List<MucSharedFile>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupSharedFileList(groupId, pageNum, pageSize));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes a shared file of the group.
     * Group members can delete their own uploaded files. The group owner or admin can delete all shared files.
	 * For the asynchronous method, see {@link #asyncDeleteGroupSharedFile(String, String, CallBack)}.
	 *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId 				The group ID.
	 * @param fileId 				The shared file ID.
	 * @throws ChatException	Only the group owner or admin can call this method. See {@link Error}.
	 */
	public void deleteGroupSharedFile(String groupId, String fileId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.deleteGroupShareFile(groupId, fileId, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Removes the shared file of the group.
     * Group members can delete their own uploaded files. The group owner or admin can delete all shared files.
	 * For the synchronous method, see {@link #deleteGroupSharedFile(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param fileId 	The shared file ID.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncDeleteGroupSharedFile(final String groupId,
                                           final String fileId,
                                           final CallBack callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					deleteGroupSharedFile(groupId, fileId);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Downloads the shared file of the group.
	 * Note: The callback is only used for progress callback.
	 * For the asynchronous method, see {@link #asyncDownloadGroupSharedFile(String, String, String, CallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId 				The group ID.
	 * @param fileId 				The ID of the shared file.
	 * @param savePath 				The local file path.
	 * @param callBack 				The file download progress callback.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void downloadGroupSharedFile(String groupId,
                                        String fileId,
                                        String savePath,
                                        CallBack callBack) throws ChatException{
		EMAError error = new EMAError();
		EMACallback aCallback = new EMACallback(callBack);
		emaObject.downloadGroupShareFile(groupId, fileId, savePath, aCallback, error);
		handleError(error);
		callBack.onSuccess();
	}

	/**
	 * \~english
	 * Downloads the shared file of the group.
     * For synchronous method, see {@link #downloadGroupSharedFile(String, String, String, CallBack)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId   The group ID.
	 * @param fileId    The shared file ID.
	 * @param savePath  The file path.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *					if this call fails, calls {@link CallBack#onError(int, String)}. 
	 *	                Calls {@link CallBack#onProgress(int, String)} to view the file download progress.
	 */
	public void asyncDownloadGroupSharedFile(final String groupId,
											 final String fileId,
											 final String savePath,
											 final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					downloadGroupSharedFile(groupId, fileId, savePath, callBack);
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(),e.getDescription());
				}
			}
		});
	}

	protected void setMemberAttributes(String groupId,String userId,Map<String,String> attributeMap) throws ChatException {
		EMAError error = new EMAError();
		emaObject.setMemberAttributes(groupId, userId, attributeMap, error);
		handleError(error);
	}

	/**
	 * 设置单个群成员的自定义属性。
	 * 
	 * @param groupId			群组 ID。
	 * @param userId			要设置自定义属性的群成员的用户 ID。
	 * @param attributeMap		要设置的群成员自定义属性的 map，为 key-value 格式。对于一个 key-value 键值对，若 value 设置空字符串即删除该自定义属性。
	 * @param callBack			结果回调，成功时回调 {@link CallBack#onSuccess()}，
	 * 							失败时回调 {@link CallBack#onError(int, String)}。
	 *
	 * Sets custom attributes of a group member.
	 * 
	 * @param groupId			The group ID.
	 * @param userId			The user ID of the group member for whom the custom attributes are set.
	 * @param attributeMap		The map of custom attributes in key-value format. In a key-value pair, if the value is set to an empty string, the custom attribute will be deleted.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 							if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncSetGroupMemberAttributes(String groupId, String userId, Map<String,String> attributeMap, CallBack callBack){
		ChatClient.getInstance().execute(()->{
			try {
				setMemberAttributes(groupId,userId,attributeMap);
				callBack.onSuccess();
			} catch (ChatException e) {
				callBack.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	protected Map<String,Map<String,String>> fetchMemberAllAttributes(String groupId,List<String> userList) throws ChatException {
		EMAError error = new EMAError();
		Map<String,Map<String,String>> result = emaObject.fetchMemberAllAttributes(groupId,userList,error);
		handleError(error);
		return result;
	}

	/**
	 * 获取单个群成员所有自定义属性。
	 * 
	 * @param groupId			群组 ID。
	 * @param userId			要获取的自定义属性的群成员的用户 ID。
	 * @param callBack			结果回调，成功时回调 {@link ValueCallBack#onSuccess(Object)}，
	 * 							失败时回调 {@link ValueCallBack#onError(int, String)}。
	 *
	 * Gets all custom attributes of a group member. 
	 * 
	 * @param groupId			The group ID.
	 * @param userId			The user ID of the group member whose all custom attributes are retrieved.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 							if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupMemberAllAttributes(String groupId,String userId,ValueCallBack<Map<String,Map<String,String>>> callBack){
		ChatClient.getInstance().execute(()->{
			try {
				List<String> userList = new ArrayList<>();
				userList.add(userId);
				callBack.onSuccess(fetchMemberAllAttributes(groupId,userList));
			} catch (ChatException e) {
				callBack.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	protected Map<String,Map<String,String>> fetchMembersAttributes(String groupId,List<String> userList,List<String> keyList) throws ChatException {
		EMAError error = new EMAError();
		Map<String,Map<String,String>> result = emaObject.fetchMembersAttributes(groupId,userList,keyList,error);
		handleError(error);
		return result;
	}

	/**
	 * 根据指定的属性 key 获取多个群成员的自定义属性。
	 * 
	 * @param groupId			群组 ID。
	 * @param userList			要获取自定义属性的群成员的用户 ID 数组。
	 * @param keyList			要获取自定义属性的 key 的数组。若 keys 为空数组或不传则获取这些群成员的所有自定义属性。
	 * @param callBack			结果回调，成功时回调 {@link ValueCallBack#onSuccess(Object)}，
	 * 							失败时回调 {@link ValueCallBack#onError(int, String)}。
	 *
	 * Gets custom attributes of multiple group members by attribute key.
	 * 
	 * @param groupId			The group ID.
	 * @param userList			The array of user IDs of group members whose custom attributes are retrieved. 
	 * @param keyList			The array of keys of custom attributes to be retrieved. If you pass in an empty array or do not set this parameter, the SDK gets all custom attributes of these group members. 
	 * @param callBack			The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 							if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupMembersAttributes(String groupId,List<String> userList,List<String> keyList,ValueCallBack<Map<String,Map<String,String>>> callBack){
		ChatClient.getInstance().execute(()->{
			try {
				callBack.onSuccess(fetchMembersAttributes(groupId,userList,keyList));
			} catch (ChatException e) {
				callBack.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	// ============================= group_reform new add api end =============================

	/**
	 * \~english
	 * Updates the group extension field.
	 * Only the group owner or admin can call this method.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId       		The group ID.
	 * @param extension     		The group extension field.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Group updateGroupExtension(String groupId, String extension) throws ChatException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.updateGroupExtension(groupId, extension, error);
		handleError(error);
		return new Group(group);
	}

	void onLogout() {
	}

	private void handleError(EMAError error)  throws ChatException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new ChatException(error);
		}
	}
}
