/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import io.agora.CallBack;
import io.agora.ChatThreadChangeListener;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAError;
import io.agora.chat.adapter.EMAThreadInfo;
import io.agora.chat.adapter.EMAThreadManager;
import io.agora.chat.adapter.EMAThreadManagerListener;
import io.agora.chat.adapter.message.EMAMessage;
import io.agora.exceptions.ChatException;
import io.agora.util.EMLog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;


/**
 * \~english
 * The message thread manager that defines how to manage message threads, including message thread creation, destruction, and member management.
 */
public class ChatThreadManager {
	private static String TAG = ChatThreadManager.class.getSimpleName();

	EMAThreadManager emaObject;
	ChatClient mClient;

	List<ChatThreadChangeListener> threadChangeListeners;

	ChatThreadManager(ChatClient client, EMAThreadManager threadManager) {
		emaObject = threadManager;
		mClient = client;
		threadChangeListeners = Collections.synchronizedList(new ArrayList<ChatThreadChangeListener>());
		emaObject.addListener(listenerImpl);
		ChatClient.getInstance().chatManager();
	}

	/**
	 * \~english
	 * Adds the message thread event listener, which listens for message thread changes, such as the message thread creation and destruction.
	 *
	 * You can call {@link #removeChatThreadChangeListener(ChatThreadChangeListener)} to remove an unnecessary message thread event listener.
	 *
	 * @param listener The message thread event listener to add.
	 */
	public void addChatThreadChangeListener(ChatThreadChangeListener listener) {
		if (listener != null && !threadChangeListeners.contains(listener)) {
			threadChangeListeners.add(listener);
		}
	}

	/**
	 * \~english
	 * Removes the message thread event listener.
	 *
	 * After a message thread event listener is added with {@link #addChatThreadChangeListener(ChatThreadChangeListener)}, you can call this method to remove it when it is not required.
	 *
	 * @param listener  The message thread event listener to remove.
	 */
	public void removeChatThreadChangeListener(ChatThreadChangeListener listener) {
		if (listener != null && threadChangeListeners.contains(listener)) {
			threadChangeListeners.remove(listener);
		}
	}

	/**
	 * \~english
	 * Creates a message thread.
	 *
	 * Each member of the chat group where the message thread belongs can call this method.
	 *
	 * Upon the creation of a message thread, the following will occur:
	 *
	 * - In a single-device login scenario, each member of the group to which the message thread belongs will receive the {@link ChatThreadChangeListener#onChatThreadCreated(ChatThreadEvent)} callback.
	 *   You can listen for message thread events by setting {@link ChatThreadChangeListener}.
	 *
	 * - In a multi-device login scenario, the devices will receive the {@link io.agora.MultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *   In this callback method, the first parameter indicates a message thread event, for example, {@link io.agora.MultiDeviceListener#THREAD_CREATE} for a message thread creation event.
	 *   You can listen for message thread events by setting {@link io.agora.MultiDeviceListener}.
	 *
	 * @param parentId 		The parent ID, which is the group ID.
	 * @param messageId 	The ID of the parent message.
	 * @param chatThreadName The name of the new message thread. It can contain a maximum of 64 characters.
	 * @param callBack   	The result callback:
	 *                      - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return the new message thread object;
	 *                      - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void createChatThread(String parentId, String messageId, String chatThreadName, ValueCallBack<ChatThread> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					ChatThread thread = createChatThread(parentId, messageId, chatThreadName);
					if(callBack != null) {
					    callBack.onSuccess(thread);
					}
				} catch (ChatException e) {
					if(callBack != null) {
					    callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"createChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private ChatThread createChatThread(String parentId, String messageId, String chatThreadName) throws ChatException{
		EMAError error = new EMAError();
		EMAThreadInfo thread = emaObject.createThread(parentId, messageId, chatThreadName, error);
		handleError(error);
		return new ChatThread(thread);
	}

	/**
	 * \~english
	 * Gets the details of the message thread from the server.
	 *
	 * @param chatThreadId   The message thread ID.
	 * @param callBack       The result callback:
	 *                       - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return the message thread details;
	 *                       - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadFromServer(String chatThreadId, ValueCallBack<ChatThread> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					ChatThread thread = getChatThreadFromServer(chatThreadId);
					if(callBack != null) {
						callBack.onSuccess(thread);
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getChatThreadFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private ChatThread getChatThreadFromServer(String chatThreadId) throws ChatException {
		EMAError error = new EMAError();
		EMAThreadInfo thread = emaObject.getThreadFromServer(chatThreadId, error);
		handleError(error);
		return new ChatThread(thread);
	}

	/**
	 * \~english
	 * Joins a message thread.
	 *
	 * Each member of the group where the message thread belongs can call this method.
	 *
	 * In a multi-device login scenario, note the following:
	 *
	 * - The devices will receive the {@link io.agora.MultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *
	 * - In this callback method, the first parameter indicates a message thread event, for example, {@link io.agora.MultiDeviceListener#THREAD_JOIN} for a message thread join event.
	 *
	 * - You can listen for message thread events by setting {@link io.agora.MultiDeviceListener}.
	 *
	 * @param chatThreadId The message thread ID.
	 * @param callBack     The result callback:
	 *                     - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return the message thread details {@link ChatThread} which do not include the member count.
	 *                     - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 *                       If you join a message thread repeatedly, the SDK will return an error with the error code {@link io.agora.Error#USER_ALREADY_EXIST}.
	 */
	public void joinChatThread(String chatThreadId, ValueCallBack<ChatThread> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					ChatThread thread = joinChatThread(chatThreadId);
					if(callBack != null) {
					    callBack.onSuccess(thread);
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"joinChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private ChatThread joinChatThread(String chatThreadId) throws ChatException {
		EMAError error = new EMAError();
		EMAThreadInfo thread = emaObject.joinThread(chatThreadId, error);
		handleError(error);
		return new ChatThread(thread);
	}

	/**
	 * \~english
	 * Destroys the message thread.
	 *
	 * Only the owner or admins of the group where the message thread belongs can call this method.
	 *
	 * **Note**
	 *
	 * - In a single-device login scenario, each member of the group to which the message thread belongs will receive the {@link ChatThreadChangeListener#onChatThreadDestroyed(ChatThreadEvent)} callback.
	 *   You can listen for message thread events by setting {@link ChatThreadChangeListener}.
	 *
	 * - In a multi-device login scenario, The devices will receive the {@link io.agora.MultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *   In this callback method, the first parameter indicates a message thread event, for example, {@link io.agora.MultiDeviceListener#THREAD_DESTROY} for a message thread destruction event.
	 *   You can listen for message thread events by setting {@link io.agora.MultiDeviceListener}.
	 *
	 * @param chatThreadId  The message thread ID.
	 * @param callBack      The result callback:
	 *                      - If success, {@link CallBack#onSuccess()} is triggered.
	 * 				        - If a failure occurs, {@link CallBack#onError(int, String)} is triggered to return an error.
	 */
	public void destroyChatThread(String chatThreadId, CallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					destroyChatThread(chatThreadId);
					if(callBack != null) {
					    callBack.onSuccess();
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"destroyChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void destroyChatThread(String chatThreadId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.destroyThread(chatThreadId, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Leaves a message thread.
	 *
	 * Each member in the message thread can call this method.
	 *
	 * In a multi-device login scenario, note the following:
	 *
	 * - The devices will receive the {@link io.agora.MultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *
	 * - In this callback method, the first parameter indicates a message thread event, for example, {@link io.agora.MultiDeviceListener#THREAD_LEAVE} for a message thread leaving event.
	 *
	 * - You can listen for message thread events by setting {@link io.agora.MultiDeviceListener}.
	 *
	 * @param chatThreadId The ID of the message thread that the current user wants to leave.
	 * @param callBack     The result callback:
	 *                     - If success, {@link CallBack#onSuccess()} is triggered;
	 * 				       - If a failure occurs, {@link CallBack#onError(int, String)} is triggered to return an error.
	 */
	public void leaveChatThread(String chatThreadId, CallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					leaveChatThread(chatThreadId);
					if(callBack != null) {
					    callBack.onSuccess();
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"leaveChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void leaveChatThread(String chatThreadId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.leaveThread(chatThreadId, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Changes the name of the message thread.
	 *
	 * Only the owner or admins of the group where the message thread belongs and the message thread creator can call this method.
	 *
	 * Each member of the group to which the message thread belongs will receive the {@link ChatThreadChangeListener#onChatThreadUpdated(ChatThreadEvent)} callback.
	 *
	 * You can listen for message thread events by setting {@link ChatThreadChangeListener}.
	 *
	 * @param chatThreadId      The message thread ID.
	 * @param chatThreadName    The new message thread name. It can contain a maximum of 64 characters.
	 * @param callBack          The result callback:
	 *                          - If success, {@link CallBack#onSuccess()} is triggered.
	 * 				            - If a failure occurs, {@link CallBack#onError(int, String)} is triggered to return an error.
	 */
	public void updateChatThreadName(String chatThreadId, String chatThreadName, CallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateChatThreadName(chatThreadId, chatThreadName);
					if(callBack != null) {
						callBack.onSuccess();
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"changeChatThreadName error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void updateChatThreadName(String chatThreadId, String chatThreadName) throws ChatException{
		EMAError error = new EMAError();
		emaObject.updateChatThreadName(chatThreadId, chatThreadName, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Removes a member from the message thread.
	 *
	 * Only the owner or admins of the group where the message thread belongs and the message thread creator can call this method.
	 *
	 * The removed member will receive the {@link ChatThreadChangeListener#onChatThreadUserRemoved(ChatThreadEvent)} callback.
	 *
	 * You can listen for message thread events by setting {@link ChatThreadChangeListener}.
	 *
	 * @param chatThreadId  The message thread ID.
	 * @param member        The user ID of the member to be removed from the message thread.
	 * @param callBack      The result callback.
	 *                      - If success, {@link CallBack#onSuccess()} is triggered.
	 * 				        - If a failure occurs, {@link CallBack#onError(int, String)} is triggered to return an error.
	 */
	public void removeMemberFromChatThread(String chatThreadId, String member, CallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					removeMemberFromChatThread(chatThreadId, member);
					if(callBack != null) {
						callBack.onSuccess();
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"removeMemberFromThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void removeMemberFromChatThread(String chatThreadId, String member) throws ChatException {
		EMAError error = new EMAError();
		emaObject.removeMemberFromThread(chatThreadId, member, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Gets a list of members in the message thread with pagination.
	 *
	 * Each member of the group to which the message thread belongs can call this method.
	 *
	 * @param chatThreadId The message thread ID.
	 * @param limit        The number of members that you expect to get on each page. The value range is [1,50].
	 * @param cursor       The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the chronological order of when members join the message thread.
	 * @param callBack     The result callback:
	 *                     - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return the result {@link CursorResult}, including the message thread member list and the cursor for the next query.
	 *                     - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadMembers(String chatThreadId, int limit, String cursor, ValueCallBack<CursorResult<String>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					CursorResult<String> cursorResult = getChatThreadMembers(chatThreadId, limit, cursor);
					if(callBack != null) {
					    callBack.onSuccess(cursorResult);
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getThreadMembers error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private CursorResult<String> getChatThreadMembers(String chatThreadId, int limit, String cursor) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<String> cursorResult = emaObject.fetchThreadMembers(chatThreadId, limit, cursor, error);
		handleError(error);

		return cursorResult;
	}

	/**
	 * \~english
	 * Uses the pagination to get the list of message threads that the current user has joined.
	 *
	 * This method gets data from the server.
	 *
	 * @param limit     The number of message threads that you expect to get on each page. The value range is [1,50].
	 * @param cursor    The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the reverse chronological order of when the user joins the message threads.
	 * @param callBack  The result callback:
	 *                  - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return the result {@link CursorResult}, including the message thread list and the cursor for the next query.
	 *                  - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getJoinedChatThreadsFromServer(int limit, String cursor, ValueCallBack<CursorResult<ChatThread>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					CursorResult<ChatThread> result = getJoinedChatThreadsFromServer(limit, cursor);
					if(callBack != null) {
						callBack.onSuccess(result);
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getJoinedThreadsFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private CursorResult<ChatThread> getJoinedChatThreadsFromServer(int limit, String cursor) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<EMAThreadInfo> cursorResult = emaObject.getJoinedThreadsFromServer(limit, cursor, error);
		handleError(error);
		CursorResult<ChatThread> result = new CursorResult<>();
		List<ChatThread> threadList = new ArrayList<>();
		List<EMAThreadInfo> data = cursorResult.getData();
		for(int i = 0; i < data.size(); i++) {
		  	threadList.add(new ChatThread(data.get(i)));
		}
		result.setData(threadList);
		result.setCursor(cursorResult.getCursor());
		return result;
	}

	/**
	 * \~english
	 * Use the pagination to get the list of message threads that the current user has joined in the specified group.
	 *
	 * This method gets data from the server.
	 *
	 * @param parentId  The parent ID, which is the group ID.
	 * @param limit     The number of message threads that you expect to get on each page. The value range is [1,50].
	 * @param cursor    The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the reverse chronological order of when the user joins the message threads.
	 * @param callBack  The result callback:
	 *                  - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return the result {@link CursorResult}, including the message thread list and the cursor for the next query.
	 *                  - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getJoinedChatThreadsFromServer(String parentId, int limit, String cursor, ValueCallBack<CursorResult<ChatThread>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					CursorResult<ChatThread> result = getJoinedChatThreadsFromServer(parentId, limit, cursor);
					if(callBack != null) {
						callBack.onSuccess(result);
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getJoinedThreadsFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private CursorResult<ChatThread> getJoinedChatThreadsFromServer(String parentId, int limit, String cursor) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<EMAThreadInfo> cursorResult = emaObject.getJoinedThreadsFromServer(parentId, limit, cursor, error);
		handleError(error);
		CursorResult<ChatThread> result = new CursorResult<>();
		List<ChatThread> threadList = new ArrayList<>();
		List<EMAThreadInfo> data = cursorResult.getData();
		for(int i = 0; i < data.size(); i++) {
			threadList.add(new ChatThread(data.get(i)));
		}
		result.setData(threadList);
		result.setCursor(cursorResult.getCursor());
		return result;
	}

	/**
	 * \~english
	 * Use the pagination to get the list of message threads in the specified group.
	 *
	 * This method gets data from the server.
	 *
	 * @param parentId  The parent ID, which is the group ID.
	 * @param limit     The number of message threads that you expect to get on each page. The value range is [1,50].
	 * @param cursor    The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the reverse chronological order of when message threads are created.
	 * @param callBack  The result callback:
	 *                  - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return the result {@link CursorResult}, including the message thread list and the cursor for the next query.
	 *                  - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadsFromServer(String parentId, int limit, String cursor, ValueCallBack<CursorResult<ChatThread>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					CursorResult<ChatThread> result = getChatThreadsFromServer(parentId, limit, cursor);
					if(callBack != null) {
					    callBack.onSuccess(result);
					}
				} catch (ChatException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getThreadsFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private CursorResult<ChatThread> getChatThreadsFromServer(String parentId, int limit, String cursor) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<EMAThreadInfo> cursorResult = emaObject.getThreadsFromServer(parentId, limit, cursor, error);
		handleError(error);
		CursorResult<ChatThread> result = new CursorResult<>();
		List<ChatThread> threadList = new ArrayList<>();
		List<EMAThreadInfo> data = cursorResult.getData();
		for(int i = 0; i < data.size(); i++) {
			threadList.add(new ChatThread(data.get(i)));
		}
		result.setData(threadList);
		result.setCursor(cursorResult.getCursor());
		return result;
	}

	/**
	 * \~english
	 * Gets the last reply in the specified message threads from the server.
	 *
	 * @param chatThreadIds The list of message thread IDs to query. You can pass a maximum of 20 message thread IDs each time.
	 * @param callBack      The result callback:
	 *                      - If success, {@link ValueCallBack#onSuccess(Object)} is triggered to return a Map collection that contains key-value pairs where the key is the message thread ID and the value is the last threaded reply.
	 * 	                    - If a failure occurs, {@link ValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadLatestMessage(List<String> chatThreadIds, ValueCallBack<Map<String, ChatMessage>> callBack) {
		mClient.execute(()-> {
			Map<String, ChatMessage> messageMap = null;
			try {
				messageMap = getChatThreadLatestMessage(chatThreadIds);
				if(callBack != null) {
					callBack.onSuccess(messageMap);
				}
			} catch (ChatException e) {
				if(callBack != null) {
				    callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	private Map<String, ChatMessage> getChatThreadLatestMessage(List<String> chatThreadIds) throws ChatException {
		EMAError error = new EMAError();
		Map<String, EMAMessage> result = emaObject.getThreadsLatestMessage(chatThreadIds, error);
		handleError(error);
		Map<String, ChatMessage> map = new HashMap<>();
		Iterator<Map.Entry<String, EMAMessage>> iterator = result.entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry<String, EMAMessage> item = iterator.next();
			String key = item.getKey();
			ChatMessage value = new ChatMessage(item.getValue());
			map.put(key, value);
		}
		return map;
	}

	private void handleError(EMAError error)  throws ChatException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new ChatException(error);
		}
	}

	private EMAThreadManagerListener listenerImpl = new EMAThreadManagerListener() {
		@Override
		public void onThreadNameUpdated(EMAThreadInfo event) {

		}

		@Override
		public void onLeaveThread(EMAThreadInfo event, int reason) {
			synchronized (threadChangeListeners) {
				for (ChatThreadChangeListener listener : threadChangeListeners) {
					try {
						if(reason == EMAThreadInfo.LeaveReason.BE_KICKED.ordinal()) {
							listener.onChatThreadUserRemoved(new ChatThreadEvent(event));
						}else {
							listener.onChatThreadDestroyed(new ChatThreadEvent(event));
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onMemberJoined(EMAThreadInfo event) {

		}

		@Override
		public void onMemberExited(EMAThreadInfo event) {

		}

		@Override
		public void onThreadNotifyChange(EMAThreadInfo event) {
			synchronized (threadChangeListeners) {
				for (ChatThreadChangeListener listener : threadChangeListeners) {
					try {
						switch (event.getType()) {
							case CREATE:
								listener.onChatThreadCreated(new ChatThreadEvent(event));
						        break;
							case UPDATE:
							case UPDATE_MSG:
								listener.onChatThreadUpdated(new ChatThreadEvent(event));
							    break;
							case DELETE:
								listener.onChatThreadDestroyed(new ChatThreadEvent(event));
							    break;
						}
					} catch (Exception e) {
						EMLog.e(TAG, e.getMessage());
					}
				}
			}
		}
	};
}
