/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import android.content.Context;

import android.support.annotation.NonNull;

import io.agora.chat.core.EMChatConfigPrivate;
import io.agora.push.PushConfig;
import io.agora.util.EMLog;

/**
 * \~english
 * The settings of the chat SDK.
 * You can set parameters and options of the SDK.
 * For example, whether to encrypt the messages before sending, whether to automatically accept the friend invitations.
 */
public class ChatOptions {

	private final String TAG=getClass().getSimpleName();

	public static class AreaCode {
		public static final int AREA_CODE_CN = 1;
		public static final int AREA_CODE_NA = 2;
		public static final int AREA_CODE_EU = 4;
		public static final int AREA_CODE_AS = 8;
		public static final int AREA_CODE_JP = 16;
		public static final int AREA_CODE_IN = 32;
		public static final int AREA_CODE_GLOB = -1;

		public AreaCode() {
		}
	}



	private int areaCode = AreaCode.AREA_CODE_GLOB;

	/**
	 * \~english
	 * Whether to accept friend invitations from other users automatically. 
	 * - (Default)`true`: Accepts friend invitations automatically;
	 * - `false`: Do not accept friend invitations automatically.
	 */
	private boolean acceptInvitationAlways = true;
	
	/**
	 * \~english
	 * Whether to accept group invitations automatically. 
	 * - (Default)`true`: Accepts group invitations automatically;
	 * - `false`: Do not accept group invitations automatically.
	 */
	private boolean autoAcceptGroupInvitation = true;
	
	/**
	 * \~english
	 * Whether to encrypt the messages. 
	 * - (Default)`false`: Do not encrypt the messages;
	 * - `true`: Encrypt the messages.
	 */
	private boolean useEncryption = false;

	/**
	 * \~english
	 * Whether the message read receipt is required. 
	 * - (Default)`true`: The message read receipt is required. 
	 * - `false`: The message read receipt is NOT required.
	 */
	private boolean requireReadAck = true;
	
	/**
	 * \~english
	 * Whether to send the message delivery receipt. 
	 * - (Default)`false`: The SDK does not automatically send the delivery receipt when you receive a chat message;
	 * - `true`: The SDK automatically sends the delivery receipt when you receive a chat message.
	 */
	private boolean requireDeliveryAck = false;
	/**
	 * \~english
	 * Whether to use fpa function.
	 * - (Default)`false`: not use fpa function;
	 * - `true`: use fpa function.
	 */
	private boolean fpaEnable = false;

    /**
	 * \~english
	 * Whether to delete all of the group messages when leaving the group.
	 * - (Default)`true`: Deletes all of the group messages when leaving the group. 
	 * - `false`: Do not delete all of the group messages when leaving the group.
	 */
	private boolean deleteMessagesAsExitGroup = true;
	
    /**
	 * \~english
	 * Whether to allow the chat room owner to leave the chat room.
	 * - (Default)`true`: Allow the chat room owner to leave the chat room. 
	 * - `false`: Do not allow the chat room owner to leave the chat room.
	 */

	private boolean isChatroomOwnerLeaveAllowed = true;

	/**
	 * \~english
	 * Whether to delete all of the chat room messages when leaving the chat room.
	 * - (Default)`true`: Deletes all of the chat room messages when leaving the chat room. 
	 * - `false`: Do not delete all of the chat room messages when leaving the chat room.
	 */
	private boolean deleteMessagesAsExitChatRoom = true;

	/**
	 * \~english
	 * Whether to include empty conversations when the SDK loads conversations from the local database.
	 * - (Default)`false`: Empty conversations are excluded.
	 * - `true`: Empty conversations are included.
	 */
	private boolean isLoadEmptyConversations = false;

	private String appkey = "";
	
	private EMChatConfigPrivate config = null;
	
	private boolean enableAutoLogin = true;

	private String fcmNumber = null;
	private boolean useFCM = true;

	private boolean enableDNSConfig = true;
	
	private boolean sortMessageByServerTime = true;

	private boolean useHttps = false;
	private String dnsUrl = "";

	private String restServer;
	private String reportServer;
	private String imServer;
	private int imPort;

	private boolean usingHttpsOnly = true;
	private boolean serverTransfer = true;
	private boolean isAutodownload = true;
	private boolean useStereoInput = false;
	private boolean enableStatistics = false;
	private boolean enableUseRtcConfig = false;
	private String  rtcConfigUrl;
	private PushConfig pushConfig;

	private int osCustomPlatform = -1;
	private String customDeviceName;

	/**
	 * \~english
	 * Gets whether the read receipt is required by the message receiver. 
	 * 
	 * @return Whether the read receipt is required.
	 * - (Default) `true`: The read receipt is required;
	 * - `false`: The read receipt is not required.
	 */
	public boolean getRequireAck() {
		
		if(config == null){
			return requireReadAck;
		}
		return  config.getRequireReadAck();
	}

	/**
	 * \~english
	 * Sets whether to require the read receipt. 
	 *
	 * @param requireAck Whether the read receipt is required.
	 * - (Default) `true`: The read receipt is required; 
	 * - `false`: The read receipt is not required.
	 */
	public void setRequireAck(boolean requireAck) {
		this.requireReadAck = requireAck;

		if (config == null) {
			return;
		}
		config.setRequireReadAck(requireAck);
	}
	
	/**
	 * \~english
	 * Gets whether the delivery receipt is required.
	 *
	 * @return Whether the delivery receipt is required.
	 * - (Default)`true`: The read receipt is required; 
	 * - `false`: The read receipt is not required.
	 */
	public boolean getRequireDeliveryAck() {
		if (config == null) {
			return requireDeliveryAck;
		}
		return config.getRequireDeliveryAck();
	}
	/**
	 * \~english
	 * Gets whether the fpa function is used.
	 *
	 * @return whether the fpa function is used.
	 * - (Default)`false`: the fpa function is not used.
	 * - `true`: the fpa function is used;
	 */
	public boolean getFpaEnable() {
		if (config == null) {
			return fpaEnable;
		}
		return config.getFpaEnable();
	}
	/**
	 * \~english
	 * sets Whether to use fpa function.
	 * @param fpaEnable Whether to use fpa function.
	 * - (Default)`false`: not use fpa function;
	 * - `true`: use fpa function.
	 */
	public void setFpaEnable(boolean fpaEnable) {
		this.fpaEnable = fpaEnable;
		if (config == null) {
			return;
		}
		config.setFpaEnable(fpaEnable);
	}

	/**
	 * \~english
	 * Sets whether the delivery receipt is required.
	 * 
	 * @param requireDeliveryAck Whether the delivery receipt is required.
	 * - (Default)`true`: The delivery receipt is required; 
	 * - `false`: The delivery receipt is not required.
	 */
	public void setRequireDeliveryAck(boolean requireDeliveryAck) {
		this.requireDeliveryAck = requireDeliveryAck;
		
		if (config == null) {
			return;
		}
		config.setRequireDeliveryAck(requireDeliveryAck);
	}

	/**
	 * \~english
	 * Gets whether to accept other user's friend invitations automatically.
	 *
	 * @return Whether to accept friend invitation automatically.
	 * - (Default)`true`: Accepting friend invitation automatically.
	 * - `false`: Do not accepting friend invitation automatically.
	 */
	public boolean getAcceptInvitationAlways() {
		if (config == null) {
			return acceptInvitationAlways;
		}
		return config.getAutoAccept();
	}

	/**
	 * \~english
	 * Sets whether to accept the friend invitation automatically. 
	 * @param value Whether to accept the friend invitation automatically.
	 * - (Default)`true`: Accepting friend invitation automatically.
	 * - `false`: Do not accepting friend invitation automatically.
	 */
	public void setAcceptInvitationAlways(boolean value) {
		if (config == null) {
			acceptInvitationAlways = value;
			return;
		}
		config.setAutoAccept(value);
	}

	/**
     * \~english
	 * Sets whether to delete the group messages when leaving the group.
	 *
     * @param delete  
	 * - (Default)`true`: Delete the messages when leaving the group.
	 * - `false`: Do not delete the messages when leaving a group.
     */
    public void setDeleteMessagesAsExitGroup(boolean delete){
		if (config == null) {
			deleteMessagesAsExitGroup = delete;
			return;
		}
		
		config.setDeleteMessageAsExitGroup(delete);
    }
    
    /**
     * \~english
	 * Gets whether to delete the group message when leaving a group.
	 * 
     * @return - (Default)`true`: Delete the messages when leaving a group.
     * - `false`: Do not delete the messages when leaving a group.
	 * 
	 * @deprecated Use {@link ChatOptions#deleteMessagesOnLeaveGroup()} instead.
     */
    @Deprecated
    public boolean isDeleteMessagesAsExitGroup() {
		return deleteMessagesOnLeaveGroup();
    }

	/**
	 * \~english
	 * Gets whether to delete the group message when leaving a group.
     * @return - (Default)`true`: Delete the messages when leaving a group.
     * - `false`: Do not delete the messages when leaving a group.
	 */
	public boolean deleteMessagesOnLeaveGroup() {
		if (config == null) {
			return deleteMessagesAsExitGroup;
		}

		return config.getDeleteMessageAsExitGroup();

	}
    
    /**
     * \~english
     * Sets whether to accept a group invitation automatically. 
	 * 
     * @param value Whether to accept group invitation automatically.
	 * - (Default)`true`: Accept group invitations automatically；
	 * - `false`: Do not accept group invitations automatically.
     */
    public void setAutoAcceptGroupInvitation(boolean value) {
        if (config == null) {
            autoAcceptGroupInvitation = value;
            return;
        }
        config.setAutoAcceptGroupInvitation(value);
    }
    
    /**
     * \~english
     * Gets whether to accept a group invitation automatically.
	 * 
     * @return	Returns the result of whether to accept group invitation automatically.
	 * - (Default)`true`: Accept group invitations automatically；
	 * - `false`: Do not accept group invitations automatically.
	 * 
	 * @deprecated Use {@link ChatOptions#autoAcceptGroupInvitations()} instead.
     */
    @Deprecated
    public boolean isAutoAcceptGroupInvitation() {
        return autoAcceptGroupInvitations();
    }

	/**
	 * \~english
	 * Gets whether to accept group invitations automatically.
	 * 
	 * @return	Returns the result of whether to accept group invitation automatically. 
	 * - (Default)`true`: Accept group invitations automatically.
	 * - `false`: Do not accept group invitations automatically.
	 */
	public boolean autoAcceptGroupInvitations() {
		if (config == null) {
			return autoAcceptGroupInvitation;
		}
		return config.isAutoAcceptGroupInvitation();
	}

	/**
	 * \~english
	 * Sets whether to allow the chat room owner to leave the chat room.
	 * @param allowed Whether to allow the owner of chat room to leave.
	 * - (Default)`true`: : When the owner leave, do not keep the conversation; 
	 * - `false`: When the owner leave do keep the conversation.
	 */
    public void allowChatroomOwnerLeave(boolean allowed){
		if (config == null) {
			this.isChatroomOwnerLeaveAllowed = allowed;
			return;
		}
		
		config.setIsChatroomOwnerLeaveAllowed(allowed);
	}
	
	/**
	 * \~english
	 * Gets whether to allow the owner of chat room to leave.
	 * @return - (Default)`true`: When the owner leave, do not keep the conversation; 
	 * - `false`: When the owner leave do keep the conversation.
	 *  @deprecated Use {@link ChatOptions#canChatroomOwnerLeave()} instead.
	 */
	@Deprecated
	public boolean isChatroomOwnerLeaveAllowed(){

		return canChatroomOwnerLeave();
	}

	/**
	 * \~english
	 * Gets whether to allow owner of chat room to leave.
	 * @return - (Default)`true`: When the owner leave, do not keep the conversation;
	 * - `false`: When the owner leave, do keep the conversation.
	 */
	public boolean canChatroomOwnerLeave(){
		if (config == null) {
			return isChatroomOwnerLeaveAllowed;
		}
		return config.getIsChatroomOwnerLeaveAllowed();
	}

	/**
	 * \~english
	 * Sets whether to delete the chat room message when leaving the chat room.
	 * 
	 * @param delete -(Default)`true`: Delete the chat room related message record when leaving the chat room.
	 * - `false`: Do not delete the chat room related message record when leaving the chat room.
	 */
	public void setDeleteMessagesAsExitChatRoom(boolean delete){
		if (config == null) {
			deleteMessagesAsExitChatRoom = delete;
			return;
		}

		config.setDeleteMessageAsExitChatRoom(delete);
	}

	/**
	 * \~english
	 * Gets whether to delete the chat room message when leaving the chat room.
	 * 
	 * @return  - (Default)`true`: Delete the chat room related message record when leaving the chat room.
	 * - `false`: Do not delete the chat room related message record when leaving the chat room.
	 * 
	 * @deprecated Use {@link ChatOptions#deleteMessagesOnLeaveChatroom()} instead.
	 */
	@Deprecated
	public boolean isDeleteMessagesAsExitChatRoom() {

		return deleteMessagesOnLeaveChatroom();

	}
	/**
	 * \~english
	 * Gets whether to delete the chat room message when leaving the chat room.
	 * @return  - (Default)`true`: Delete the chat room related message record when leaving the chat room.
	 * - `false`: Do not delete the chat room related message record when leaving the chat room.
	 */
	public boolean deleteMessagesOnLeaveChatroom() {
		if (config == null) {
			return deleteMessagesAsExitChatRoom;
		}

		return config.getDeleteMessageAsExitChatRoom();

	}
	 
	/**
	 * \~english
	 * Gets whether to sort messages by the server received time.
	 * @return Returns the result of whether sort message by the server received time.
	 * - (Default)`true`: Sort messages by the server received time;
	 * - `false`: Do not sort messages by the server received time.
	 */
    public boolean isSortMessageByServerTime() {
        if (config == null) {
            return sortMessageByServerTime;
        }
        return config.getSortMessageByServerTime();
    }

	/**
	 * \~english
	 * Sets whether to sort messages by server received time. 
	 * @param sortByServerTime
	 * - (Default)`true`: Sort messages by the server received time.
	 * - `false`: Do not sort messages by the server received time.
	 */
	public void setSortMessageByServerTime(boolean sortByServerTime) {
        if (config == null) {
            this.sortMessageByServerTime = sortByServerTime;
            return;
        }
        
        config.setSortMessageByServerTime(sortByServerTime);
    }
	
	/**
	 * \~english
	 * Sets the app key.
	 * @param appkey The app key you got from the console when creating a chat app.
	 */
	void updatePath(String appkey){
		if(config != null) {
            config.updatePath(appkey);
		}
	}

	/**
	 * \~english
	 * Sets the app key.
	 * @param appkey The app key you got from the console when create an app.
	 */
	public void setAppKey(String appkey) {
		this.appkey = appkey;
		updatePath(appkey);
	}
	
	/**
	 * \~english
	 * Gets the app key.
	 */
	public String getAppKey(){
		if(config == null){
			return appkey;
		}
		
		return config.getAppKey();
	}

	/**
	 * \~english
	 * Sets the custom im message server url.
	 * @param imServer The im message server url.
     */
	public void setIMServer(String imServer){
		if(imServer == null){
			return;
		}

		this.imServer = imServer;
		if(config == null){
			return;
		}
		config.setChatServer(imServer);
	}

	/**
	 * \~english
	 * Gets the im server url.
	 * @return The im server url.
	 */
	public String getImServer(){
		return imServer;
	}

	/**
	 * \~english
	 * Sets the custom im server port.
	 * @param imPort The im server port.
	 */
	public void setImPort(int imPort){
		this.imPort = imPort;
		if(config == null){
			return;
		}
		config.setChatPort(imPort);
	}

	/**
	 * \~english
	 * Gets the im server port.
	 * @return The im server port.
	 */
	public int getImPort(){
		return imPort;
	}

	/**
	 * \~english
	 * Sets the custom REST server.
	 * @param restServer The REST server url.
	 */
	public void setRestServer(String restServer){
		if(restServer == null)
			return;

		this.restServer = restServer;
		if(config == null){
			return;
		}
		config.setRestServer(restServer);
	}

	/**
	 * \~english
	 * Gets the REST server.
	 * @return The REST server url.
	 */
	public String getRestServer(){
		return restServer;
	}


	/**
	 * \~english
	 * Gets the report server.
	 * @return The report server url, may be null.
     */
	public String getReportServer(){
		return reportServer;
	}


	/**
	 * \~english
	 * Sets the custom report server.
	 * @param reportServer The report server url.
	 */
	public void setReportServer(String reportServer){
		if(reportServer == null)
			return;
		this.reportServer = reportServer;
	}

	/**
	 * \~english
	 * Enables/Disables automatic login.
	 * @param autoLogin 
	 * - (Default)`true`: Enables automatic login;
	 * - `false`: Disables automatic login.
	 */
	public void setAutoLogin(boolean autoLogin){
		enableAutoLogin = autoLogin;
	}
	
	/**
	 * \~english
	 * Checks whether the automatic login is enabled.
	 * @return - (Default)`true`: Automatic login is enabled;
	 * - `false`: Automatic login is disabled.
	 */
	public boolean getAutoLogin(){
		return enableAutoLogin;
	}

	void setConfig(EMChatConfigPrivate config){
		this.config = config;
	}

	/**
	 * \~english
	 * Sets whether to disable DNS.
	 * - (Default)`true`: Disable DNS;
	 * - `false`: Do not disable DNS.
	 */
	public void enableDNSConfig(boolean enable) {
		enableDNSConfig = enable;
        if (config == null) {
            return;
        }
	    config.enableDnsConfig(enable);
	}

	/**
	 * \~english
	 * Gets whether to disable DNS. 
	 * - (Default) `true`: Disable DNS;
	 * - `false`: Do not disable DNS.
	 *
	 * @return Returns whether to disable DNS.
	 */
	public boolean getEnableDNSConfig() {
	    return enableDNSConfig;
	}

	/**
	 * \~english
	 * Sets whether only HTTPS is used for REST operation.
	 * @param _usingHttpsOnly Whether only HTTPS is used.
	 * - (Default)`true`: Only HTTPS is used;
	 * - `false`: Both HTTP and HTTPS can be used. 
	 */
	public void setUsingHttpsOnly(boolean _usingHttpsOnly) {
        usingHttpsOnly = _usingHttpsOnly;
		if (config == null) {
			return;
		}
		config.setUsingHttpsOnly(_usingHttpsOnly);
	}

	/**
	 * \~english
	 * Gets whether only HTTPS is used for REST operations. 
	 *
	 * @return Whether only HTTPS is used for REST operations. 
	 * - (Default) `true`: Only HTTPS is used;
	 * - `false`: Both HTTP and HTTPS can be used. 
	 */
	public boolean getUsingHttpsOnly() {
		if (config != null) {
			return config.getUsingHttpsOnly();
		}
		return usingHttpsOnly;
	}
	/**
	 * \~english
	 * Sets whether to upload the message attachments automatically to the chat server.
	 * 
	 * @param transfer Whether to upload the message attachments to the chat server.
	 * - (Default)`true`: Use the default way to upload and download the message attachments by chat server;
	 * - `false`: Do not use the default way to upload and download the message attachments by chat server, using a customized path instead.
	 */
	public void setAutoTransferMessageAttachments(boolean transfer) {
		if (config == null) {
			serverTransfer = transfer;
			return;
		}
		config.setAutoTransferMessageAttachments(transfer);
	}
	/**
	 * \~english
	 * Gets whether to upload the message attachments automatically to the chat server.
	 *
	 * @return Whether to upload the message attachments automatically to the chat server.
	 * - (Default)`true`: Use the default way to upload and download the message attachments by chat server;
	 * - `false`: Do not use the default way to upload and download the message attachments by chat server, using a customized path instead.
	 */
	public boolean getAutoTransferMessageAttachments() {
		if (config == null) {
			return serverTransfer;
		}
		return config.getAutoTransferMessageAttachments();
	}
	/**
	 * \~english
	 * Sets whether to auto download the thumbnail.
	 *
	 * @param autodownload Whether to download the thumbnail automatically.
	 * - (Default)`true`: Download the thumbnail automatically;
	 * - `false`: Do not download the thumbnail automatically.
	 * 
	 */
	public void setAutoDownloadThumbnail(boolean autodownload) {
		if (config == null) {
			isAutodownload = autodownload;
			return;
		}
		config.setAutodownloadThumbnail(autodownload);
	}
	/**
	 * \~english
	 * Gets whether to download the thumbnail automatically. 
	 * 
	 * @return Whether to download the thumbnail automatically.
	 * - (Default)`true`: Download the thumbnail automatically;
	 * - `false`: Do not download the thumbnail automatically.
	 */
	public boolean getAutodownloadThumbnail() {
		if (config == null) {
			return isAutodownload;
		}
		return config.getAutodownloadThumbnail();
	}


	private int fixedInterval = -1;

	/**
	 * \~english
	 * Sets a fixed heartbeat interval. If not set, the optimal heartbeat interval is automatically detected.
	 * @param interval  The interval of heart beat, in seconds (30~300).
	 */
	public void setFixedHBInterval(int interval) {
		if (interval < 10) this.fixedInterval = 10;
		if (interval > 300) this.fixedInterval = 300;
		this.fixedInterval = interval;
	}

	/**
	 * \~english
	 * Gets the fixed heart beat interval.
	 * @return The heartbeat interval, in seconds. We recommend setting this value between 30 to 300.
	 */
	public int getFixedInterval() {
		return this.fixedInterval;
	}


	/**
	 * \~english
	 * Gets the SDK version.
	 * @return The SDK version.
	 */
	public String getVersion() {
		return config.getVersion();
	}

	/**
	 * \~english
	 * Gets the access token from the local database.
	 *
	 * @return The access token.
	 */
	public String getAccessToken() {
		return config.getAccessToken();
	}

	/**
	 * \~english
	 * Whether to get the access token from the server.
	 * @param b Whether to get the access token from server.
	 * - (Default) `true`: Get the access token from server.
	 * - `false`: Do not get the access token from server.
	 * @return The access token.
	 */
	public String getAccessToken(boolean b) {
		return config.getAccessToken(b);
	}

	/**
	 * \~english
	 * Sets the DNS url.
	 * @param dnsUrl The DNS url.
	 */
	public void setDnsUrl(String dnsUrl) {
		this.dnsUrl = dnsUrl;
		if (config == null) {
			return;
		}
		config.setDnsUrl(dnsUrl);
	}

	/**
	 * \~english
	 * Gets the DNS url.
	 * @return  The DNS url.
	 */
	public String getDnsUrl() {
		if (config != null) {
			return config.getDnsUrl();
		}
		return dnsUrl;
	}

	/**
	 * \~english
	 * Gets the push configuration setting options by user.
	 */
	public PushConfig getPushConfig() {
		return pushConfig;
	}

	/**
	 * \~english
	 * Sets the push configuration.
	 *
	 * @param pushConfig  The push configuration.
	 */
	public void setPushConfig(PushConfig pushConfig) {
		this.pushConfig = pushConfig;
	}

	/**
	 * \~english
	 * Checks whether the traffic statistics collection is enabled for local messages.
	 *
	 * @return Whether the message traffic statistics collection is enabled:
	 * 			- `true`: Enabled;
	 * 			- (Default)`false`: Disabled.
	 */
	public boolean isEnableStatistics() {
		if (config == null) {
			return enableStatistics;
		}
		return config.isEnableStatistics();
	}

	/**
	 * \~english
	 * Sets whether the traffic statistics collection is enabled for local messages.
	 * Sets before calling {@link ChatClient#init(Context, ChatOptions)}.
	 * 
	 * @param enableStatistics Whether the message traffic statistics collection can be enabled:
	 * - `true`: Enable;
	 * - (Default)`false`: Disable.
	 */
	public void setEnableStatistics(boolean enableStatistics) {
		this.enableStatistics = enableStatistics;
		if (config == null) {
			return;
		}
		config.setEnableStatistics(enableStatistics);
	}
  
  /**
	 * \~english
	 * sets area code, will follow the area when using edge node.
	 * @param code area code.
	 * - (Default)`AREA_CODE_GLOB`: No limit when try to access server node;
	 *
	 */
	public void setAreaCode(int code) {
		this.areaCode = code;
		if (config == null) {
			return;
		}
		config.setAreaCode(code);
	}

	/**
	 * \~english
	 * Gets the area code and follows zone restrictions when using edge nodes
	 * @return area code .
	 */
	public int getAreaCode() {
		return areaCode;
	}


	/**
	 * \~english
	 * Gets custom platform code
	 * @return  custom platform code
	 */
	public int getCustomOSPlatform() {
		if(config==null) {
			return osCustomPlatform;
		}
		return config.getCustomOSPlatform();
	}

	/**
	 * \~english
	 * set custom platform code .
	 * Sets before calling {@link ChatClient#init(Context, ChatOptions)} .
	 * @param platform custom platform code，The value ranges from 1 to 100
	 */
	public void setCustomOSPlatform(int platform) {
		if(platform<1||platform>100) {
			EMLog.e(TAG,"ChatOptions setCustomOSPlatform() error : platform ="+platform);
		    return;
		}
		this.osCustomPlatform = platform;
		if (config == null) {
			return;
		}
		config.setCustomOSPlatform(platform);
	}

	/**
	 * \~english
	 * Obtain the name of the user-defined device
	 * @return the name of the user-defined device
	 */
	public String getCustomDeviceName() {
		if(config==null) {
			return customDeviceName;
		}
		return config.getDeviceName();

	}

	/**
	 *\~english
	 * Set the current device custom device name. If it is not defined, the system device brand and model name are used by default,
	 * for example, HUAWEICOL-AL10. It can be used to distinguish devices when multiple devices log in.Empty strings and null values are not supported.
	 * developers can use {@link ChatClient#getLoggedInDevicesFromServer(String, String)} for more equipment related information.
	 * Sets before calling {@link ChatClient#init(Context, ChatOptions)}  .
	 * @param customDeviceName custom device name
	 */
	public void setCustomDeviceName(@NonNull String customDeviceName) {
		this.customDeviceName = customDeviceName;
		if (config == null) {
			return;
		}
		config.setDeviceName(customDeviceName);
	}

	/**
	 * \~english
	 * Checks whether to include empty conversations when the SDK loads conversations from the local database.
	 *
	 * @return Whether to include empty conversations when the SDK loads conversations from the local database.
	 * - (Default)`false`: Empty conversations are excluded.
	 * - `true`: Empty conversations are included.
	 */
	public boolean isLoadEmptyConversations() {
		if (config == null) {
			return isLoadEmptyConversations;
		}
		return config.isLoadEmptyConversations();
	}

	/**
	 * \~english
	 * Sets whether to include empty conversations when the SDK loads conversations from the local database.
	 * As for the call sequence, this method comes before {@link ChatClient#init(Context, ChatOptions)}.
	 *
	 * @param isLoadEmptyConversations Whether to include empty conversations when the SDK loads conversations from the local database:
	 * - `true`: Empty conversations are included.
	 * - (Default)`false`: Empty conversations are excluded.
	 */
	public void setLoadEmptyConversations(boolean isLoadEmptyConversations) {
		this.isLoadEmptyConversations = isLoadEmptyConversations;
		if (config == null) {
			return;
		}
		config.setLoadEmptyConversations(isLoadEmptyConversations);
	}

}
