/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora;

import io.agora.chat.GroupReadAck;
import io.agora.chat.ChatMessage;
import io.agora.chat.MessageReactionChange;

import java.util.List;

/**
 * \~english
 * The message event listener.
 * This listener is used to check whether messages are received. If messages are sent successfully, a delivery receipt will be returned (delivery receipt needs to be enabled: {@link io.agora.chat.ChatOptions#setRequireDeliveryAck(boolean)}.
 * If the peer reads the received message, a read receipt will be returned (read receipt needs to be enabled: {@link io.agora.chat.ChatOptions#setRequireAck(boolean)})
 * During message delivery, the message ID will be changed from a local uuid to a global unique ID that is generated by the server to uniquely identify a message on all devices using the SDK.
 * This API should be implemented in the app to listen for message status changes.
 * 
 *     Adds the message listener:
 *     ```java
 *     ChatClient.getInstance().chatManager().addMessageListener(mMessageListener);
 *     ```
 *
 *     Removes the message listener:
 *     ```java
 *     ChatClient.getInstance().chatManager().removeMessageListener(mMessageListener);
 *     ```
 * 
 */
public interface MessageListener
{
	/**
     * \~english
	 * Occurs when a message is received.
	 * This callback is triggered to notify the user when a message such as texts or an image, video, voice, location, or file is received.
	 */
	void onMessageReceived(List<ChatMessage> messages);

	/**
     * \~english
	 * Occurs when a command message is received.
	 * Unlike {@link MessageListener#onMessageReceived(List)}, this callback only contains a command message body that is usually invisible to users.
	 *
	 */
	default void onCmdMessageReceived(List<ChatMessage> messages) {
	}

    /**
     * \~english
     * Occurs when a read receipt is received for a message. 
     */
	default void onMessageRead(List<ChatMessage> messages) {}

	/**
	 * \~english
	 * Occurs when a read receipt is received for a group message.
	 */
	default void onGroupMessageRead(List<GroupReadAck> groupReadAcks) {}

	/**
	 * \~english
	 * Occurs when the update for the group message read status is received.
	 */
	default void onReadAckForGroupMessageUpdated() {}

    /**
     * \~english
     * Occurs when a delivery receipt is received.
     */
	default void onMessageDelivered(List<ChatMessage> messages){}

	/**
 	 * \~english
 	 * Occurs when a received message is recalled.
 	 */
	default void onMessageRecalled(List<ChatMessage> messages){}
    
	/**
     * \~english
     * Occurs when a message modification is received, including the changed message ID.
     *
     * @param message   The modified message.
     * @param change    The message modification.
	 * @deprecated Deprecated. Use {@link ChatMessage#setMessageStatusCallback(CallBack)} instead.
     */
	@Deprecated
	default void onMessageChanged(ChatMessage message, Object change) {}

	/**
	 * \~english
	 * Occurs when a message Reaction is changed.
	 * 
	 */
	default void onReactionChanged(List<MessageReactionChange> messageReactionChangeList) {}

	/**
	 * \~english
	 * Occurs when the message content is modified.
	 * 
	 * @param messageModified  The modified message object, where the message body contains the information such as the number of message modifications, the operator of the last modification, and the last modification time.
	 * 	Also, you can get the operator of the last message modification and the last modification time via the `onMessageContentChanged` method.
	 * @param operatorId The user ID of the operator that modified the message last time.
	 * @param operationTime The last message modification time. It is a UNIX timestamp in milliseconds.
	 */
	default void onMessageContentChanged(ChatMessage messageModified, String operatorId, long operationTime){
	}
}
