/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora;

import io.agora.chat.Group;
import io.agora.chat.MucSharedFile;

import java.util.List;
import java.util.Map;

/**
 *  \~english
 *  The group change listener.
 *  Listens for group events such as requesting to join a group, approving or declining a group request, and kicking a user out of a group.
 * 
 *  
 *      Registers a group change listener:
 * ```java
 *      ChatClient.getInstance().groupManager().addGroupChangeListener(mGroupChangeListener);
 * ```
 *
 *      Unregisters a group change listener:
 * ```java
 *      ChatClient.getInstance().groupManager().removeGroupChangeListener(mGroupChangeListener);
 *  ```
 */
public interface GroupChangeListener {
	/**
     * \~english
	 * Occurs when the user receives a group invitation.
	 * @param groupId		The group ID.
	 * @param groupName		The group name.
	 * @param inviter		The invitee ID.
	 * @param reason		The reason for invitation.
	 */
	void onInvitationReceived(String groupId, String groupName, String inviter, String reason);
	
	/**
     * \~english
     * Occurs when the group owner or administrator receives a group request from a user.
	 *
	 * @param groupId		The group ID.
	 * @param groupName		The group name.
	 * @param applicant		The ID of the user requesting to join the group.
	 * @param reason		The reason for requesting to join the group.
	 */
	void onRequestToJoinReceived(String groupId, String groupName, String applicant, String reason);

	/**
     * \~english
	 * Occurs when a group request is accepted.
	 *
	 * @param groupId 		The group ID.
	 * @param groupName 	The group name.
	 * @param accepter 		The ID of the user that accepts the group request.
	 */
	void onRequestToJoinAccepted(String groupId, String groupName, String accepter);

	/**
     * \~english
     * Occurs when a group request is declined.
	 *
	 * @param groupId 		The group ID.
	 * @param groupName 	The group name.
	 * @param decliner 		The ID of the user that declines the group request.
	 * @param reason 		The reason for declining.
	 */
	void onRequestToJoinDeclined(String groupId, String groupName, String decliner, String reason);

	/**
     * \~english
     * Occurs when a group invitation is accepted.
	 *
	 * @param groupId 		The group ID.
	 * @param invitee 		The invitee ID.
	 * @param reason		The reason for acceptance.
	 */
	void onInvitationAccepted(String groupId, String invitee, String reason);
    
	/**
     * \~english
     * Occurs when a group invitation is declined.
	 *
	 * @param groupId 		The group ID.
	 * @param invitee		The invitee ID.
	 * @param reason 		The reason for declining.
	 */
	void onInvitationDeclined(String groupId, String invitee, String reason);
    
	/**
     * \~english
     * Occurs when the current user is removed from the group by the group admin.
	 *
	 * @param groupId 		The group ID.
	 * @param groupName		The group name.
	 */
	void onUserRemoved(String groupId, String groupName);
    
    /**
     * \~english
     * Occurs when a group is destroyed.
     * The SDK will remove the group from the local database and local cache before notifying the app of the group removal.
	 *
	 * @param groupId		The group ID.
	 * @param groupName 	The group name.
     */
	void onGroupDestroyed(String groupId, String groupName);
    
    /**
     * \~english
     * Occurs when the group invitation is accepted automatically.
	 * The SDK will join the group before notifying the app of the acceptance of the group invitation.
     * For settings, see {@link io.agora.chat.ChatOptions#setAutoAcceptGroupInvitation(boolean value)}.
	 *
	 * @param groupId			The group ID.
	 * @param inviter			The inviter ID.
	 * @param inviteMessage		The invitation message.
     */
	void onAutoAcceptInvitationFromGroup(String groupId, String inviter, String inviteMessage);

	/**
	 * \~english
	 * Occurs when one or more group members are muted.
	 * Note: The mute function is different from a blocklist. A user, when muted, can still see group messages, but cannot send messages in the group. However, a user on the blocklist can neither see nor send group messages.
	 *
	 * @param groupId		The group ID.
	 * @param mutes 		The member(s) added to the mute list.
	 *	 			        Map.entry.key is the muted username; Map.entry.value is the mute duration in milliseconds.
	 */
	void onMuteListAdded(String groupId, final List<String> mutes, final long muteExpire);

	/**
	 * \~english
	 * Occurs when one or more group members are unmuted.
	 * Note: The mute function is different from a blocklist. A user, when muted, still can see group messages, but cannot send messages in the group. However, a user on the blocklist can neither see nor send group messages.
	 *
	 * @param groupId		The group ID.
	 * @param mutes 		The member(s) removed from the mute list.
	 */
	void onMuteListRemoved(String groupId, final List<String> mutes);

	/**
	 * \~english
	 * Occurs when one or more group members are added to the allowlist.
	 *
	 * @param groupId    	The group ID.
	 * @param whitelist     The member(s) added to the allowlist.
	 */
	void onWhiteListAdded(final String groupId, final List<String> whitelist);

	/**
	 * \~english
	 * Occurs when one or more members are removed from the allowlist.
	 *
	 * @param groupId    	The group ID.
	 * @param whitelist     Member(s) removed from the allowlist.
	 */
	void onWhiteListRemoved(final String groupId, final List<String> whitelist);

	/**
	 * \~english
	 * Occurs when all group members are muted or unmuted.
	 *
	 * @param groupId    	The group ID.
	 * @param isMuted       Whether all group members are muted or unmuted. true: all group members are muted; false: all group members are unmuted.
	 */
	void onAllMemberMuteStateChanged(final String groupId, final boolean isMuted);

	/**
	 * \~english
	 * Occurs when a member is set as an admin.
	 *
	 * @param groupId		The group ID.
	 * @param administrator The member that is set as an admin.
	 */
	void onAdminAdded(String groupId, String administrator);

	/**
	 * \~english
	 * Occurs when a member's admin privileges are removed.
	 *
	 * @param groupId 		The group ID.
	 * @param administrator The member whose admin privileges are removed.
	 */
	void onAdminRemoved(String groupId, String administrator);

	/**
	 * \~english
	 * Occurs when the group ownership is transferred.
	 * @param groupId 		The group ID.
	 * @param newOwner 		The new owner.
	 * @param oldOwner 		The previous owner.
	 */
	void onOwnerChanged(String groupId, String newOwner, String oldOwner);
	
	/**
     * \~english
     * Occurs when a member joins a group.
     * 
     * @param groupId  The group ID.
     * @param member   The ID of the new member.
     */
    void onMemberJoined(final String groupId, final String member);

    /**
     * \~english
     * Occurs when a member proactively leaves the group.
     * 
     * @param groupId   The group ID.
     * @param member  	The member leaving the group.
      */
    void onMemberExited(final String groupId,  final String member);

	/**
	 * \~english
	 * Occurs when the announcement is updated.
	 * @param groupId The group ID.
	 * @param announcement The updated announcement content.
	 */
	void onAnnouncementChanged(String groupId, String announcement);

	/**
	 * \~english
	 * Occurs when a shared file is added to a group.
	 * @param groupId The group ID.
	 * @param sharedFile The new shared file.
	 */
	void onSharedFileAdded(String groupId, MucSharedFile sharedFile);

	/**
	 * \~english
	 * Occurs when a shared file is removed from a group.
	 * @param groupId The group ID.
	 * @param fileId  The ID of the removed shared file.
	 */
	void onSharedFileDeleted(String groupId, String fileId);

	/**
	 * \~english
	 * Occurs when the group detail information is updated,need to call{@link io.agora.chat.GroupManager#asyncGetGroupFromServer(String, ValueCallBack)} to get the latest group information.
	 * @param group The group.
	 */
	default void onSpecificationChanged(Group group){}

	/**
	 * \~english
	 * Occurs when the group is enabled or disabled.
	 * @param group The group.
	 * @param isDisabled  Whether the group is disabled.
	 */
	default void onStateChanged(Group group,  boolean isDisabled){}

	/**
	 * 群组成员自定义属性有变更。
	 * 
	 * @param groupId		群组 ID。
	 * @param userId		自定义属性变更的群成员的用户 ID。
	 * @param attribute		修改后的自定义属性，key-value 格式。
	 * @param from			操作者的用户 ID。
	 *
	 * \~english
	 * Occurs when a custom attribute(s) of a group member is/are changed.
	 * 
	 * @param groupId		The group ID.
	 * @param userId		The user ID of the group member whose custom attributes are changed.
	 * @param attribute		The modified custom attributes, in key-value format.
	 * @param from			The user ID of the operator.
	 */
	default void onGroupMemberAttributeChanged(String groupId,String userId, Map<String,String> attribute , String from){}
}
