/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora;

import io.agora.chat.ChatRoom;

import java.util.List;
import java.util.Map;

/**
 * \~english
 * The chat room change listener.
 * 
 * You can add or remove a chat room change listener by using {@link io.agora.chat.ChatRoomManager}.
 * 
 * Add a chat room change listener:
 * ```java
 * ChatRoomManager manager = ChatClient.getInstance().chatroomManager();
 * manager.addChatRoomChangeListener(mChatRoomChangeListener);
 * ```
 *
 * Remove a chat room change listener:
 * ```java
 * ChatRoomManager manager = ChatClient.getInstance().chatroomManager();
 * manager.removeChatRoomListener(mChatRoomChangeListener);
 * ```
 */
public interface ChatRoomChangeListener {
  /**
   * \~english
   * Occurs when the chat room is destroyed.
   * 
   * All chat room members receive this event.
   * 
   * @param roomId   The chat room ID.
   * @param roomName The chat room name.
   */
  void onChatRoomDestroyed(final String roomId, final String roomName);

  /**
   * \~english
   * Occurs when a member joins the chat room.
   * 
   * All chat room members, except the new member, receive this event.
   * 
   * @param roomId      The chat room ID.
   * @param participant The new member.
   */
  void onMemberJoined(final String roomId, final String participant);

  /**
   * \~english
   * Occurs when a member exits the chat room.
   * 
   * All chat room members, except the member exiting the chat room, receive this event.
   * 
   * @param roomId      The chat room ID.
   * @param roomName    The chat room name.
   * @param participant The member exiting the chat room.
   */
  void onMemberExited(final String roomId, final String roomName, final String participant);

  /**
   * \~english
   * Occurs when a member is removed from a chat room.
   * 
   * The member that is kicked out of the chat room receive this event.
   *
   * @param reason      The reason why the member is removed from the chat room:
   *                    - xxx BE_KICKED: The member is removed by the chat room owner.
   *                    - xxx BE_KICKED_FOR_OFFLINE: The member is disconnected from the server, probably due to network interruption.
   * @param roomId      The chat room ID.
   * @param roomName    The chat room name.
   * @param participant The member removed from a chat room.
   */
  void onRemovedFromChatRoom(final int reason, final String roomId, final String roomName, final String participant);

  /**
   * \~english
   * Occurs when the chat room member(s) is/are added to the mute list.
   * 
   * The muted members receive this event. The muted members cannot send message during the mute duration.
   *
   * @param chatRoomId The chat room ID.
   * @param mutes      The muted number(s).
   * @param expireTime The Unix timestamp when the mute expires. The unit is millisecond. Reserved parameter.
   */
  void onMuteListAdded(final String chatRoomId, final List<String> mutes, final long expireTime);

  /**
   * \~english
   * Occurs when the chat room member(s) is/are removed from the mute list.
   * 
   * The members that are removed from the mute list receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param mutes      The member(s) removed from the mute list.
   */
  void onMuteListRemoved(final String chatRoomId, final List<String> mutes);

  /**
   * \~english
   * Occurs when the chat room member(s) is/are added to the allow list.
   * 
   * The members added to the allow list receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param whitelist  The member(s) added to the allow list.
   */
  void onWhiteListAdded(final String chatRoomId, final List<String> whitelist);

  /**
   * \~english
   * Occurs when the chat room member(s) is/are removed from the allow list.
   * 
   * The members that are removed from the allow list receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param whitelist  The member(s) removed from the allow list.
   */
  void onWhiteListRemoved(final String chatRoomId, final List<String> whitelist);

  /**
   * \~english
   * Occurs when all members in the chat room are muted or unmuted.
   * 
   * All chat room members receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param isMuted    Whether all chat room members are muted or unmuted.
   */
  void onAllMemberMuteStateChanged(final String chatRoomId, final boolean isMuted);

  /**
   * \~english
   * Occurs when a chat room member is set as an admin.
   * 
   * The member set as the chat room admin receives this event.
   *
   * @param chatRoomId The chat room ID.
   * @param admin      The chat room member set as an admin.
   */
  void onAdminAdded(final String chatRoomId, final String admin);

  /**
   * \~english
   * Occurs when the chat room member(s) is/are removed from the admin list.
   * 
   * The admin removed from the admin list receives this event.
   *
   * @param chatRoomId The chat room ID.
   * @param admin      The member(s) removed from the admin list.
   */
  void onAdminRemoved(final String chatRoomId, final String admin);

  /**
   * \~english
   *  Occurs when the chat room owner is changed.
   * 
   *  The chat room owner receives this event.
   *
   * @param chatRoomId The chat room ID.
   * @param newOwner   The new chat room owner.
   * @param oldOwner   The previous chat room owner.
   */
  void onOwnerChanged(final String chatRoomId, final String newOwner, final String oldOwner);

  /**
   * \~english
   * Occurs when the chat room announcement changes.
   * 
   * All chat room members receive this event.
   * 
   * @param chatRoomId   The chat room ID.
   * @param announcement The modified chat room announcement.
   */
  void onAnnouncementChanged(String chatRoomId, String announcement);

  /**
   * \~english
   * Occurs when the chat room specifications changes.
   * 
   * All chat room members receive this event.
   * 
   * @param chatRoom The chat room instance.
   */
  default void onSpecificationChanged(ChatRoom chatRoom) {}

  /**
   * \~english
   * The custom chat room attribute(s) is/are updated.
   * 
   * All chat room members receive this event.
   *
   * @param chatRoomId   The chat room ID.
   * @param attributeMap The map of custom chat room attributes.
   * @param from         The user ID of the operator.
   */
  default void onAttributesUpdate(String chatRoomId, Map<String, String> attributeMap, String from) {}

  /**
   * \~english
   * The custom chat room attribute(s) is/are removed.
   * 
   * All chat room members receive this event.
   *
   * @param chatRoomId   The chat room ID.
   * @param keyList      The list of custom chat room attributes key.
   * @param from         The user ID of the operator.
   */

    default void onAttributesRemoved(String chatRoomId, List<String> keyList , String from){}

}
