/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.build.api.variant.impl

import com.android.build.api.artifact.impl.ArtifactsImpl
import com.android.build.api.component.analytics.AnalyticsEnabledApplicationVariantProperties
import com.android.build.api.component.impl.ApkCreationConfigImpl
import com.android.build.api.variant.AaptOptions
import com.android.build.api.variant.AndroidVersion
import com.android.build.api.variant.ApkPackagingOptions
import com.android.build.api.variant.ApplicationVariantProperties
import com.android.build.api.variant.DependenciesInfo
import com.android.build.api.variant.SigningConfig
import com.android.build.gradle.internal.component.ApplicationCreationConfig
import com.android.build.gradle.internal.core.VariantDslInfo
import com.android.build.gradle.internal.core.VariantSources
import com.android.build.gradle.internal.dependency.VariantDependencies
import com.android.build.gradle.internal.pipeline.TransformManager
import com.android.build.gradle.internal.scope.BuildFeatureValues
import com.android.build.gradle.internal.scope.GlobalScope
import com.android.build.gradle.internal.scope.VariantScope
import com.android.build.gradle.internal.services.ProjectServices
import com.android.build.gradle.internal.services.TaskCreationServices
import com.android.build.gradle.internal.services.VariantPropertiesApiServices
import com.android.build.gradle.internal.variant.BaseVariantData
import com.android.build.gradle.internal.variant.VariantPathHelper
import com.android.build.gradle.options.IntegerOption
import com.android.builder.dexing.DexingType
import com.android.builder.model.CodeShrinker
import com.google.wireless.android.sdk.stats.GradleBuildVariant
import com.android.build.gradle.options.StringOption
import org.gradle.api.provider.Property
import javax.inject.Inject

open class ApplicationVariantPropertiesImpl @Inject constructor(
    override val variant: ApplicationVariantImpl,
    buildFeatureValues: BuildFeatureValues,
    variantDslInfo: VariantDslInfo,
    variantDependencies: VariantDependencies,
    variantSources: VariantSources,
    paths: VariantPathHelper,
    artifacts: ArtifactsImpl,
    variantScope: VariantScope,
    variantData: BaseVariantData,
    variantDependencyInfo: DependenciesInfo,
    transformManager: TransformManager,
    internalServices: VariantPropertiesApiServices,
    taskCreationServices: TaskCreationServices,
    globalScope: GlobalScope
) : VariantPropertiesImpl(
    variant,
    buildFeatureValues,
    variantDslInfo,
    variantDependencies,
    variantSources,
    paths,
    artifacts,
    variantScope,
    variantData,
    transformManager,
    internalServices,
    taskCreationServices,
    globalScope
), ApplicationVariantProperties, ApplicationCreationConfig {

    val delegate by lazy { ApkCreationConfigImpl(this, globalScope, variantDslInfo) }

    // ---------------------------------------------------------------------------------------------
    // PUBLIC API
    // ---------------------------------------------------------------------------------------------

    override val applicationId: Property<String> = variantDslInfo.applicationId

    override val embedsMicroApp: Boolean
        get() = variantDslInfo.isEmbedMicroApp

    override val dependenciesInfo: DependenciesInfo = variantDependencyInfo

    override val aaptOptions: AaptOptions by lazy {
        initializeAaptOptionsFromDsl(
            globalScope.extension.aaptOptions,
            internalServices
        )
    }

    override fun aaptOptions(action: AaptOptions.() -> Unit) {
        action.invoke(aaptOptions)
    }

    override val signingConfig: SigningConfig by lazy {
        SigningConfigImpl(
            variantDslInfo.signingConfig,
            internalServices,
            minSdkVersion.apiLevel,
            globalScope.projectOptions.get(IntegerOption.IDE_TARGET_DEVICE_API)
        )
    }

    override fun signingConfig(action: SigningConfig.() -> Unit) {
        action.invoke(signingConfig)
    }

    override val packagingOptions: ApkPackagingOptions by lazy {
        ApkPackagingOptionsImpl(
            globalScope.extension.packagingOptions,
            internalServices,
            minSdkVersion.apiLevel
        )
    }

    override fun packagingOptions(action: ApkPackagingOptions.() -> Unit) {
        action.invoke(packagingOptions)
    }

    override val minifiedEnabled: Boolean
        get() = variantDslInfo.isMinifyEnabled

    // ---------------------------------------------------------------------------------------------
    // INTERNAL API
    // ---------------------------------------------------------------------------------------------

    override val testOnlyApk: Boolean
        get() = variantScope.isTestOnly(this)

    override val needAssetPackTasks: Property<Boolean> =
        internalServices.propertyOf(Boolean::class.java, false)

    override val shouldPackageDesugarLibDex: Boolean
        get() = delegate.isCoreLibraryDesugaringEnabled
    override val debuggable: Boolean
        get() = delegate.isDebuggable

    override val shouldPackageProfilerDependencies: Boolean
        get() = advancedProfilingTransforms.isNotEmpty()

    override val advancedProfilingTransforms: List<String>
        get() {
            return services.projectOptions[StringOption.IDE_ANDROID_CUSTOM_CLASS_TRANSFORMS]?.split(
                ","
            ) ?: emptyList()
        }

    // ---------------------------------------------------------------------------------------------
    // Private stuff
    // ---------------------------------------------------------------------------------------------

    override fun createVersionNameProperty(): Property<String?> =
        internalServices.newNullablePropertyBackingDeprecatedApi(
            String::class.java,
            variantDslInfo.versionName,
            "$name::versionName"
        )

    override fun createVersionCodeProperty() : Property<Int?> =
        internalServices.newNullablePropertyBackingDeprecatedApi(
            Int::class.java,
            variantDslInfo.versionCode,
            "$name::versionCode"
        )

    override val renderscriptTargetApi: Int
        get() {
            return variant.renderscriptTargetApi
        }

    override val dexingType: DexingType
        get() = delegate.dexingType

    override val needsMainDexListForBundle: Boolean
        get() = (variantType.isBaseModule
                    && globalScope.hasDynamicFeatures()
                    && dexingType.needsMainDexList)

    override fun createUserVisibleVariantPropertiesObject(
        projectServices: ProjectServices,
        stats: GradleBuildVariant.Builder
    ): AnalyticsEnabledApplicationVariantProperties =
        projectServices.objectFactory.newInstance(
            AnalyticsEnabledApplicationVariantProperties::class.java,
            this,
            stats
        )
    override val minSdkVersionWithTargetDeviceApi: AndroidVersion
        get() = delegate.minSdkVersionWithTargetDeviceApi

    override val codeShrinker: CodeShrinker?
        get() = delegate.getCodeShrinker()

    override fun getNeedsMergedJavaResStream(): Boolean = delegate.getNeedsMergedJavaResStream()

    override fun getJava8LangSupportType(): VariantScope.Java8LangSupport = delegate.getJava8LangSupportType()

    override val needsShrinkDesugarLibrary: Boolean
        get() = delegate.needsShrinkDesugarLibrary


}
