/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.internal.tasks

import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.component.ModuleComponentIdentifier
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.OutputDirectory
import java.io.File

/**
 * Abstract class used to compare two configurations, and to report differences in versions of
 * artifacts. This is useful to warn user about potential issues that could arise from such
 * differences. E.g. for application, differences in runtime and compile classpath could result in
 * runtime failure.
 */
abstract class ClasspathComparisonTask : NonIncrementalTask() {

    @get:Internal
    protected lateinit var runtimeClasspath: Configuration

    @get:Input
    val runtimeVersionMap: Map<Info, String> by lazy {
        runtimeClasspath.toVersionMap()
    }

    @get:Internal
    protected lateinit var compileClasspath: Configuration

    @get:Input
    val compileVersionMap: Map<Info, String> by lazy {
        compileClasspath.toVersionMap()
    }

    // fake output dir so that the task doesn't run unless an input has changed.
    @get:OutputDirectory
    var fakeOutputDirectory: File? = null
        protected set

    protected abstract fun onDifferentVersionsFound(
        group: String,
        module: String,
        runtimeVersion: String,
        compileVersion: String
    )

    override fun doTaskAction() {
        val runtimeMap = runtimeVersionMap
        for (compileEntry in compileVersionMap.entries) {
            val runtimeVersion = runtimeMap[compileEntry.key] ?: continue

            if (runtimeVersion != compileEntry.value) {
                onDifferentVersionsFound(
                    compileEntry.key.group,
                    compileEntry.key.module,
                    runtimeVersion,
                    compileEntry.value
                )
            }
        }
    }
}

data class Info(
    val group: String,
    val module: String
)

private fun Configuration.toVersionMap(): Map<Info, String> =
    incoming.resolutionResult.allComponents
        .asSequence()
        .filter { it.id is ModuleComponentIdentifier }
        .map { it.id as ModuleComponentIdentifier }
        .map { Info(it.group, it.module) to it.version }
        .toMap()
