/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.schemas.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.schemas.SchemasAsyncClient;
import software.amazon.awssdk.services.schemas.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.schemas.model.DescribeCodeBindingRequest;
import software.amazon.awssdk.services.schemas.model.DescribeCodeBindingResponse;
import software.amazon.awssdk.services.schemas.model.SchemasRequest;
import software.amazon.awssdk.services.schemas.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultSchemasAsyncWaiter implements SchemasAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final SchemasAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeCodeBindingResponse> codeBindingExistsWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultSchemasAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = SchemasAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.codeBindingExistsWaiter = AsyncWaiter.builder(DescribeCodeBindingResponse.class)
                .acceptors(codeBindingExistsWaiterAcceptors())
                .overrideConfiguration(codeBindingExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCodeBindingResponse>> waitUntilCodeBindingExists(
            DescribeCodeBindingRequest describeCodeBindingRequest) {
        return codeBindingExistsWaiter.runAsync(() -> client
                .describeCodeBinding(applyWaitersUserAgent(describeCodeBindingRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCodeBindingResponse>> waitUntilCodeBindingExists(
            DescribeCodeBindingRequest describeCodeBindingRequest, WaiterOverrideConfiguration overrideConfig) {
        return codeBindingExistsWaiter.runAsync(
                () -> client.describeCodeBinding(applyWaitersUserAgent(describeCodeBindingRequest)),
                codeBindingExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeCodeBindingResponse>> codeBindingExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCodeBindingResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "CREATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "CREATE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NotFoundException"),
                "A waiter acceptor was matched on error condition (NotFoundException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration codeBindingExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static SchemasAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends SchemasRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements SchemasAsyncWaiter.Builder {
        private SchemasAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public SchemasAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public SchemasAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public SchemasAsyncWaiter.Builder client(SchemasAsyncClient client) {
            this.client = client;
            return this;
        }

        public SchemasAsyncWaiter build() {
            return new DefaultSchemasAsyncWaiter(this);
        }
    }
}
