/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The collection of settings used by an AutoML job V2 for the text generation problem type.
 * </p>
 * <note>
 * <p>
 * The text generation models that support fine-tuning in Autopilot are currently accessible exclusively in regions
 * supported by Canvas. Refer to the documentation of Canvas for the <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/canvas.html">full list of its supported Regions</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TextGenerationJobConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<TextGenerationJobConfig.Builder, TextGenerationJobConfig> {
    private static final SdkField<AutoMLJobCompletionCriteria> COMPLETION_CRITERIA_FIELD = SdkField
            .<AutoMLJobCompletionCriteria> builder(MarshallingType.SDK_POJO).memberName("CompletionCriteria")
            .getter(getter(TextGenerationJobConfig::completionCriteria)).setter(setter(Builder::completionCriteria))
            .constructor(AutoMLJobCompletionCriteria::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionCriteria").build())
            .build();

    private static final SdkField<String> BASE_MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BaseModelName").getter(getter(TextGenerationJobConfig::baseModelName))
            .setter(setter(Builder::baseModelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseModelName").build()).build();

    private static final SdkField<Map<String, String>> TEXT_GENERATION_HYPER_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("TextGenerationHyperParameters")
            .getter(getter(TextGenerationJobConfig::textGenerationHyperParameters))
            .setter(setter(Builder::textGenerationHyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextGenerationHyperParameters")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ModelAccessConfig> MODEL_ACCESS_CONFIG_FIELD = SdkField
            .<ModelAccessConfig> builder(MarshallingType.SDK_POJO).memberName("ModelAccessConfig")
            .getter(getter(TextGenerationJobConfig::modelAccessConfig)).setter(setter(Builder::modelAccessConfig))
            .constructor(ModelAccessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelAccessConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPLETION_CRITERIA_FIELD,
            BASE_MODEL_NAME_FIELD, TEXT_GENERATION_HYPER_PARAMETERS_FIELD, MODEL_ACCESS_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final AutoMLJobCompletionCriteria completionCriteria;

    private final String baseModelName;

    private final Map<String, String> textGenerationHyperParameters;

    private final ModelAccessConfig modelAccessConfig;

    private TextGenerationJobConfig(BuilderImpl builder) {
        this.completionCriteria = builder.completionCriteria;
        this.baseModelName = builder.baseModelName;
        this.textGenerationHyperParameters = builder.textGenerationHyperParameters;
        this.modelAccessConfig = builder.modelAccessConfig;
    }

    /**
     * <p>
     * How long a fine-tuning job is allowed to run. For <code>TextGenerationJobConfig</code> problem types, the
     * <code>MaxRuntimePerTrainingJobInSeconds</code> attribute of <code>AutoMLJobCompletionCriteria</code> defaults to
     * 72h (259200s).
     * </p>
     * 
     * @return How long a fine-tuning job is allowed to run. For <code>TextGenerationJobConfig</code> problem types, the
     *         <code>MaxRuntimePerTrainingJobInSeconds</code> attribute of <code>AutoMLJobCompletionCriteria</code>
     *         defaults to 72h (259200s).
     */
    public final AutoMLJobCompletionCriteria completionCriteria() {
        return completionCriteria;
    }

    /**
     * <p>
     * The name of the base model to fine-tune. Autopilot supports fine-tuning a variety of large language models. For
     * information on the list of supported models, see <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-models.html#autopilot-llms-finetuning-supported-llms"
     * >Text generation models supporting fine-tuning in Autopilot</a>. If no <code>BaseModelName</code> is provided,
     * the default model used is <b>Falcon7BInstruct</b>.
     * </p>
     * 
     * @return The name of the base model to fine-tune. Autopilot supports fine-tuning a variety of large language
     *         models. For information on the list of supported models, see <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-models.html#autopilot-llms-finetuning-supported-llms"
     *         >Text generation models supporting fine-tuning in Autopilot</a>. If no <code>BaseModelName</code> is
     *         provided, the default model used is <b>Falcon7BInstruct</b>.
     */
    public final String baseModelName() {
        return baseModelName;
    }

    /**
     * For responses, this returns true if the service returned a value for the TextGenerationHyperParameters property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasTextGenerationHyperParameters() {
        return textGenerationHyperParameters != null && !(textGenerationHyperParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The hyperparameters used to configure and optimize the learning process of the base model. You can set any
     * combination of the following hyperparameters for all base models. For more information on each supported
     * hyperparameter, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-set-hyperparameters.html"
     * >Optimize the learning process of your text generation models with hyperparameters</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"epochCount"</code>: The number of times the model goes through the entire training dataset. Its value
     * should be a string containing an integer value within the range of "1" to "10".
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"batchSize"</code>: The number of data samples used in each iteration of training. Its value should be a
     * string containing an integer value within the range of "1" to "64".
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"learningRate"</code>: The step size at which a model's parameters are updated during training. Its value
     * should be a string containing a floating-point value within the range of "0" to "1".
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"learningRateWarmupSteps"</code>: The number of training steps during which the learning rate gradually
     * increases before reaching its target or maximum value. Its value should be a string containing an integer value
     * within the range of "0" to "250".
     * </p>
     * </li>
     * </ul>
     * <p>
     * Here is an example where all four hyperparameters are configured.
     * </p>
     * <p>
     * <code>{ "epochCount":"5", "learningRate":"0.5", "batchSize": "32", "learningRateWarmupSteps": "10" }</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTextGenerationHyperParameters} method.
     * </p>
     * 
     * @return The hyperparameters used to configure and optimize the learning process of the base model. You can set
     *         any combination of the following hyperparameters for all base models. For more information on each
     *         supported hyperparameter, see <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-set-hyperparameters.html"
     *         >Optimize the learning process of your text generation models with hyperparameters</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"epochCount"</code>: The number of times the model goes through the entire training dataset. Its
     *         value should be a string containing an integer value within the range of "1" to "10".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"batchSize"</code>: The number of data samples used in each iteration of training. Its value should
     *         be a string containing an integer value within the range of "1" to "64".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"learningRate"</code>: The step size at which a model's parameters are updated during training. Its
     *         value should be a string containing a floating-point value within the range of "0" to "1".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"learningRateWarmupSteps"</code>: The number of training steps during which the learning rate
     *         gradually increases before reaching its target or maximum value. Its value should be a string containing
     *         an integer value within the range of "0" to "250".
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Here is an example where all four hyperparameters are configured.
     *         </p>
     *         <p>
     *         <code>{ "epochCount":"5", "learningRate":"0.5", "batchSize": "32", "learningRateWarmupSteps": "10" }</code>
     */
    public final Map<String, String> textGenerationHyperParameters() {
        return textGenerationHyperParameters;
    }

    /**
     * Returns the value of the ModelAccessConfig property for this object.
     * 
     * @return The value of the ModelAccessConfig property for this object.
     */
    public final ModelAccessConfig modelAccessConfig() {
        return modelAccessConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(completionCriteria());
        hashCode = 31 * hashCode + Objects.hashCode(baseModelName());
        hashCode = 31 * hashCode + Objects.hashCode(hasTextGenerationHyperParameters() ? textGenerationHyperParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(modelAccessConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TextGenerationJobConfig)) {
            return false;
        }
        TextGenerationJobConfig other = (TextGenerationJobConfig) obj;
        return Objects.equals(completionCriteria(), other.completionCriteria())
                && Objects.equals(baseModelName(), other.baseModelName())
                && hasTextGenerationHyperParameters() == other.hasTextGenerationHyperParameters()
                && Objects.equals(textGenerationHyperParameters(), other.textGenerationHyperParameters())
                && Objects.equals(modelAccessConfig(), other.modelAccessConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("TextGenerationJobConfig")
                .add("CompletionCriteria", completionCriteria())
                .add("BaseModelName", baseModelName())
                .add("TextGenerationHyperParameters", hasTextGenerationHyperParameters() ? textGenerationHyperParameters() : null)
                .add("ModelAccessConfig", modelAccessConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CompletionCriteria":
            return Optional.ofNullable(clazz.cast(completionCriteria()));
        case "BaseModelName":
            return Optional.ofNullable(clazz.cast(baseModelName()));
        case "TextGenerationHyperParameters":
            return Optional.ofNullable(clazz.cast(textGenerationHyperParameters()));
        case "ModelAccessConfig":
            return Optional.ofNullable(clazz.cast(modelAccessConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TextGenerationJobConfig, T> g) {
        return obj -> g.apply((TextGenerationJobConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TextGenerationJobConfig> {
        /**
         * <p>
         * How long a fine-tuning job is allowed to run. For <code>TextGenerationJobConfig</code> problem types, the
         * <code>MaxRuntimePerTrainingJobInSeconds</code> attribute of <code>AutoMLJobCompletionCriteria</code> defaults
         * to 72h (259200s).
         * </p>
         * 
         * @param completionCriteria
         *        How long a fine-tuning job is allowed to run. For <code>TextGenerationJobConfig</code> problem types,
         *        the <code>MaxRuntimePerTrainingJobInSeconds</code> attribute of
         *        <code>AutoMLJobCompletionCriteria</code> defaults to 72h (259200s).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionCriteria(AutoMLJobCompletionCriteria completionCriteria);

        /**
         * <p>
         * How long a fine-tuning job is allowed to run. For <code>TextGenerationJobConfig</code> problem types, the
         * <code>MaxRuntimePerTrainingJobInSeconds</code> attribute of <code>AutoMLJobCompletionCriteria</code> defaults
         * to 72h (259200s).
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoMLJobCompletionCriteria.Builder}
         * avoiding the need to create one manually via {@link AutoMLJobCompletionCriteria#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoMLJobCompletionCriteria.Builder#build()} is called
         * immediately and its result is passed to {@link #completionCriteria(AutoMLJobCompletionCriteria)}.
         * 
         * @param completionCriteria
         *        a consumer that will call methods on {@link AutoMLJobCompletionCriteria.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #completionCriteria(AutoMLJobCompletionCriteria)
         */
        default Builder completionCriteria(Consumer<AutoMLJobCompletionCriteria.Builder> completionCriteria) {
            return completionCriteria(AutoMLJobCompletionCriteria.builder().applyMutation(completionCriteria).build());
        }

        /**
         * <p>
         * The name of the base model to fine-tune. Autopilot supports fine-tuning a variety of large language models.
         * For information on the list of supported models, see <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-models.html#autopilot-llms-finetuning-supported-llms"
         * >Text generation models supporting fine-tuning in Autopilot</a>. If no <code>BaseModelName</code> is
         * provided, the default model used is <b>Falcon7BInstruct</b>.
         * </p>
         * 
         * @param baseModelName
         *        The name of the base model to fine-tune. Autopilot supports fine-tuning a variety of large language
         *        models. For information on the list of supported models, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-models.html#autopilot-llms-finetuning-supported-llms"
         *        >Text generation models supporting fine-tuning in Autopilot</a>. If no <code>BaseModelName</code> is
         *        provided, the default model used is <b>Falcon7BInstruct</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseModelName(String baseModelName);

        /**
         * <p>
         * The hyperparameters used to configure and optimize the learning process of the base model. You can set any
         * combination of the following hyperparameters for all base models. For more information on each supported
         * hyperparameter, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-set-hyperparameters.html"
         * >Optimize the learning process of your text generation models with hyperparameters</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"epochCount"</code>: The number of times the model goes through the entire training dataset. Its value
         * should be a string containing an integer value within the range of "1" to "10".
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"batchSize"</code>: The number of data samples used in each iteration of training. Its value should be
         * a string containing an integer value within the range of "1" to "64".
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"learningRate"</code>: The step size at which a model's parameters are updated during training. Its
         * value should be a string containing a floating-point value within the range of "0" to "1".
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"learningRateWarmupSteps"</code>: The number of training steps during which the learning rate gradually
         * increases before reaching its target or maximum value. Its value should be a string containing an integer
         * value within the range of "0" to "250".
         * </p>
         * </li>
         * </ul>
         * <p>
         * Here is an example where all four hyperparameters are configured.
         * </p>
         * <p>
         * <code>{ "epochCount":"5", "learningRate":"0.5", "batchSize": "32", "learningRateWarmupSteps": "10" }</code>
         * </p>
         * 
         * @param textGenerationHyperParameters
         *        The hyperparameters used to configure and optimize the learning process of the base model. You can set
         *        any combination of the following hyperparameters for all base models. For more information on each
         *        supported hyperparameter, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-llms-finetuning-set-hyperparameters.html"
         *        >Optimize the learning process of your text generation models with hyperparameters</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"epochCount"</code>: The number of times the model goes through the entire training dataset. Its
         *        value should be a string containing an integer value within the range of "1" to "10".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"batchSize"</code>: The number of data samples used in each iteration of training. Its value
         *        should be a string containing an integer value within the range of "1" to "64".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"learningRate"</code>: The step size at which a model's parameters are updated during training.
         *        Its value should be a string containing a floating-point value within the range of "0" to "1".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"learningRateWarmupSteps"</code>: The number of training steps during which the learning rate
         *        gradually increases before reaching its target or maximum value. Its value should be a string
         *        containing an integer value within the range of "0" to "250".
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Here is an example where all four hyperparameters are configured.
         *        </p>
         *        <p>
         *        <code>{ "epochCount":"5", "learningRate":"0.5", "batchSize": "32", "learningRateWarmupSteps": "10" }</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textGenerationHyperParameters(Map<String, String> textGenerationHyperParameters);

        /**
         * Sets the value of the ModelAccessConfig property for this object.
         *
         * @param modelAccessConfig
         *        The new value for the ModelAccessConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelAccessConfig(ModelAccessConfig modelAccessConfig);

        /**
         * Sets the value of the ModelAccessConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ModelAccessConfig.Builder} avoiding the
         * need to create one manually via {@link ModelAccessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelAccessConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #modelAccessConfig(ModelAccessConfig)}.
         * 
         * @param modelAccessConfig
         *        a consumer that will call methods on {@link ModelAccessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelAccessConfig(ModelAccessConfig)
         */
        default Builder modelAccessConfig(Consumer<ModelAccessConfig.Builder> modelAccessConfig) {
            return modelAccessConfig(ModelAccessConfig.builder().applyMutation(modelAccessConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private AutoMLJobCompletionCriteria completionCriteria;

        private String baseModelName;

        private Map<String, String> textGenerationHyperParameters = DefaultSdkAutoConstructMap.getInstance();

        private ModelAccessConfig modelAccessConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(TextGenerationJobConfig model) {
            completionCriteria(model.completionCriteria);
            baseModelName(model.baseModelName);
            textGenerationHyperParameters(model.textGenerationHyperParameters);
            modelAccessConfig(model.modelAccessConfig);
        }

        public final AutoMLJobCompletionCriteria.Builder getCompletionCriteria() {
            return completionCriteria != null ? completionCriteria.toBuilder() : null;
        }

        public final void setCompletionCriteria(AutoMLJobCompletionCriteria.BuilderImpl completionCriteria) {
            this.completionCriteria = completionCriteria != null ? completionCriteria.build() : null;
        }

        @Override
        public final Builder completionCriteria(AutoMLJobCompletionCriteria completionCriteria) {
            this.completionCriteria = completionCriteria;
            return this;
        }

        public final String getBaseModelName() {
            return baseModelName;
        }

        public final void setBaseModelName(String baseModelName) {
            this.baseModelName = baseModelName;
        }

        @Override
        public final Builder baseModelName(String baseModelName) {
            this.baseModelName = baseModelName;
            return this;
        }

        public final Map<String, String> getTextGenerationHyperParameters() {
            if (textGenerationHyperParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return textGenerationHyperParameters;
        }

        public final void setTextGenerationHyperParameters(Map<String, String> textGenerationHyperParameters) {
            this.textGenerationHyperParameters = TextGenerationHyperParametersCopier.copy(textGenerationHyperParameters);
        }

        @Override
        public final Builder textGenerationHyperParameters(Map<String, String> textGenerationHyperParameters) {
            this.textGenerationHyperParameters = TextGenerationHyperParametersCopier.copy(textGenerationHyperParameters);
            return this;
        }

        public final ModelAccessConfig.Builder getModelAccessConfig() {
            return modelAccessConfig != null ? modelAccessConfig.toBuilder() : null;
        }

        public final void setModelAccessConfig(ModelAccessConfig.BuilderImpl modelAccessConfig) {
            this.modelAccessConfig = modelAccessConfig != null ? modelAccessConfig.build() : null;
        }

        @Override
        public final Builder modelAccessConfig(ModelAccessConfig modelAccessConfig) {
            this.modelAccessConfig = modelAccessConfig;
            return this;
        }

        @Override
        public TextGenerationJobConfig build() {
            return new TextGenerationJobConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
