/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.ram.RamClient;
import software.amazon.awssdk.services.ram.internal.UserAgentUtils;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest;
import software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.ram.RamClient#listResourceSharePermissionsPaginator(software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ram.paginators.ListResourceSharePermissionsIterable responses = client.listResourceSharePermissionsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.ram.paginators.ListResourceSharePermissionsIterable responses = client
 *             .listResourceSharePermissionsPaginator(request);
 *     for (software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ram.paginators.ListResourceSharePermissionsIterable responses = client.listResourceSharePermissionsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the paginator.
 * It only limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listResourceSharePermissions(software.amazon.awssdk.services.ram.model.ListResourceSharePermissionsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListResourceSharePermissionsIterable implements SdkIterable<ListResourceSharePermissionsResponse> {
    private final RamClient client;

    private final ListResourceSharePermissionsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListResourceSharePermissionsIterable(RamClient client, ListResourceSharePermissionsRequest firstRequest) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.nextPageFetcher = new ListResourceSharePermissionsResponseFetcher();
    }

    @Override
    public Iterator<ListResourceSharePermissionsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class ListResourceSharePermissionsResponseFetcher implements SyncPageFetcher<ListResourceSharePermissionsResponse> {
        @Override
        public boolean hasNextPage(ListResourceSharePermissionsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListResourceSharePermissionsResponse nextPage(ListResourceSharePermissionsResponse previousPage) {
            if (previousPage == null) {
                return client.listResourceSharePermissions(firstRequest);
            }
            return client.listResourceSharePermissions(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
