/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.nimble.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.nimble.NimbleAsyncClient;
import software.amazon.awssdk.services.nimble.model.GetLaunchProfileRequest;
import software.amazon.awssdk.services.nimble.model.GetLaunchProfileResponse;
import software.amazon.awssdk.services.nimble.model.GetStreamingImageRequest;
import software.amazon.awssdk.services.nimble.model.GetStreamingImageResponse;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionRequest;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionResponse;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionStreamRequest;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionStreamResponse;
import software.amazon.awssdk.services.nimble.model.GetStudioComponentRequest;
import software.amazon.awssdk.services.nimble.model.GetStudioComponentResponse;
import software.amazon.awssdk.services.nimble.model.GetStudioRequest;
import software.amazon.awssdk.services.nimble.model.GetStudioResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface NimbleAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getLaunchProfileRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileDeleted(
            GetLaunchProfileRequest getLaunchProfileRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetLaunchProfileRequest#builder()}
     *
     * @param getLaunchProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileDeleted(
            Consumer<GetLaunchProfileRequest.Builder> getLaunchProfileRequest) {
        return waitUntilLaunchProfileDeleted(GetLaunchProfileRequest.builder().applyMutation(getLaunchProfileRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getLaunchProfileRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileDeleted(
            GetLaunchProfileRequest getLaunchProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getLaunchProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileDeleted(
            Consumer<GetLaunchProfileRequest.Builder> getLaunchProfileRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilLaunchProfileDeleted(GetLaunchProfileRequest.builder().applyMutation(getLaunchProfileRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileReady} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getLaunchProfileRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileReady(
            GetLaunchProfileRequest getLaunchProfileRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileReady} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetLaunchProfileRequest#builder()}
     *
     * @param getLaunchProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileReady(
            Consumer<GetLaunchProfileRequest.Builder> getLaunchProfileRequest) {
        return waitUntilLaunchProfileReady(GetLaunchProfileRequest.builder().applyMutation(getLaunchProfileRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileReady} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getLaunchProfileRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileReady(
            GetLaunchProfileRequest getLaunchProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getLaunchProfile} API until the desired condition {@code LaunchProfileReady} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getLaunchProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetLaunchProfileResponse>> waitUntilLaunchProfileReady(
            Consumer<GetLaunchProfileRequest.Builder> getLaunchProfileRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilLaunchProfileReady(GetLaunchProfileRequest.builder().applyMutation(getLaunchProfileRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamingImageRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageDeleted(
            GetStreamingImageRequest getStreamingImageRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamingImageRequest#builder()}
     *
     * @param getStreamingImageRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageDeleted(
            Consumer<GetStreamingImageRequest.Builder> getStreamingImageRequest) {
        return waitUntilStreamingImageDeleted(GetStreamingImageRequest.builder().applyMutation(getStreamingImageRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamingImageRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageDeleted(
            GetStreamingImageRequest getStreamingImageRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamingImageRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageDeleted(
            Consumer<GetStreamingImageRequest.Builder> getStreamingImageRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamingImageDeleted(GetStreamingImageRequest.builder().applyMutation(getStreamingImageRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageReady} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamingImageRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageReady(
            GetStreamingImageRequest getStreamingImageRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageReady} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamingImageRequest#builder()}
     *
     * @param getStreamingImageRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageReady(
            Consumer<GetStreamingImageRequest.Builder> getStreamingImageRequest) {
        return waitUntilStreamingImageReady(GetStreamingImageRequest.builder().applyMutation(getStreamingImageRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageReady} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamingImageRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageReady(
            GetStreamingImageRequest getStreamingImageRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingImage} API until the desired condition {@code StreamingImageReady} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamingImageRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingImageResponse>> waitUntilStreamingImageReady(
            Consumer<GetStreamingImageRequest.Builder> getStreamingImageRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamingImageReady(GetStreamingImageRequest.builder().applyMutation(getStreamingImageRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param getStreamingSessionRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionDeleted(
            GetStreamingSessionRequest getStreamingSessionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamingSessionRequest#builder()}
     *
     * @param getStreamingSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionDeleted(
            Consumer<GetStreamingSessionRequest.Builder> getStreamingSessionRequest) {
        return waitUntilStreamingSessionDeleted(GetStreamingSessionRequest.builder().applyMutation(getStreamingSessionRequest)
                .build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param getStreamingSessionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionDeleted(
            GetStreamingSessionRequest getStreamingSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamingSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionDeleted(
            Consumer<GetStreamingSessionRequest.Builder> getStreamingSessionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamingSessionDeleted(GetStreamingSessionRequest.builder().applyMutation(getStreamingSessionRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition {@code StreamingSessionReady}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamingSessionRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionReady(
            GetStreamingSessionRequest getStreamingSessionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition {@code StreamingSessionReady}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamingSessionRequest#builder()}
     *
     * @param getStreamingSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionReady(
            Consumer<GetStreamingSessionRequest.Builder> getStreamingSessionRequest) {
        return waitUntilStreamingSessionReady(GetStreamingSessionRequest.builder().applyMutation(getStreamingSessionRequest)
                .build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition {@code StreamingSessionReady}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamingSessionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionReady(
            GetStreamingSessionRequest getStreamingSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition {@code StreamingSessionReady}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamingSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionReady(
            Consumer<GetStreamingSessionRequest.Builder> getStreamingSessionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamingSessionReady(GetStreamingSessionRequest.builder().applyMutation(getStreamingSessionRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionStopped} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param getStreamingSessionRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionStopped(
            GetStreamingSessionRequest getStreamingSessionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionStopped} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamingSessionRequest#builder()}
     *
     * @param getStreamingSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionStopped(
            Consumer<GetStreamingSessionRequest.Builder> getStreamingSessionRequest) {
        return waitUntilStreamingSessionStopped(GetStreamingSessionRequest.builder().applyMutation(getStreamingSessionRequest)
                .build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionStopped} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param getStreamingSessionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionStopped(
            GetStreamingSessionRequest getStreamingSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSession} API until the desired condition
     * {@code StreamingSessionStopped} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamingSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionResponse>> waitUntilStreamingSessionStopped(
            Consumer<GetStreamingSessionRequest.Builder> getStreamingSessionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamingSessionStopped(GetStreamingSessionRequest.builder().applyMutation(getStreamingSessionRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSessionStream} API until the desired condition
     * {@code StreamingSessionStreamReady} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param getStreamingSessionStreamRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionStreamResponse>> waitUntilStreamingSessionStreamReady(
            GetStreamingSessionStreamRequest getStreamingSessionStreamRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSessionStream} API until the desired condition
     * {@code StreamingSessionStreamReady} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamingSessionStreamRequest#builder()}
     *
     * @param getStreamingSessionStreamRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionStreamResponse>> waitUntilStreamingSessionStreamReady(
            Consumer<GetStreamingSessionStreamRequest.Builder> getStreamingSessionStreamRequest) {
        return waitUntilStreamingSessionStreamReady(GetStreamingSessionStreamRequest.builder()
                .applyMutation(getStreamingSessionStreamRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSessionStream} API until the desired condition
     * {@code StreamingSessionStreamReady} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param getStreamingSessionStreamRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionStreamResponse>> waitUntilStreamingSessionStreamReady(
            GetStreamingSessionStreamRequest getStreamingSessionStreamRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStreamingSessionStream} API until the desired condition
     * {@code StreamingSessionStreamReady} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamingSessionStreamRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamingSessionStreamResponse>> waitUntilStreamingSessionStreamReady(
            Consumer<GetStreamingSessionStreamRequest.Builder> getStreamingSessionStreamRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamingSessionStreamReady(
                GetStreamingSessionStreamRequest.builder().applyMutation(getStreamingSessionStreamRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStudioComponentRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentDeleted(
            GetStudioComponentRequest getStudioComponentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStudioComponentRequest#builder()}
     *
     * @param getStudioComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentDeleted(
            Consumer<GetStudioComponentRequest.Builder> getStudioComponentRequest) {
        return waitUntilStudioComponentDeleted(GetStudioComponentRequest.builder().applyMutation(getStudioComponentRequest)
                .build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStudioComponentRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentDeleted(
            GetStudioComponentRequest getStudioComponentRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStudioComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentDeleted(
            Consumer<GetStudioComponentRequest.Builder> getStudioComponentRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStudioComponentDeleted(GetStudioComponentRequest.builder().applyMutation(getStudioComponentRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentReady}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStudioComponentRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentReady(
            GetStudioComponentRequest getStudioComponentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentReady}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStudioComponentRequest#builder()}
     *
     * @param getStudioComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentReady(
            Consumer<GetStudioComponentRequest.Builder> getStudioComponentRequest) {
        return waitUntilStudioComponentReady(GetStudioComponentRequest.builder().applyMutation(getStudioComponentRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentReady}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStudioComponentRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentReady(
            GetStudioComponentRequest getStudioComponentRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudioComponent} API until the desired condition {@code StudioComponentReady}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStudioComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioComponentResponse>> waitUntilStudioComponentReady(
            Consumer<GetStudioComponentRequest.Builder> getStudioComponentRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStudioComponentReady(
                GetStudioComponentRequest.builder().applyMutation(getStudioComponentRequest).build(), WaiterOverrideConfiguration
                        .builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param getStudioRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioDeleted(GetStudioRequest getStudioRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStudioRequest#builder()}
     *
     * @param getStudioRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioDeleted(
            Consumer<GetStudioRequest.Builder> getStudioRequest) {
        return waitUntilStudioDeleted(GetStudioRequest.builder().applyMutation(getStudioRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param getStudioRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioDeleted(GetStudioRequest getStudioRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStudioRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioDeleted(
            Consumer<GetStudioRequest.Builder> getStudioRequest, Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStudioDeleted(GetStudioRequest.builder().applyMutation(getStudioRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioReady} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getStudioRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioReady(GetStudioRequest getStudioRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioReady} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStudioRequest#builder()}
     *
     * @param getStudioRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioReady(
            Consumer<GetStudioRequest.Builder> getStudioRequest) {
        return waitUntilStudioReady(GetStudioRequest.builder().applyMutation(getStudioRequest).build());
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioReady} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getStudioRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioReady(GetStudioRequest getStudioRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link NimbleAsyncClient#getStudio} API until the desired condition {@code StudioReady} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStudioRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStudioResponse>> waitUntilStudioReady(
            Consumer<GetStudioRequest.Builder> getStudioRequest, Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStudioReady(GetStudioRequest.builder().applyMutation(getStudioRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link NimbleAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultNimbleAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link NimbleAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link NimbleAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link NimbleAsyncWaiter}
     */
    static NimbleAsyncWaiter create() {
        return DefaultNimbleAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link NimbleAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(NimbleAsyncClient client);

        /**
         * Builds an instance of {@link NimbleAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link NimbleAsyncWaiter}
         */
        NimbleAsyncWaiter build();
    }
}
