/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that contains details about when the IAM entities (users or roles) were last used in an attempt to access
 * the specified Amazon Web Services service.
 * </p>
 * <p>
 * This data type is a response element in the <a>GetServiceLastAccessedDetailsWithEntities</a> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EntityDetails implements SdkPojo, Serializable, ToCopyableBuilder<EntityDetails.Builder, EntityDetails> {
    private static final SdkField<EntityInfo> ENTITY_INFO_FIELD = SdkField.<EntityInfo> builder(MarshallingType.SDK_POJO)
            .memberName("EntityInfo").getter(getter(EntityDetails::entityInfo)).setter(setter(Builder::entityInfo))
            .constructor(EntityInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntityInfo").build()).build();

    private static final SdkField<Instant> LAST_AUTHENTICATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastAuthenticated").getter(getter(EntityDetails::lastAuthenticated))
            .setter(setter(Builder::lastAuthenticated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastAuthenticated").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENTITY_INFO_FIELD,
            LAST_AUTHENTICATED_FIELD));

    private static final long serialVersionUID = 1L;

    private final EntityInfo entityInfo;

    private final Instant lastAuthenticated;

    private EntityDetails(BuilderImpl builder) {
        this.entityInfo = builder.entityInfo;
        this.lastAuthenticated = builder.lastAuthenticated;
    }

    /**
     * <p>
     * The <code>EntityInfo</code> object that contains details about the entity (user or role).
     * </p>
     * 
     * @return The <code>EntityInfo</code> object that contains details about the entity (user or role).
     */
    public final EntityInfo entityInfo() {
        return entityInfo;
    }

    /**
     * <p>
     * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
     * authenticated entity last attempted to access Amazon Web Services. Amazon Web Services does not report
     * unauthenticated requests.
     * </p>
     * <p>
     * This field is null if no IAM entities attempted to access the service within the <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#service-last-accessed-reporting-period"
     * >tracking period</a>.
     * </p>
     * 
     * @return The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
     *         authenticated entity last attempted to access Amazon Web Services. Amazon Web Services does not report
     *         unauthenticated requests.</p>
     *         <p>
     *         This field is null if no IAM entities attempted to access the service within the <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#service-last-accessed-reporting-period"
     *         >tracking period</a>.
     */
    public final Instant lastAuthenticated() {
        return lastAuthenticated;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(entityInfo());
        hashCode = 31 * hashCode + Objects.hashCode(lastAuthenticated());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EntityDetails)) {
            return false;
        }
        EntityDetails other = (EntityDetails) obj;
        return Objects.equals(entityInfo(), other.entityInfo()) && Objects.equals(lastAuthenticated(), other.lastAuthenticated());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EntityDetails").add("EntityInfo", entityInfo()).add("LastAuthenticated", lastAuthenticated())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EntityInfo":
            return Optional.ofNullable(clazz.cast(entityInfo()));
        case "LastAuthenticated":
            return Optional.ofNullable(clazz.cast(lastAuthenticated()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EntityDetails, T> g) {
        return obj -> g.apply((EntityDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EntityDetails> {
        /**
         * <p>
         * The <code>EntityInfo</code> object that contains details about the entity (user or role).
         * </p>
         * 
         * @param entityInfo
         *        The <code>EntityInfo</code> object that contains details about the entity (user or role).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityInfo(EntityInfo entityInfo);

        /**
         * <p>
         * The <code>EntityInfo</code> object that contains details about the entity (user or role).
         * </p>
         * This is a convenience method that creates an instance of the {@link EntityInfo.Builder} avoiding the need to
         * create one manually via {@link EntityInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntityInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #entityInfo(EntityInfo)}.
         * 
         * @param entityInfo
         *        a consumer that will call methods on {@link EntityInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entityInfo(EntityInfo)
         */
        default Builder entityInfo(Consumer<EntityInfo.Builder> entityInfo) {
            return entityInfo(EntityInfo.builder().applyMutation(entityInfo).build());
        }

        /**
         * <p>
         * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
         * authenticated entity last attempted to access Amazon Web Services. Amazon Web Services does not report
         * unauthenticated requests.
         * </p>
         * <p>
         * This field is null if no IAM entities attempted to access the service within the <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#service-last-accessed-reporting-period"
         * >tracking period</a>.
         * </p>
         * 
         * @param lastAuthenticated
         *        The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
         *        authenticated entity last attempted to access Amazon Web Services. Amazon Web Services does not report
         *        unauthenticated requests.</p>
         *        <p>
         *        This field is null if no IAM entities attempted to access the service within the <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#service-last-accessed-reporting-period"
         *        >tracking period</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAuthenticated(Instant lastAuthenticated);
    }

    static final class BuilderImpl implements Builder {
        private EntityInfo entityInfo;

        private Instant lastAuthenticated;

        private BuilderImpl() {
        }

        private BuilderImpl(EntityDetails model) {
            entityInfo(model.entityInfo);
            lastAuthenticated(model.lastAuthenticated);
        }

        public final EntityInfo.Builder getEntityInfo() {
            return entityInfo != null ? entityInfo.toBuilder() : null;
        }

        public final void setEntityInfo(EntityInfo.BuilderImpl entityInfo) {
            this.entityInfo = entityInfo != null ? entityInfo.build() : null;
        }

        @Override
        public final Builder entityInfo(EntityInfo entityInfo) {
            this.entityInfo = entityInfo;
            return this;
        }

        public final Instant getLastAuthenticated() {
            return lastAuthenticated;
        }

        public final void setLastAuthenticated(Instant lastAuthenticated) {
            this.lastAuthenticated = lastAuthenticated;
        }

        @Override
        public final Builder lastAuthenticated(Instant lastAuthenticated) {
            this.lastAuthenticated = lastAuthenticated;
            return this;
        }

        @Override
        public EntityDetails build() {
            return new EntityDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
