/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure used to define a table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TableInput implements SdkPojo, Serializable, ToCopyableBuilder<TableInput.Builder, TableInput> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(TableInput::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(TableInput::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Owner")
            .getter(getter(TableInput::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<Instant> LAST_ACCESS_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastAccessTime").getter(getter(TableInput::lastAccessTime)).setter(setter(Builder::lastAccessTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastAccessTime").build()).build();

    private static final SdkField<Instant> LAST_ANALYZED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastAnalyzedTime").getter(getter(TableInput::lastAnalyzedTime))
            .setter(setter(Builder::lastAnalyzedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastAnalyzedTime").build()).build();

    private static final SdkField<Integer> RETENTION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Retention").getter(getter(TableInput::retention)).setter(setter(Builder::retention))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Retention").build()).build();

    private static final SdkField<StorageDescriptor> STORAGE_DESCRIPTOR_FIELD = SdkField
            .<StorageDescriptor> builder(MarshallingType.SDK_POJO).memberName("StorageDescriptor")
            .getter(getter(TableInput::storageDescriptor)).setter(setter(Builder::storageDescriptor))
            .constructor(StorageDescriptor::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageDescriptor").build()).build();

    private static final SdkField<List<Column>> PARTITION_KEYS_FIELD = SdkField
            .<List<Column>> builder(MarshallingType.LIST)
            .memberName("PartitionKeys")
            .getter(getter(TableInput::partitionKeys))
            .setter(setter(Builder::partitionKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartitionKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Column> builder(MarshallingType.SDK_POJO)
                                            .constructor(Column::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VIEW_ORIGINAL_TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ViewOriginalText").getter(getter(TableInput::viewOriginalText))
            .setter(setter(Builder::viewOriginalText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViewOriginalText").build()).build();

    private static final SdkField<String> VIEW_EXPANDED_TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ViewExpandedText").getter(getter(TableInput::viewExpandedText))
            .setter(setter(Builder::viewExpandedText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViewExpandedText").build()).build();

    private static final SdkField<String> TABLE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TableType").getter(getter(TableInput::tableType)).setter(setter(Builder::tableType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TableType").build()).build();

    private static final SdkField<Map<String, String>> PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(TableInput::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<TableIdentifier> TARGET_TABLE_FIELD = SdkField
            .<TableIdentifier> builder(MarshallingType.SDK_POJO).memberName("TargetTable")
            .getter(getter(TableInput::targetTable)).setter(setter(Builder::targetTable)).constructor(TableIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetTable").build()).build();

    private static final SdkField<ViewDefinitionInput> VIEW_DEFINITION_FIELD = SdkField
            .<ViewDefinitionInput> builder(MarshallingType.SDK_POJO).memberName("ViewDefinition")
            .getter(getter(TableInput::viewDefinition)).setter(setter(Builder::viewDefinition))
            .constructor(ViewDefinitionInput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViewDefinition").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            OWNER_FIELD, LAST_ACCESS_TIME_FIELD, LAST_ANALYZED_TIME_FIELD, RETENTION_FIELD, STORAGE_DESCRIPTOR_FIELD,
            PARTITION_KEYS_FIELD, VIEW_ORIGINAL_TEXT_FIELD, VIEW_EXPANDED_TEXT_FIELD, TABLE_TYPE_FIELD, PARAMETERS_FIELD,
            TARGET_TABLE_FIELD, VIEW_DEFINITION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String owner;

    private final Instant lastAccessTime;

    private final Instant lastAnalyzedTime;

    private final Integer retention;

    private final StorageDescriptor storageDescriptor;

    private final List<Column> partitionKeys;

    private final String viewOriginalText;

    private final String viewExpandedText;

    private final String tableType;

    private final Map<String, String> parameters;

    private final TableIdentifier targetTable;

    private final ViewDefinitionInput viewDefinition;

    private TableInput(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.owner = builder.owner;
        this.lastAccessTime = builder.lastAccessTime;
        this.lastAnalyzedTime = builder.lastAnalyzedTime;
        this.retention = builder.retention;
        this.storageDescriptor = builder.storageDescriptor;
        this.partitionKeys = builder.partitionKeys;
        this.viewOriginalText = builder.viewOriginalText;
        this.viewExpandedText = builder.viewExpandedText;
        this.tableType = builder.tableType;
        this.parameters = builder.parameters;
        this.targetTable = builder.targetTable;
        this.viewDefinition = builder.viewDefinition;
    }

    /**
     * <p>
     * The table name. For Hive compatibility, this is folded to lowercase when it is stored.
     * </p>
     * 
     * @return The table name. For Hive compatibility, this is folded to lowercase when it is stored.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A description of the table.
     * </p>
     * 
     * @return A description of the table.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The table owner. Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
     * </p>
     * 
     * @return The table owner. Included for Apache Hive compatibility. Not used in the normal course of Glue
     *         operations.
     */
    public final String owner() {
        return owner;
    }

    /**
     * <p>
     * The last time that the table was accessed.
     * </p>
     * 
     * @return The last time that the table was accessed.
     */
    public final Instant lastAccessTime() {
        return lastAccessTime;
    }

    /**
     * <p>
     * The last time that column statistics were computed for this table.
     * </p>
     * 
     * @return The last time that column statistics were computed for this table.
     */
    public final Instant lastAnalyzedTime() {
        return lastAnalyzedTime;
    }

    /**
     * <p>
     * The retention time for this table.
     * </p>
     * 
     * @return The retention time for this table.
     */
    public final Integer retention() {
        return retention;
    }

    /**
     * <p>
     * A storage descriptor containing information about the physical storage of this table.
     * </p>
     * 
     * @return A storage descriptor containing information about the physical storage of this table.
     */
    public final StorageDescriptor storageDescriptor() {
        return storageDescriptor;
    }

    /**
     * For responses, this returns true if the service returned a value for the PartitionKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPartitionKeys() {
        return partitionKeys != null && !(partitionKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
     * </p>
     * <p>
     * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you must at
     * least set the value of <code>partitionKeys</code> to an empty list. For example:
     * </p>
     * <p>
     * <code>"PartitionKeys": []</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPartitionKeys} method.
     * </p>
     * 
     * @return A list of columns by which the table is partitioned. Only primitive types are supported as partition
     *         keys.</p>
     *         <p>
     *         When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
     *         must at least set the value of <code>partitionKeys</code> to an empty list. For example:
     *         </p>
     *         <p>
     *         <code>"PartitionKeys": []</code>
     */
    public final List<Column> partitionKeys() {
        return partitionKeys;
    }

    /**
     * <p>
     * Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table is a
     * <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
     * </p>
     * 
     * @return Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table is
     *         a <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
     */
    public final String viewOriginalText() {
        return viewOriginalText;
    }

    /**
     * <p>
     * Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
     * </p>
     * 
     * @return Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
     */
    public final String viewExpandedText() {
        return viewExpandedText;
    }

    /**
     * <p>
     * The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other services, such
     * as Athena, may create tables with additional table types.
     * </p>
     * <p>
     * Glue related table types:
     * </p>
     * <dl>
     * <dt>EXTERNAL_TABLE</dt>
     * <dd>
     * <p>
     * Hive compatible attribute - indicates a non-Hive managed table.
     * </p>
     * </dd>
     * <dt>GOVERNED</dt>
     * <dd>
     * <p>
     * Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
     * </p>
     * </dd>
     * </dl>
     * 
     * @return The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other
     *         services, such as Athena, may create tables with additional table types. </p>
     *         <p>
     *         Glue related table types:
     *         </p>
     *         <dl>
     *         <dt>EXTERNAL_TABLE</dt>
     *         <dd>
     *         <p>
     *         Hive compatible attribute - indicates a non-Hive managed table.
     *         </p>
     *         </dd>
     *         <dt>GOVERNED</dt>
     *         <dd>
     *         <p>
     *         Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
     *         </p>
     *         </dd>
     */
    public final String tableType() {
        return tableType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * These key-value pairs define properties associated with the table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return These key-value pairs define properties associated with the table.
     */
    public final Map<String, String> parameters() {
        return parameters;
    }

    /**
     * <p>
     * A <code>TableIdentifier</code> structure that describes a target table for resource linking.
     * </p>
     * 
     * @return A <code>TableIdentifier</code> structure that describes a target table for resource linking.
     */
    public final TableIdentifier targetTable() {
        return targetTable;
    }

    /**
     * <p>
     * A structure that contains all the information that defines the view, including the dialect or dialects for the
     * view, and the query.
     * </p>
     * 
     * @return A structure that contains all the information that defines the view, including the dialect or dialects
     *         for the view, and the query.
     */
    public final ViewDefinitionInput viewDefinition() {
        return viewDefinition;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(lastAccessTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastAnalyzedTime());
        hashCode = 31 * hashCode + Objects.hashCode(retention());
        hashCode = 31 * hashCode + Objects.hashCode(storageDescriptor());
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitionKeys() ? partitionKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(viewOriginalText());
        hashCode = 31 * hashCode + Objects.hashCode(viewExpandedText());
        hashCode = 31 * hashCode + Objects.hashCode(tableType());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(targetTable());
        hashCode = 31 * hashCode + Objects.hashCode(viewDefinition());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TableInput)) {
            return false;
        }
        TableInput other = (TableInput) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(owner(), other.owner()) && Objects.equals(lastAccessTime(), other.lastAccessTime())
                && Objects.equals(lastAnalyzedTime(), other.lastAnalyzedTime()) && Objects.equals(retention(), other.retention())
                && Objects.equals(storageDescriptor(), other.storageDescriptor())
                && hasPartitionKeys() == other.hasPartitionKeys() && Objects.equals(partitionKeys(), other.partitionKeys())
                && Objects.equals(viewOriginalText(), other.viewOriginalText())
                && Objects.equals(viewExpandedText(), other.viewExpandedText()) && Objects.equals(tableType(), other.tableType())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters())
                && Objects.equals(targetTable(), other.targetTable()) && Objects.equals(viewDefinition(), other.viewDefinition());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TableInput").add("Name", name()).add("Description", description()).add("Owner", owner())
                .add("LastAccessTime", lastAccessTime()).add("LastAnalyzedTime", lastAnalyzedTime())
                .add("Retention", retention()).add("StorageDescriptor", storageDescriptor())
                .add("PartitionKeys", hasPartitionKeys() ? partitionKeys() : null).add("ViewOriginalText", viewOriginalText())
                .add("ViewExpandedText", viewExpandedText()).add("TableType", tableType())
                .add("Parameters", hasParameters() ? parameters() : null).add("TargetTable", targetTable())
                .add("ViewDefinition", viewDefinition()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "LastAccessTime":
            return Optional.ofNullable(clazz.cast(lastAccessTime()));
        case "LastAnalyzedTime":
            return Optional.ofNullable(clazz.cast(lastAnalyzedTime()));
        case "Retention":
            return Optional.ofNullable(clazz.cast(retention()));
        case "StorageDescriptor":
            return Optional.ofNullable(clazz.cast(storageDescriptor()));
        case "PartitionKeys":
            return Optional.ofNullable(clazz.cast(partitionKeys()));
        case "ViewOriginalText":
            return Optional.ofNullable(clazz.cast(viewOriginalText()));
        case "ViewExpandedText":
            return Optional.ofNullable(clazz.cast(viewExpandedText()));
        case "TableType":
            return Optional.ofNullable(clazz.cast(tableType()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "TargetTable":
            return Optional.ofNullable(clazz.cast(targetTable()));
        case "ViewDefinition":
            return Optional.ofNullable(clazz.cast(viewDefinition()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TableInput, T> g) {
        return obj -> g.apply((TableInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TableInput> {
        /**
         * <p>
         * The table name. For Hive compatibility, this is folded to lowercase when it is stored.
         * </p>
         * 
         * @param name
         *        The table name. For Hive compatibility, this is folded to lowercase when it is stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the table.
         * </p>
         * 
         * @param description
         *        A description of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The table owner. Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
         * </p>
         * 
         * @param owner
         *        The table owner. Included for Apache Hive compatibility. Not used in the normal course of Glue
         *        operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * The last time that the table was accessed.
         * </p>
         * 
         * @param lastAccessTime
         *        The last time that the table was accessed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAccessTime(Instant lastAccessTime);

        /**
         * <p>
         * The last time that column statistics were computed for this table.
         * </p>
         * 
         * @param lastAnalyzedTime
         *        The last time that column statistics were computed for this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAnalyzedTime(Instant lastAnalyzedTime);

        /**
         * <p>
         * The retention time for this table.
         * </p>
         * 
         * @param retention
         *        The retention time for this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retention(Integer retention);

        /**
         * <p>
         * A storage descriptor containing information about the physical storage of this table.
         * </p>
         * 
         * @param storageDescriptor
         *        A storage descriptor containing information about the physical storage of this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageDescriptor(StorageDescriptor storageDescriptor);

        /**
         * <p>
         * A storage descriptor containing information about the physical storage of this table.
         * </p>
         * This is a convenience method that creates an instance of the {@link StorageDescriptor.Builder} avoiding the
         * need to create one manually via {@link StorageDescriptor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StorageDescriptor.Builder#build()} is called immediately and its
         * result is passed to {@link #storageDescriptor(StorageDescriptor)}.
         * 
         * @param storageDescriptor
         *        a consumer that will call methods on {@link StorageDescriptor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storageDescriptor(StorageDescriptor)
         */
        default Builder storageDescriptor(Consumer<StorageDescriptor.Builder> storageDescriptor) {
            return storageDescriptor(StorageDescriptor.builder().applyMutation(storageDescriptor).build());
        }

        /**
         * <p>
         * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
         * </p>
         * <p>
         * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
         * must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         * </p>
         * <p>
         * <code>"PartitionKeys": []</code>
         * </p>
         * 
         * @param partitionKeys
         *        A list of columns by which the table is partitioned. Only primitive types are supported as partition
         *        keys.</p>
         *        <p>
         *        When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>,
         *        you must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         *        </p>
         *        <p>
         *        <code>"PartitionKeys": []</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKeys(Collection<Column> partitionKeys);

        /**
         * <p>
         * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
         * </p>
         * <p>
         * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
         * must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         * </p>
         * <p>
         * <code>"PartitionKeys": []</code>
         * </p>
         * 
         * @param partitionKeys
         *        A list of columns by which the table is partitioned. Only primitive types are supported as partition
         *        keys.</p>
         *        <p>
         *        When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>,
         *        you must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         *        </p>
         *        <p>
         *        <code>"PartitionKeys": []</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKeys(Column... partitionKeys);

        /**
         * <p>
         * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
         * </p>
         * <p>
         * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
         * must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         * </p>
         * <p>
         * <code>"PartitionKeys": []</code>
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.Column.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.glue.model.Column#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.Column.Builder#build()} is called immediately and its
         * result is passed to {@link #partitionKeys(List<Column>)}.
         * 
         * @param partitionKeys
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.glue.model.Column.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #partitionKeys(java.util.Collection<Column>)
         */
        Builder partitionKeys(Consumer<Column.Builder>... partitionKeys);

        /**
         * <p>
         * Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table is a
         * <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
         * </p>
         * 
         * @param viewOriginalText
         *        Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table
         *        is a <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewOriginalText(String viewOriginalText);

        /**
         * <p>
         * Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
         * </p>
         * 
         * @param viewExpandedText
         *        Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewExpandedText(String viewExpandedText);

        /**
         * <p>
         * The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other services,
         * such as Athena, may create tables with additional table types.
         * </p>
         * <p>
         * Glue related table types:
         * </p>
         * <dl>
         * <dt>EXTERNAL_TABLE</dt>
         * <dd>
         * <p>
         * Hive compatible attribute - indicates a non-Hive managed table.
         * </p>
         * </dd>
         * <dt>GOVERNED</dt>
         * <dd>
         * <p>
         * Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param tableType
         *        The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other
         *        services, such as Athena, may create tables with additional table types. </p>
         *        <p>
         *        Glue related table types:
         *        </p>
         *        <dl>
         *        <dt>EXTERNAL_TABLE</dt>
         *        <dd>
         *        <p>
         *        Hive compatible attribute - indicates a non-Hive managed table.
         *        </p>
         *        </dd>
         *        <dt>GOVERNED</dt>
         *        <dd>
         *        <p>
         *        Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableType(String tableType);

        /**
         * <p>
         * These key-value pairs define properties associated with the table.
         * </p>
         * 
         * @param parameters
         *        These key-value pairs define properties associated with the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, String> parameters);

        /**
         * <p>
         * A <code>TableIdentifier</code> structure that describes a target table for resource linking.
         * </p>
         * 
         * @param targetTable
         *        A <code>TableIdentifier</code> structure that describes a target table for resource linking.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTable(TableIdentifier targetTable);

        /**
         * <p>
         * A <code>TableIdentifier</code> structure that describes a target table for resource linking.
         * </p>
         * This is a convenience method that creates an instance of the {@link TableIdentifier.Builder} avoiding the
         * need to create one manually via {@link TableIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TableIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #targetTable(TableIdentifier)}.
         * 
         * @param targetTable
         *        a consumer that will call methods on {@link TableIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetTable(TableIdentifier)
         */
        default Builder targetTable(Consumer<TableIdentifier.Builder> targetTable) {
            return targetTable(TableIdentifier.builder().applyMutation(targetTable).build());
        }

        /**
         * <p>
         * A structure that contains all the information that defines the view, including the dialect or dialects for
         * the view, and the query.
         * </p>
         * 
         * @param viewDefinition
         *        A structure that contains all the information that defines the view, including the dialect or dialects
         *        for the view, and the query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewDefinition(ViewDefinitionInput viewDefinition);

        /**
         * <p>
         * A structure that contains all the information that defines the view, including the dialect or dialects for
         * the view, and the query.
         * </p>
         * This is a convenience method that creates an instance of the {@link ViewDefinitionInput.Builder} avoiding the
         * need to create one manually via {@link ViewDefinitionInput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ViewDefinitionInput.Builder#build()} is called immediately and
         * its result is passed to {@link #viewDefinition(ViewDefinitionInput)}.
         * 
         * @param viewDefinition
         *        a consumer that will call methods on {@link ViewDefinitionInput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #viewDefinition(ViewDefinitionInput)
         */
        default Builder viewDefinition(Consumer<ViewDefinitionInput.Builder> viewDefinition) {
            return viewDefinition(ViewDefinitionInput.builder().applyMutation(viewDefinition).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String owner;

        private Instant lastAccessTime;

        private Instant lastAnalyzedTime;

        private Integer retention;

        private StorageDescriptor storageDescriptor;

        private List<Column> partitionKeys = DefaultSdkAutoConstructList.getInstance();

        private String viewOriginalText;

        private String viewExpandedText;

        private String tableType;

        private Map<String, String> parameters = DefaultSdkAutoConstructMap.getInstance();

        private TableIdentifier targetTable;

        private ViewDefinitionInput viewDefinition;

        private BuilderImpl() {
        }

        private BuilderImpl(TableInput model) {
            name(model.name);
            description(model.description);
            owner(model.owner);
            lastAccessTime(model.lastAccessTime);
            lastAnalyzedTime(model.lastAnalyzedTime);
            retention(model.retention);
            storageDescriptor(model.storageDescriptor);
            partitionKeys(model.partitionKeys);
            viewOriginalText(model.viewOriginalText);
            viewExpandedText(model.viewExpandedText);
            tableType(model.tableType);
            parameters(model.parameters);
            targetTable(model.targetTable);
            viewDefinition(model.viewDefinition);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final Instant getLastAccessTime() {
            return lastAccessTime;
        }

        public final void setLastAccessTime(Instant lastAccessTime) {
            this.lastAccessTime = lastAccessTime;
        }

        @Override
        public final Builder lastAccessTime(Instant lastAccessTime) {
            this.lastAccessTime = lastAccessTime;
            return this;
        }

        public final Instant getLastAnalyzedTime() {
            return lastAnalyzedTime;
        }

        public final void setLastAnalyzedTime(Instant lastAnalyzedTime) {
            this.lastAnalyzedTime = lastAnalyzedTime;
        }

        @Override
        public final Builder lastAnalyzedTime(Instant lastAnalyzedTime) {
            this.lastAnalyzedTime = lastAnalyzedTime;
            return this;
        }

        public final Integer getRetention() {
            return retention;
        }

        public final void setRetention(Integer retention) {
            this.retention = retention;
        }

        @Override
        public final Builder retention(Integer retention) {
            this.retention = retention;
            return this;
        }

        public final StorageDescriptor.Builder getStorageDescriptor() {
            return storageDescriptor != null ? storageDescriptor.toBuilder() : null;
        }

        public final void setStorageDescriptor(StorageDescriptor.BuilderImpl storageDescriptor) {
            this.storageDescriptor = storageDescriptor != null ? storageDescriptor.build() : null;
        }

        @Override
        public final Builder storageDescriptor(StorageDescriptor storageDescriptor) {
            this.storageDescriptor = storageDescriptor;
            return this;
        }

        public final List<Column.Builder> getPartitionKeys() {
            List<Column.Builder> result = ColumnListCopier.copyToBuilder(this.partitionKeys);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPartitionKeys(Collection<Column.BuilderImpl> partitionKeys) {
            this.partitionKeys = ColumnListCopier.copyFromBuilder(partitionKeys);
        }

        @Override
        public final Builder partitionKeys(Collection<Column> partitionKeys) {
            this.partitionKeys = ColumnListCopier.copy(partitionKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionKeys(Column... partitionKeys) {
            partitionKeys(Arrays.asList(partitionKeys));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionKeys(Consumer<Column.Builder>... partitionKeys) {
            partitionKeys(Stream.of(partitionKeys).map(c -> Column.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getViewOriginalText() {
            return viewOriginalText;
        }

        public final void setViewOriginalText(String viewOriginalText) {
            this.viewOriginalText = viewOriginalText;
        }

        @Override
        public final Builder viewOriginalText(String viewOriginalText) {
            this.viewOriginalText = viewOriginalText;
            return this;
        }

        public final String getViewExpandedText() {
            return viewExpandedText;
        }

        public final void setViewExpandedText(String viewExpandedText) {
            this.viewExpandedText = viewExpandedText;
        }

        @Override
        public final Builder viewExpandedText(String viewExpandedText) {
            this.viewExpandedText = viewExpandedText;
            return this;
        }

        public final String getTableType() {
            return tableType;
        }

        public final void setTableType(String tableType) {
            this.tableType = tableType;
        }

        @Override
        public final Builder tableType(String tableType) {
            this.tableType = tableType;
            return this;
        }

        public final Map<String, String> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
            return this;
        }

        public final TableIdentifier.Builder getTargetTable() {
            return targetTable != null ? targetTable.toBuilder() : null;
        }

        public final void setTargetTable(TableIdentifier.BuilderImpl targetTable) {
            this.targetTable = targetTable != null ? targetTable.build() : null;
        }

        @Override
        public final Builder targetTable(TableIdentifier targetTable) {
            this.targetTable = targetTable;
            return this;
        }

        public final ViewDefinitionInput.Builder getViewDefinition() {
            return viewDefinition != null ? viewDefinition.toBuilder() : null;
        }

        public final void setViewDefinition(ViewDefinitionInput.BuilderImpl viewDefinition) {
            this.viewDefinition = viewDefinition != null ? viewDefinition.build() : null;
        }

        @Override
        public final Builder viewDefinition(ViewDefinitionInput viewDefinition) {
            this.viewDefinition = viewDefinition;
            return this;
        }

        @Override
        public TableInput build() {
            return new TableInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
