/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eventbridge.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A global endpoint used to improve your application's availability by making it regional-fault tolerant. For more
 * information about global endpoints, see <a
 * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-global-endpoints.html">Making applications
 * Regional-fault tolerant with global endpoints and event replication</a> in the <i> <i>Amazon EventBridge User
 * Guide</i> </i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Endpoint::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Endpoint::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Endpoint::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<RoutingConfig> ROUTING_CONFIG_FIELD = SdkField
            .<RoutingConfig> builder(MarshallingType.SDK_POJO).memberName("RoutingConfig")
            .getter(getter(Endpoint::routingConfig)).setter(setter(Builder::routingConfig)).constructor(RoutingConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoutingConfig").build()).build();

    private static final SdkField<ReplicationConfig> REPLICATION_CONFIG_FIELD = SdkField
            .<ReplicationConfig> builder(MarshallingType.SDK_POJO).memberName("ReplicationConfig")
            .getter(getter(Endpoint::replicationConfig)).setter(setter(Builder::replicationConfig))
            .constructor(ReplicationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationConfig").build()).build();

    private static final SdkField<List<EndpointEventBus>> EVENT_BUSES_FIELD = SdkField
            .<List<EndpointEventBus>> builder(MarshallingType.LIST)
            .memberName("EventBuses")
            .getter(getter(Endpoint::eventBuses))
            .setter(setter(Builder::eventBuses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventBuses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EndpointEventBus> builder(MarshallingType.SDK_POJO)
                                            .constructor(EndpointEventBus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(Endpoint::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<String> ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointId").getter(getter(Endpoint::endpointId)).setter(setter(Builder::endpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointId").build()).build();

    private static final SdkField<String> ENDPOINT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointUrl").getter(getter(Endpoint::endpointUrl)).setter(setter(Builder::endpointUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointUrl").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(Endpoint::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateReason").getter(getter(Endpoint::stateReason)).setter(setter(Builder::stateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateReason").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Endpoint::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(Endpoint::lastModifiedTime)).setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            ARN_FIELD, ROUTING_CONFIG_FIELD, REPLICATION_CONFIG_FIELD, EVENT_BUSES_FIELD, ROLE_ARN_FIELD, ENDPOINT_ID_FIELD,
            ENDPOINT_URL_FIELD, STATE_FIELD, STATE_REASON_FIELD, CREATION_TIME_FIELD, LAST_MODIFIED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String arn;

    private final RoutingConfig routingConfig;

    private final ReplicationConfig replicationConfig;

    private final List<EndpointEventBus> eventBuses;

    private final String roleArn;

    private final String endpointId;

    private final String endpointUrl;

    private final String state;

    private final String stateReason;

    private final Instant creationTime;

    private final Instant lastModifiedTime;

    private Endpoint(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.arn = builder.arn;
        this.routingConfig = builder.routingConfig;
        this.replicationConfig = builder.replicationConfig;
        this.eventBuses = builder.eventBuses;
        this.roleArn = builder.roleArn;
        this.endpointId = builder.endpointId;
        this.endpointUrl = builder.endpointUrl;
        this.state = builder.state;
        this.stateReason = builder.stateReason;
        this.creationTime = builder.creationTime;
        this.lastModifiedTime = builder.lastModifiedTime;
    }

    /**
     * <p>
     * The name of the endpoint.
     * </p>
     * 
     * @return The name of the endpoint.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A description for the endpoint.
     * </p>
     * 
     * @return A description for the endpoint.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ARN of the endpoint.
     * </p>
     * 
     * @return The ARN of the endpoint.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The routing configuration of the endpoint.
     * </p>
     * 
     * @return The routing configuration of the endpoint.
     */
    public final RoutingConfig routingConfig() {
        return routingConfig;
    }

    /**
     * <p>
     * Whether event replication was enabled or disabled for this endpoint. The default state is <code>ENABLED</code>
     * which means you must supply a <code>RoleArn</code>. If you don't have a <code>RoleArn</code> or you don't want
     * event replication enabled, set the state to <code>DISABLED</code>.
     * </p>
     * 
     * @return Whether event replication was enabled or disabled for this endpoint. The default state is
     *         <code>ENABLED</code> which means you must supply a <code>RoleArn</code>. If you don't have a
     *         <code>RoleArn</code> or you don't want event replication enabled, set the state to <code>DISABLED</code>.
     */
    public final ReplicationConfig replicationConfig() {
        return replicationConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the EventBuses property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEventBuses() {
        return eventBuses != null && !(eventBuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The event buses being used by the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventBuses} method.
     * </p>
     * 
     * @return The event buses being used by the endpoint.
     */
    public final List<EndpointEventBus> eventBuses() {
        return eventBuses;
    }

    /**
     * <p>
     * The ARN of the role used by event replication for the endpoint.
     * </p>
     * 
     * @return The ARN of the role used by event replication for the endpoint.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The URL subdomain of the endpoint. For example, if the URL for Endpoint is
     * https://abcde.veo.endpoints.event.amazonaws.com, then the EndpointId is <code>abcde.veo</code>.
     * </p>
     * 
     * @return The URL subdomain of the endpoint. For example, if the URL for Endpoint is
     *         https://abcde.veo.endpoints.event.amazonaws.com, then the EndpointId is <code>abcde.veo</code>.
     */
    public final String endpointId() {
        return endpointId;
    }

    /**
     * <p>
     * The URL of the endpoint.
     * </p>
     * 
     * @return The URL of the endpoint.
     */
    public final String endpointUrl() {
        return endpointUrl;
    }

    /**
     * <p>
     * The current state of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link EndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the endpoint.
     * @see EndpointState
     */
    public final EndpointState state() {
        return EndpointState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link EndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the endpoint.
     * @see EndpointState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The reason the endpoint is in its current state.
     * </p>
     * 
     * @return The reason the endpoint is in its current state.
     */
    public final String stateReason() {
        return stateReason;
    }

    /**
     * <p>
     * The time the endpoint was created.
     * </p>
     * 
     * @return The time the endpoint was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The last time the endpoint was modified.
     * </p>
     * 
     * @return The last time the endpoint was modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(routingConfig());
        hashCode = 31 * hashCode + Objects.hashCode(replicationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasEventBuses() ? eventBuses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(endpointId());
        hashCode = 31 * hashCode + Objects.hashCode(endpointUrl());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateReason());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(arn(), other.arn()) && Objects.equals(routingConfig(), other.routingConfig())
                && Objects.equals(replicationConfig(), other.replicationConfig()) && hasEventBuses() == other.hasEventBuses()
                && Objects.equals(eventBuses(), other.eventBuses()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(endpointId(), other.endpointId()) && Objects.equals(endpointUrl(), other.endpointUrl())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(stateReason(), other.stateReason())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Endpoint").add("Name", name()).add("Description", description()).add("Arn", arn())
                .add("RoutingConfig", routingConfig()).add("ReplicationConfig", replicationConfig())
                .add("EventBuses", hasEventBuses() ? eventBuses() : null).add("RoleArn", roleArn())
                .add("EndpointId", endpointId()).add("EndpointUrl", endpointUrl()).add("State", stateAsString())
                .add("StateReason", stateReason()).add("CreationTime", creationTime())
                .add("LastModifiedTime", lastModifiedTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "RoutingConfig":
            return Optional.ofNullable(clazz.cast(routingConfig()));
        case "ReplicationConfig":
            return Optional.ofNullable(clazz.cast(replicationConfig()));
        case "EventBuses":
            return Optional.ofNullable(clazz.cast(eventBuses()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "EndpointId":
            return Optional.ofNullable(clazz.cast(endpointId()));
        case "EndpointUrl":
            return Optional.ofNullable(clazz.cast(endpointUrl()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateReason":
            return Optional.ofNullable(clazz.cast(stateReason()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The name of the endpoint.
         * </p>
         * 
         * @param name
         *        The name of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description for the endpoint.
         * </p>
         * 
         * @param description
         *        A description for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN of the endpoint.
         * </p>
         * 
         * @param arn
         *        The ARN of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The routing configuration of the endpoint.
         * </p>
         * 
         * @param routingConfig
         *        The routing configuration of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routingConfig(RoutingConfig routingConfig);

        /**
         * <p>
         * The routing configuration of the endpoint.
         * </p>
         * This is a convenience method that creates an instance of the {@link RoutingConfig.Builder} avoiding the need
         * to create one manually via {@link RoutingConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RoutingConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #routingConfig(RoutingConfig)}.
         * 
         * @param routingConfig
         *        a consumer that will call methods on {@link RoutingConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #routingConfig(RoutingConfig)
         */
        default Builder routingConfig(Consumer<RoutingConfig.Builder> routingConfig) {
            return routingConfig(RoutingConfig.builder().applyMutation(routingConfig).build());
        }

        /**
         * <p>
         * Whether event replication was enabled or disabled for this endpoint. The default state is
         * <code>ENABLED</code> which means you must supply a <code>RoleArn</code>. If you don't have a
         * <code>RoleArn</code> or you don't want event replication enabled, set the state to <code>DISABLED</code>.
         * </p>
         * 
         * @param replicationConfig
         *        Whether event replication was enabled or disabled for this endpoint. The default state is
         *        <code>ENABLED</code> which means you must supply a <code>RoleArn</code>. If you don't have a
         *        <code>RoleArn</code> or you don't want event replication enabled, set the state to
         *        <code>DISABLED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationConfig(ReplicationConfig replicationConfig);

        /**
         * <p>
         * Whether event replication was enabled or disabled for this endpoint. The default state is
         * <code>ENABLED</code> which means you must supply a <code>RoleArn</code>. If you don't have a
         * <code>RoleArn</code> or you don't want event replication enabled, set the state to <code>DISABLED</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ReplicationConfig.Builder} avoiding the
         * need to create one manually via {@link ReplicationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ReplicationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #replicationConfig(ReplicationConfig)}.
         * 
         * @param replicationConfig
         *        a consumer that will call methods on {@link ReplicationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #replicationConfig(ReplicationConfig)
         */
        default Builder replicationConfig(Consumer<ReplicationConfig.Builder> replicationConfig) {
            return replicationConfig(ReplicationConfig.builder().applyMutation(replicationConfig).build());
        }

        /**
         * <p>
         * The event buses being used by the endpoint.
         * </p>
         * 
         * @param eventBuses
         *        The event buses being used by the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventBuses(Collection<EndpointEventBus> eventBuses);

        /**
         * <p>
         * The event buses being used by the endpoint.
         * </p>
         * 
         * @param eventBuses
         *        The event buses being used by the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventBuses(EndpointEventBus... eventBuses);

        /**
         * <p>
         * The event buses being used by the endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.eventbridge.model.EndpointEventBus.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.eventbridge.model.EndpointEventBus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.eventbridge.model.EndpointEventBus.Builder#build()} is called
         * immediately and its result is passed to {@link #eventBuses(List<EndpointEventBus>)}.
         * 
         * @param eventBuses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.eventbridge.model.EndpointEventBus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eventBuses(java.util.Collection<EndpointEventBus>)
         */
        Builder eventBuses(Consumer<EndpointEventBus.Builder>... eventBuses);

        /**
         * <p>
         * The ARN of the role used by event replication for the endpoint.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the role used by event replication for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The URL subdomain of the endpoint. For example, if the URL for Endpoint is
         * https://abcde.veo.endpoints.event.amazonaws.com, then the EndpointId is <code>abcde.veo</code>.
         * </p>
         * 
         * @param endpointId
         *        The URL subdomain of the endpoint. For example, if the URL for Endpoint is
         *        https://abcde.veo.endpoints.event.amazonaws.com, then the EndpointId is <code>abcde.veo</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointId(String endpointId);

        /**
         * <p>
         * The URL of the endpoint.
         * </p>
         * 
         * @param endpointUrl
         *        The URL of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointUrl(String endpointUrl);

        /**
         * <p>
         * The current state of the endpoint.
         * </p>
         * 
         * @param state
         *        The current state of the endpoint.
         * @see EndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the endpoint.
         * </p>
         * 
         * @param state
         *        The current state of the endpoint.
         * @see EndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointState
         */
        Builder state(EndpointState state);

        /**
         * <p>
         * The reason the endpoint is in its current state.
         * </p>
         * 
         * @param stateReason
         *        The reason the endpoint is in its current state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReason(String stateReason);

        /**
         * <p>
         * The time the endpoint was created.
         * </p>
         * 
         * @param creationTime
         *        The time the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The last time the endpoint was modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The last time the endpoint was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String arn;

        private RoutingConfig routingConfig;

        private ReplicationConfig replicationConfig;

        private List<EndpointEventBus> eventBuses = DefaultSdkAutoConstructList.getInstance();

        private String roleArn;

        private String endpointId;

        private String endpointUrl;

        private String state;

        private String stateReason;

        private Instant creationTime;

        private Instant lastModifiedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            name(model.name);
            description(model.description);
            arn(model.arn);
            routingConfig(model.routingConfig);
            replicationConfig(model.replicationConfig);
            eventBuses(model.eventBuses);
            roleArn(model.roleArn);
            endpointId(model.endpointId);
            endpointUrl(model.endpointUrl);
            state(model.state);
            stateReason(model.stateReason);
            creationTime(model.creationTime);
            lastModifiedTime(model.lastModifiedTime);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final RoutingConfig.Builder getRoutingConfig() {
            return routingConfig != null ? routingConfig.toBuilder() : null;
        }

        public final void setRoutingConfig(RoutingConfig.BuilderImpl routingConfig) {
            this.routingConfig = routingConfig != null ? routingConfig.build() : null;
        }

        @Override
        public final Builder routingConfig(RoutingConfig routingConfig) {
            this.routingConfig = routingConfig;
            return this;
        }

        public final ReplicationConfig.Builder getReplicationConfig() {
            return replicationConfig != null ? replicationConfig.toBuilder() : null;
        }

        public final void setReplicationConfig(ReplicationConfig.BuilderImpl replicationConfig) {
            this.replicationConfig = replicationConfig != null ? replicationConfig.build() : null;
        }

        @Override
        public final Builder replicationConfig(ReplicationConfig replicationConfig) {
            this.replicationConfig = replicationConfig;
            return this;
        }

        public final List<EndpointEventBus.Builder> getEventBuses() {
            List<EndpointEventBus.Builder> result = EndpointEventBusListCopier.copyToBuilder(this.eventBuses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEventBuses(Collection<EndpointEventBus.BuilderImpl> eventBuses) {
            this.eventBuses = EndpointEventBusListCopier.copyFromBuilder(eventBuses);
        }

        @Override
        public final Builder eventBuses(Collection<EndpointEventBus> eventBuses) {
            this.eventBuses = EndpointEventBusListCopier.copy(eventBuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventBuses(EndpointEventBus... eventBuses) {
            eventBuses(Arrays.asList(eventBuses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventBuses(Consumer<EndpointEventBus.Builder>... eventBuses) {
            eventBuses(Stream.of(eventBuses).map(c -> EndpointEventBus.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getEndpointId() {
            return endpointId;
        }

        public final void setEndpointId(String endpointId) {
            this.endpointId = endpointId;
        }

        @Override
        public final Builder endpointId(String endpointId) {
            this.endpointId = endpointId;
            return this;
        }

        public final String getEndpointUrl() {
            return endpointUrl;
        }

        public final void setEndpointUrl(String endpointUrl) {
            this.endpointUrl = endpointUrl;
        }

        @Override
        public final Builder endpointUrl(String endpointUrl) {
            this.endpointUrl = endpointUrl;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(EndpointState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStateReason() {
            return stateReason;
        }

        public final void setStateReason(String stateReason) {
            this.stateReason = stateReason;
        }

        @Override
        public final Builder stateReason(String stateReason) {
            this.stateReason = stateReason;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
