/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a projected utilization metric of a recommendation option.
 * </p>
 * <note>
 * <p>
 * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned when you run
 * the <a>GetEC2RecommendationProjectedMetrics</a> action. Additionally, the <code>Memory</code> metric is returned only
 * for resources that have the unified CloudWatch agent installed on them. For more information, see <a
 * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization with
 * the CloudWatch Agent</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecommendedOptionProjectedMetric implements SdkPojo, Serializable,
        ToCopyableBuilder<RecommendedOptionProjectedMetric.Builder, RecommendedOptionProjectedMetric> {
    private static final SdkField<String> RECOMMENDED_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recommendedInstanceType").getter(getter(RecommendedOptionProjectedMetric::recommendedInstanceType))
            .setter(setter(Builder::recommendedInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendedInstanceType").build())
            .build();

    private static final SdkField<Integer> RANK_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("rank")
            .getter(getter(RecommendedOptionProjectedMetric::rank)).setter(setter(Builder::rank))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rank").build()).build();

    private static final SdkField<List<ProjectedMetric>> PROJECTED_METRICS_FIELD = SdkField
            .<List<ProjectedMetric>> builder(MarshallingType.LIST)
            .memberName("projectedMetrics")
            .getter(getter(RecommendedOptionProjectedMetric::projectedMetrics))
            .setter(setter(Builder::projectedMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("projectedMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProjectedMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProjectedMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            RECOMMENDED_INSTANCE_TYPE_FIELD, RANK_FIELD, PROJECTED_METRICS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String recommendedInstanceType;

    private final Integer rank;

    private final List<ProjectedMetric> projectedMetrics;

    private RecommendedOptionProjectedMetric(BuilderImpl builder) {
        this.recommendedInstanceType = builder.recommendedInstanceType;
        this.rank = builder.rank;
        this.projectedMetrics = builder.projectedMetrics;
    }

    /**
     * <p>
     * The recommended instance type.
     * </p>
     * 
     * @return The recommended instance type.
     */
    public final String recommendedInstanceType() {
        return recommendedInstanceType;
    }

    /**
     * <p>
     * The rank of the recommendation option projected metric.
     * </p>
     * <p>
     * The top recommendation option is ranked as <code>1</code>.
     * </p>
     * <p>
     * The projected metric rank correlates to the recommendation option rank. For example, the projected metric ranked
     * as <code>1</code> is related to the recommendation option that is also ranked as <code>1</code> in the same
     * response.
     * </p>
     * 
     * @return The rank of the recommendation option projected metric.</p>
     *         <p>
     *         The top recommendation option is ranked as <code>1</code>.
     *         </p>
     *         <p>
     *         The projected metric rank correlates to the recommendation option rank. For example, the projected metric
     *         ranked as <code>1</code> is related to the recommendation option that is also ranked as <code>1</code> in
     *         the same response.
     */
    public final Integer rank() {
        return rank;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProjectedMetrics property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProjectedMetrics() {
        return projectedMetrics != null && !(projectedMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe a projected utilization metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProjectedMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe a projected utilization metric.
     */
    public final List<ProjectedMetric> projectedMetrics() {
        return projectedMetrics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(recommendedInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(rank());
        hashCode = 31 * hashCode + Objects.hashCode(hasProjectedMetrics() ? projectedMetrics() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecommendedOptionProjectedMetric)) {
            return false;
        }
        RecommendedOptionProjectedMetric other = (RecommendedOptionProjectedMetric) obj;
        return Objects.equals(recommendedInstanceType(), other.recommendedInstanceType()) && Objects.equals(rank(), other.rank())
                && hasProjectedMetrics() == other.hasProjectedMetrics()
                && Objects.equals(projectedMetrics(), other.projectedMetrics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecommendedOptionProjectedMetric").add("RecommendedInstanceType", recommendedInstanceType())
                .add("Rank", rank()).add("ProjectedMetrics", hasProjectedMetrics() ? projectedMetrics() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "recommendedInstanceType":
            return Optional.ofNullable(clazz.cast(recommendedInstanceType()));
        case "rank":
            return Optional.ofNullable(clazz.cast(rank()));
        case "projectedMetrics":
            return Optional.ofNullable(clazz.cast(projectedMetrics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecommendedOptionProjectedMetric, T> g) {
        return obj -> g.apply((RecommendedOptionProjectedMetric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecommendedOptionProjectedMetric> {
        /**
         * <p>
         * The recommended instance type.
         * </p>
         * 
         * @param recommendedInstanceType
         *        The recommended instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendedInstanceType(String recommendedInstanceType);

        /**
         * <p>
         * The rank of the recommendation option projected metric.
         * </p>
         * <p>
         * The top recommendation option is ranked as <code>1</code>.
         * </p>
         * <p>
         * The projected metric rank correlates to the recommendation option rank. For example, the projected metric
         * ranked as <code>1</code> is related to the recommendation option that is also ranked as <code>1</code> in the
         * same response.
         * </p>
         * 
         * @param rank
         *        The rank of the recommendation option projected metric.</p>
         *        <p>
         *        The top recommendation option is ranked as <code>1</code>.
         *        </p>
         *        <p>
         *        The projected metric rank correlates to the recommendation option rank. For example, the projected
         *        metric ranked as <code>1</code> is related to the recommendation option that is also ranked as
         *        <code>1</code> in the same response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rank(Integer rank);

        /**
         * <p>
         * An array of objects that describe a projected utilization metric.
         * </p>
         * 
         * @param projectedMetrics
         *        An array of objects that describe a projected utilization metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedMetrics(Collection<ProjectedMetric> projectedMetrics);

        /**
         * <p>
         * An array of objects that describe a projected utilization metric.
         * </p>
         * 
         * @param projectedMetrics
         *        An array of objects that describe a projected utilization metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedMetrics(ProjectedMetric... projectedMetrics);

        /**
         * <p>
         * An array of objects that describe a projected utilization metric.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ProjectedMetric.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ProjectedMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ProjectedMetric.Builder#build()} is called
         * immediately and its result is passed to {@link #projectedMetrics(List<ProjectedMetric>)}.
         * 
         * @param projectedMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.ProjectedMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projectedMetrics(java.util.Collection<ProjectedMetric>)
         */
        Builder projectedMetrics(Consumer<ProjectedMetric.Builder>... projectedMetrics);
    }

    static final class BuilderImpl implements Builder {
        private String recommendedInstanceType;

        private Integer rank;

        private List<ProjectedMetric> projectedMetrics = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RecommendedOptionProjectedMetric model) {
            recommendedInstanceType(model.recommendedInstanceType);
            rank(model.rank);
            projectedMetrics(model.projectedMetrics);
        }

        public final String getRecommendedInstanceType() {
            return recommendedInstanceType;
        }

        public final void setRecommendedInstanceType(String recommendedInstanceType) {
            this.recommendedInstanceType = recommendedInstanceType;
        }

        @Override
        public final Builder recommendedInstanceType(String recommendedInstanceType) {
            this.recommendedInstanceType = recommendedInstanceType;
            return this;
        }

        public final Integer getRank() {
            return rank;
        }

        public final void setRank(Integer rank) {
            this.rank = rank;
        }

        @Override
        public final Builder rank(Integer rank) {
            this.rank = rank;
            return this;
        }

        public final List<ProjectedMetric.Builder> getProjectedMetrics() {
            List<ProjectedMetric.Builder> result = ProjectedMetricsCopier.copyToBuilder(this.projectedMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProjectedMetrics(Collection<ProjectedMetric.BuilderImpl> projectedMetrics) {
            this.projectedMetrics = ProjectedMetricsCopier.copyFromBuilder(projectedMetrics);
        }

        @Override
        public final Builder projectedMetrics(Collection<ProjectedMetric> projectedMetrics) {
            this.projectedMetrics = ProjectedMetricsCopier.copy(projectedMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedMetrics(ProjectedMetric... projectedMetrics) {
            projectedMetrics(Arrays.asList(projectedMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedMetrics(Consumer<ProjectedMetric.Builder>... projectedMetrics) {
            projectedMetrics(Stream.of(projectedMetrics).map(c -> ProjectedMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public RecommendedOptionProjectedMetric build() {
            return new RecommendedOptionProjectedMetric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
