/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the projected metrics of an Amazon ECS service recommendation option.
 * </p>
 * <p>
 * To determine the performance difference between your current Amazon ECS service and the recommended option, compare
 * the metric data of your service against its projected metric data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ECSServiceProjectedMetric implements SdkPojo, Serializable,
        ToCopyableBuilder<ECSServiceProjectedMetric.Builder, ECSServiceProjectedMetric> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ECSServiceProjectedMetric::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<Instant>> TIMESTAMPS_FIELD = SdkField
            .<List<Instant>> builder(MarshallingType.LIST)
            .memberName("timestamps")
            .getter(getter(ECSServiceProjectedMetric::timestamps))
            .setter(setter(Builder::timestamps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timestamps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Instant> builder(MarshallingType.INSTANT)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> UPPER_BOUND_VALUES_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("upperBoundValues")
            .getter(getter(ECSServiceProjectedMetric::upperBoundValues))
            .setter(setter(Builder::upperBoundValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("upperBoundValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> LOWER_BOUND_VALUES_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("lowerBoundValues")
            .getter(getter(ECSServiceProjectedMetric::lowerBoundValues))
            .setter(setter(Builder::lowerBoundValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lowerBoundValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TIMESTAMPS_FIELD,
            UPPER_BOUND_VALUES_FIELD, LOWER_BOUND_VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<Instant> timestamps;

    private final List<Double> upperBoundValues;

    private final List<Double> lowerBoundValues;

    private ECSServiceProjectedMetric(BuilderImpl builder) {
        this.name = builder.name;
        this.timestamps = builder.timestamps;
        this.upperBoundValues = builder.upperBoundValues;
        this.lowerBoundValues = builder.lowerBoundValues;
    }

    /**
     * <p>
     * The name of the projected metric.
     * </p>
     * <p>
     * The following metrics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service tasks.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link ECSServiceMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the projected metric. </p>
     *         <p>
     *         The following metrics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service
     *         tasks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
     *         </p>
     *         </li>
     * @see ECSServiceMetricName
     */
    public final ECSServiceMetricName name() {
        return ECSServiceMetricName.fromValue(name);
    }

    /**
     * <p>
     * The name of the projected metric.
     * </p>
     * <p>
     * The following metrics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service tasks.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link ECSServiceMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the projected metric. </p>
     *         <p>
     *         The following metrics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service
     *         tasks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
     *         </p>
     *         </li>
     * @see ECSServiceMetricName
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Timestamps property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTimestamps() {
        return timestamps != null && !(timestamps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The timestamps of the projected metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTimestamps} method.
     * </p>
     * 
     * @return The timestamps of the projected metric.
     */
    public final List<Instant> timestamps() {
        return timestamps;
    }

    /**
     * For responses, this returns true if the service returned a value for the UpperBoundValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUpperBoundValues() {
        return upperBoundValues != null && !(upperBoundValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The upper bound values for the projected metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUpperBoundValues} method.
     * </p>
     * 
     * @return The upper bound values for the projected metric.
     */
    public final List<Double> upperBoundValues() {
        return upperBoundValues;
    }

    /**
     * For responses, this returns true if the service returned a value for the LowerBoundValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLowerBoundValues() {
        return lowerBoundValues != null && !(lowerBoundValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The lower bound values for the projected metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLowerBoundValues} method.
     * </p>
     * 
     * @return The lower bound values for the projected metric.
     */
    public final List<Double> lowerBoundValues() {
        return lowerBoundValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTimestamps() ? timestamps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUpperBoundValues() ? upperBoundValues() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLowerBoundValues() ? lowerBoundValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ECSServiceProjectedMetric)) {
            return false;
        }
        ECSServiceProjectedMetric other = (ECSServiceProjectedMetric) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && hasTimestamps() == other.hasTimestamps()
                && Objects.equals(timestamps(), other.timestamps()) && hasUpperBoundValues() == other.hasUpperBoundValues()
                && Objects.equals(upperBoundValues(), other.upperBoundValues())
                && hasLowerBoundValues() == other.hasLowerBoundValues()
                && Objects.equals(lowerBoundValues(), other.lowerBoundValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ECSServiceProjectedMetric").add("Name", nameAsString())
                .add("Timestamps", hasTimestamps() ? timestamps() : null)
                .add("UpperBoundValues", hasUpperBoundValues() ? upperBoundValues() : null)
                .add("LowerBoundValues", hasLowerBoundValues() ? lowerBoundValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "timestamps":
            return Optional.ofNullable(clazz.cast(timestamps()));
        case "upperBoundValues":
            return Optional.ofNullable(clazz.cast(upperBoundValues()));
        case "lowerBoundValues":
            return Optional.ofNullable(clazz.cast(lowerBoundValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ECSServiceProjectedMetric, T> g) {
        return obj -> g.apply((ECSServiceProjectedMetric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ECSServiceProjectedMetric> {
        /**
         * <p>
         * The name of the projected metric.
         * </p>
         * <p>
         * The following metrics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service tasks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the projected metric. </p>
         *        <p>
         *        The following metrics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service
         *        tasks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
         *        </p>
         *        </li>
         * @see ECSServiceMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ECSServiceMetricName
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the projected metric.
         * </p>
         * <p>
         * The following metrics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service tasks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the projected metric. </p>
         *        <p>
         *        The following metrics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Cpu</code> — The percentage of allocated compute units that are currently in use on the service
         *        tasks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memory</code> — The percentage of memory that's currently in use on the service tasks.
         *        </p>
         *        </li>
         * @see ECSServiceMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ECSServiceMetricName
         */
        Builder name(ECSServiceMetricName name);

        /**
         * <p>
         * The timestamps of the projected metric.
         * </p>
         * 
         * @param timestamps
         *        The timestamps of the projected metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Collection<Instant> timestamps);

        /**
         * <p>
         * The timestamps of the projected metric.
         * </p>
         * 
         * @param timestamps
         *        The timestamps of the projected metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Instant... timestamps);

        /**
         * <p>
         * The upper bound values for the projected metric.
         * </p>
         * 
         * @param upperBoundValues
         *        The upper bound values for the projected metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upperBoundValues(Collection<Double> upperBoundValues);

        /**
         * <p>
         * The upper bound values for the projected metric.
         * </p>
         * 
         * @param upperBoundValues
         *        The upper bound values for the projected metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upperBoundValues(Double... upperBoundValues);

        /**
         * <p>
         * The lower bound values for the projected metric.
         * </p>
         * 
         * @param lowerBoundValues
         *        The lower bound values for the projected metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lowerBoundValues(Collection<Double> lowerBoundValues);

        /**
         * <p>
         * The lower bound values for the projected metric.
         * </p>
         * 
         * @param lowerBoundValues
         *        The lower bound values for the projected metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lowerBoundValues(Double... lowerBoundValues);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<Instant> timestamps = DefaultSdkAutoConstructList.getInstance();

        private List<Double> upperBoundValues = DefaultSdkAutoConstructList.getInstance();

        private List<Double> lowerBoundValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ECSServiceProjectedMetric model) {
            name(model.name);
            timestamps(model.timestamps);
            upperBoundValues(model.upperBoundValues);
            lowerBoundValues(model.lowerBoundValues);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(ECSServiceMetricName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final Collection<Instant> getTimestamps() {
            if (timestamps instanceof SdkAutoConstructList) {
                return null;
            }
            return timestamps;
        }

        public final void setTimestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
        }

        @Override
        public final Builder timestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder timestamps(Instant... timestamps) {
            timestamps(Arrays.asList(timestamps));
            return this;
        }

        public final Collection<Double> getUpperBoundValues() {
            if (upperBoundValues instanceof SdkAutoConstructList) {
                return null;
            }
            return upperBoundValues;
        }

        public final void setUpperBoundValues(Collection<Double> upperBoundValues) {
            this.upperBoundValues = MetricValuesCopier.copy(upperBoundValues);
        }

        @Override
        public final Builder upperBoundValues(Collection<Double> upperBoundValues) {
            this.upperBoundValues = MetricValuesCopier.copy(upperBoundValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder upperBoundValues(Double... upperBoundValues) {
            upperBoundValues(Arrays.asList(upperBoundValues));
            return this;
        }

        public final Collection<Double> getLowerBoundValues() {
            if (lowerBoundValues instanceof SdkAutoConstructList) {
                return null;
            }
            return lowerBoundValues;
        }

        public final void setLowerBoundValues(Collection<Double> lowerBoundValues) {
            this.lowerBoundValues = MetricValuesCopier.copy(lowerBoundValues);
        }

        @Override
        public final Builder lowerBoundValues(Collection<Double> lowerBoundValues) {
            this.lowerBoundValues = MetricValuesCopier.copy(lowerBoundValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lowerBoundValues(Double... lowerBoundValues) {
            lowerBoundValues(Arrays.asList(lowerBoundValues));
            return this;
        }

        @Override
        public ECSServiceProjectedMetric build() {
            return new ECSServiceProjectedMetric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
