// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import org.refcodes.component.AbstractConnectableAutomaton;
import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;

/**
 * The {@link AbstractByteReceiver} is a base abstract implementation of the
 * {@link ByteReceiver} interface providing common functionality for concrete
 * real live {@link ByteDatagramReceiver} and {@link ByteBlockReceiver} (=
 * {@link ByteReceiver}) implementations.
 * 
 * A blocking queue is used internally to which received datagrams are put via
 * {@link #pushDatagram(byte)} and which can be retrieved via
 * {@link #readDatagram()}. The {@link #pushDatagram(byte)} method is your hook
 * when extending this class.
 * 
 * Make sure your code fetches the datagrams quick enough to prevent filling up
 * of the queue. In case the queue is filled up, adding elements via
 * {@link #pushDatagram(byte)} to the queue is blocked until elements are taken
 * from the queue via {@link #readDatagram()}. So cautions are taken to prevent
 * a memory leak.
 */
public abstract class AbstractByteReceiver extends AbstractByteProvider implements ByteReceiver {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected ConnectableAutomaton _automaton = new ConnectableAutomaton();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link AbstractByteReceiver} with a default sized blocking
	 * queue enabling a maximum of {@link #DATAGRAM_QUEUE_SIZE} datagrams.
	 * -------------------------------------------------------------------------
	 * Make sure your code fetches the datagrams quick enough to prevent filling
	 * up of the queue. In case the queue is filled up, adding elements via
	 * {@link #pushDatagram(byte)} to the queue is blocked until elements are
	 * taken from the queue via {@link #readDatagram()}.
	 */
	public AbstractByteReceiver() {
		super();
	}

	/**
	 * Constructs a {@link AbstractByteReceiver} with a custom sized blocking
	 * queue enabling a maximum of datagrams as specified by the capacity
	 * parameter.
	 * -------------------------------------------------------------------------
	 * Make sure your code fetches the datagrams quick enough to prevent filling
	 * up of the queue. In case the queue is filled up, adding elements via
	 * {@link #pushDatagram(byte)} to the queue is blocked until elements are
	 * taken from the queue via {@link #readDatagram()}.
	 * 
	 * @param aCapacity The capacity of the queue holding the received
	 *        datagrams.
	 */
	public AbstractByteReceiver( int aCapacity ) {
		super( aCapacity );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public byte readDatagram() throws OpenException, InterruptedException {
		if ( _datagramQueue.isEmpty() && !isOpened() ) {
			throw new OpenException( "Unable to read datagram  as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		return super.readDatagram();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public byte[] readDatagrams() throws OpenException, InterruptedException {
		if ( _datagramQueue.isEmpty() && !isOpened() ) {
			throw new OpenException( "Unable to read datagram  as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		return super.readDatagrams();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public byte[] readDatagrams( int aBlockSize ) throws OpenException, InterruptedException {
		if ( _datagramQueue.isEmpty() && !isOpened() ) {
			throw new OpenException( "Unable to read datagram  as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		return super.readDatagrams( aBlockSize );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasDatagram() throws OpenException {
		// Enable the caller to get all elements from the queue:
		// if ( _datagramQueue.isEmpty() && !isOpened() ) { throw new
		// OpenException( "Unable to read datagram as the connection is NOT
		// OPEN; connection status is " + getConnectionStatus() + "." ); }
		return !_datagramQueue.isEmpty();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		if ( isOpened() ) {
			_automaton.close();
			releaseAll();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void releaseAll() {
		synchronized ( _datagramQueue ) {
			_datagramQueue.notifyAll();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isOpened() {
		return _automaton.isOpened();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isClosable() {
		return _automaton.isClosable();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isClosed() {
		return _automaton.isClosed();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionStatus getConnectionStatus() {
		return _automaton.getConnectionStatus();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Open.
	 *
	 * @throws OpenException the open exception
	 */
	protected void open() throws OpenException {
		_automaton.open();
	}

	/**
	 * Sets the connection status.
	 *
	 * @param aConnectionStatus the new connection status
	 */
	protected void setConnectionStatus( ConnectionStatus aConnectionStatus ) {
		_automaton.setConnectionStatus( aConnectionStatus );
	}

	/**
	 * Checks if is openable.
	 *
	 * @return true, if is openable
	 */
	protected boolean isOpenable() {
		return _automaton.isOpenable();
	}

	///////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class ConnectableAutomaton.
	 */
	protected static class ConnectableAutomaton extends AbstractConnectableAutomaton {

		/**
		 * Checks if is openable.
		 *
		 * @return true, if is openable
		 */
		@Override
		public boolean isOpenable() {
			return super.isOpenable();
		}

		/**
		 * Open.
		 *
		 * @throws OpenException the open exception
		 */
		@Override
		public void open() throws OpenException {
			super.open();
		}

		/**
		 * Sets the connection status.
		 *
		 * @param aConnectionStatus the new connection status
		 */
		@Override
		public void setConnectionStatus( ConnectionStatus aConnectionStatus ) {
			super.setConnectionStatus( aConnectionStatus );
		}

	};
}
