/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.saml.saml2.ecp.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.opensaml.core.xml.AbstractXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.schema.XSBooleanValue;
import org.opensaml.saml.saml2.core.IDPList;
import org.opensaml.saml.saml2.core.Issuer;
import org.opensaml.saml.saml2.ecp.Request;
import org.opensaml.soap.soap11.ActorBearing;
import org.opensaml.soap.soap11.MustUnderstandBearing;

/**
 *  A concrete implementation of {@link Request}.
 */
public class RequestImpl extends AbstractXMLObject implements Request {
    
    /** IDPList child element. */
    private IDPList idpList;
    
    /** Issuer child element. */
    private Issuer issuer;
    
    /** ProviderName attribute. */
    private String providerName;
    
    /** IsPassive attribute value. */
    private XSBooleanValue isPassive;
    
    /** soap11:actor attribute. */
    private String soap11Actor;
    
    /** soap11:mustUnderstand. */
    private XSBooleanValue soap11MustUnderstand;
    
    /**
     * Constructor.
     * 
     * @param namespaceURI the namespace the element is in
     * @param elementLocalName the local name of the XML element this Object represents
     * @param namespacePrefix the prefix for the given namespace
     */
    protected RequestImpl(final String namespaceURI, final String elementLocalName, final String namespacePrefix) {
        super(namespaceURI, elementLocalName, namespacePrefix);
    }

    /** {@inheritDoc} */
    public IDPList getIDPList() {
        return idpList;
    }
    
    /** {@inheritDoc} */
    public void setIDPList(final IDPList newIDPList) {
        idpList = prepareForAssignment(idpList, newIDPList);
    }

    /** {@inheritDoc} */
    public Issuer getIssuer() {
        return issuer;
    }
    
    /** {@inheritDoc} */
    public void setIssuer(final Issuer newIssuer) {
        issuer = prepareForAssignment(issuer, newIssuer);
    }

    /** {@inheritDoc} */
    public String getProviderName() {
        return providerName;
    }

    /** {@inheritDoc} */
    public void setProviderName(final String newProviderName) {
        providerName = prepareForAssignment(providerName, newProviderName);
    }

    /** {@inheritDoc} */
    public Boolean isPassive() {
        if (isPassive != null) {
            return isPassive.getValue();
        }

        return Boolean.FALSE;
    }

    /** {@inheritDoc} */
    public XSBooleanValue isPassiveXSBoolean() {
        return isPassive;
    }

    /** {@inheritDoc} */
    public void setPassive(final Boolean newIsPassive) {
        if (newIsPassive != null) {
            isPassive = prepareForAssignment(isPassive, new XSBooleanValue(newIsPassive, false));
        } else {
            isPassive = prepareForAssignment(isPassive, null);
        }
    }

    /** {@inheritDoc} */
    public void setPassive(final XSBooleanValue newIsPassive) {
        this.isPassive = prepareForAssignment(this.isPassive, newIsPassive);
    }
    
    /** {@inheritDoc} */
    public Boolean isSOAP11MustUnderstand() {
        if (soap11MustUnderstand != null) {
            return soap11MustUnderstand.getValue();
        }
        return Boolean.FALSE;
    }

    /** {@inheritDoc} */
    public XSBooleanValue isSOAP11MustUnderstandXSBoolean() {
        return soap11MustUnderstand;
    }

    /** {@inheritDoc} */
    public void setSOAP11MustUnderstand(final Boolean newMustUnderstand) {
        if (newMustUnderstand != null) {
            soap11MustUnderstand = prepareForAssignment(soap11MustUnderstand, 
                    new XSBooleanValue(newMustUnderstand, true));
        } else {
            soap11MustUnderstand = prepareForAssignment(soap11MustUnderstand, null);
        }
        manageQualifiedAttributeNamespace(MustUnderstandBearing.SOAP11_MUST_UNDERSTAND_ATTR_NAME, 
                soap11MustUnderstand != null);
    }

    /** {@inheritDoc} */
    public void setSOAP11MustUnderstand(final XSBooleanValue newMustUnderstand) {
            soap11MustUnderstand = prepareForAssignment(soap11MustUnderstand, newMustUnderstand);
            manageQualifiedAttributeNamespace(MustUnderstandBearing.SOAP11_MUST_UNDERSTAND_ATTR_NAME, 
                    soap11MustUnderstand != null);
    }

    /** {@inheritDoc} */
    public String getSOAP11Actor() {
        return soap11Actor;
    }

    /** {@inheritDoc} */
    public void setSOAP11Actor(final String newActor) {
        soap11Actor = prepareForAssignment(soap11Actor, newActor);
        manageQualifiedAttributeNamespace(ActorBearing.SOAP11_ACTOR_ATTR_NAME, soap11Actor != null);
    }

    /** {@inheritDoc} */
    public List<XMLObject> getOrderedChildren() {
        final ArrayList<XMLObject> children = new ArrayList<>();
        if (issuer!=null) {
            children.add(issuer);
        }
        if (idpList!=null) {
            children.add(idpList);
        }
        return Collections.unmodifiableList(children);
    }

}
