/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.internal.helpers.progress;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.function.BooleanSupplier;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.api.parallel.ResourceLock;
import org.mockito.ArgumentMatchers;
import org.mockito.InOrder;
import org.mockito.Mockito;
import org.mockito.stubbing.Answer;
import org.neo4j.internal.helpers.progress.Indicator;
import org.neo4j.internal.helpers.progress.ProgressListener;
import org.neo4j.internal.helpers.progress.ProgressMonitorFactory;
import org.neo4j.test.Race;
import org.neo4j.test.extension.SuppressOutputExtension;

@ExtendWith(value={SuppressOutputExtension.class})
@ResourceLock(value="java.lang.System.out")
class ProgressMonitorTest {
    private static final String EXPECTED_TEXTUAL_OUTPUT = ProgressMonitorTest.buildExpectedOutput();
    private Indicator indicator;
    private ProgressMonitorFactory factory;

    ProgressMonitorTest() {
    }

    @BeforeEach
    void setUp() {
        this.indicator = ProgressMonitorTest.indicatorMock();
        Mockito.when((Object)this.indicator.reportResolution()).thenReturn((Object)10);
        this.factory = (ProgressMonitorFactory)Mockito.mock(ProgressMonitorFactory.class);
        Mockito.when((Object)this.factory.newIndicator((String)ArgumentMatchers.any(String.class))).thenReturn((Object)this.indicator);
    }

    @Test
    void shouldReportProgressInTheSpecifiedIntervals(TestInfo testInfo) {
        ProgressListener progressListener = this.factory.singlePart(testInfo.getDisplayName(), 16L);
        progressListener.started();
        for (int i = 0; i < 16; ++i) {
            progressListener.add(1L);
        }
        progressListener.done();
        InOrder order = Mockito.inOrder((Object[])new Object[]{this.indicator});
        ((Indicator)order.verify((Object)this.indicator)).startProcess(16L);
        for (int i = 0; i < 10; ++i) {
            ((Indicator)order.verify((Object)this.indicator)).progress(i, i + 1);
        }
        ((Indicator)order.verify((Object)this.indicator)).completeProcess();
        order.verifyNoMoreInteractions();
    }

    @Test
    void shouldAggregateProgressFromMultipleProcesses(TestInfo testInfo) {
        int i;
        ProgressMonitorFactory.MultiPartBuilder builder = this.factory.multipleParts(testInfo.getDisplayName());
        ProgressListener first = builder.progressForPart("first", 5L);
        ProgressListener other = builder.progressForPart("other", 5L);
        builder.build();
        InOrder order = Mockito.inOrder((Object[])new Object[]{this.indicator});
        ((Indicator)order.verify((Object)this.indicator)).startProcess(10L);
        order.verifyNoMoreInteractions();
        first.started();
        for (i = 0; i < 5; ++i) {
            first.add(1L);
        }
        first.done();
        ((Indicator)order.verify((Object)this.indicator)).startPart("first", 5L);
        for (i = 0; i < 5; ++i) {
            ((Indicator)order.verify((Object)this.indicator)).progress(i, i + 1);
        }
        ((Indicator)order.verify((Object)this.indicator)).completePart("first");
        order.verifyNoMoreInteractions();
        other.started();
        for (i = 0; i < 5; ++i) {
            other.add(1L);
        }
        other.done();
        ((Indicator)order.verify((Object)this.indicator)).startPart("other", 5L);
        for (i = 5; i < 10; ++i) {
            ((Indicator)order.verify((Object)this.indicator)).progress(i, i + 1);
        }
        ((Indicator)order.verify((Object)this.indicator)).completePart("other");
        ((Indicator)order.verify((Object)this.indicator)).completeProcess();
        order.verifyNoMoreInteractions();
    }

    @Test
    void shouldNotAllowAddingPartsAfterCompletingMultiPartBuilder(TestInfo testInfo) {
        ProgressMonitorFactory.MultiPartBuilder builder = this.factory.multipleParts(testInfo.getDisplayName());
        builder.progressForPart("first", 10L);
        builder.build();
        IllegalStateException exception = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> builder.progressForPart("other", 10L));
        Assertions.assertEquals((Object)"Builder has been completed.", (Object)exception.getMessage());
    }

    @Test
    void shouldNotAllowAddingMultiplePartsWithSameIdentifier(TestInfo testInfo) {
        ProgressMonitorFactory.MultiPartBuilder builder = ((ProgressMonitorFactory)Mockito.mock(ProgressMonitorFactory.class)).multipleParts(testInfo.getDisplayName());
        builder.progressForPart("first", 10L);
        IllegalArgumentException exception = (IllegalArgumentException)Assertions.assertThrows(IllegalArgumentException.class, () -> builder.progressForPart("first", 10L));
        Assertions.assertEquals((Object)"Part 'first' has already been defined.", (Object)exception.getMessage());
    }

    @Test
    void shouldStartProcessAutomaticallyIfNotDoneBefore(TestInfo testInfo) {
        ProgressListener progressListener = this.factory.singlePart(testInfo.getDisplayName(), 16L);
        for (int i = 0; i < 16; ++i) {
            progressListener.add(1L);
        }
        progressListener.done();
        InOrder order = Mockito.inOrder((Object[])new Object[]{this.indicator});
        ((Indicator)order.verify((Object)this.indicator)).startProcess(16L);
        for (int i = 0; i < 10; ++i) {
            ((Indicator)order.verify((Object)this.indicator)).progress(i, i + 1);
        }
        ((Indicator)order.verify((Object)this.indicator)).completeProcess();
        order.verifyNoMoreInteractions();
    }

    @Test
    void shouldStartMultiPartProcessAutomaticallyIfNotDoneBefore(TestInfo testInfo) {
        int i;
        ProgressMonitorFactory.MultiPartBuilder builder = this.factory.multipleParts(testInfo.getDisplayName());
        ProgressListener first = builder.progressForPart("first", 5L);
        ProgressListener other = builder.progressForPart("other", 5L);
        builder.build();
        InOrder order = Mockito.inOrder((Object[])new Object[]{this.indicator});
        ((Indicator)order.verify((Object)this.indicator)).startProcess(10L);
        order.verifyNoMoreInteractions();
        for (i = 0; i < 5; ++i) {
            first.add(1L);
        }
        first.done();
        ((Indicator)order.verify((Object)this.indicator)).startPart("first", 5L);
        for (i = 0; i < 5; ++i) {
            ((Indicator)order.verify((Object)this.indicator)).progress(i, i + 1);
        }
        ((Indicator)order.verify((Object)this.indicator)).completePart("first");
        order.verifyNoMoreInteractions();
        for (i = 0; i < 5; ++i) {
            other.add(1L);
        }
        other.done();
        ((Indicator)order.verify((Object)this.indicator)).startPart("other", 5L);
        for (i = 5; i < 10; ++i) {
            ((Indicator)order.verify((Object)this.indicator)).progress(i, i + 1);
        }
        ((Indicator)order.verify((Object)this.indicator)).completePart("other");
        ((Indicator)order.verify((Object)this.indicator)).completeProcess();
        order.verifyNoMoreInteractions();
    }

    @Test
    void shouldCompleteMultiPartProgressWithNoPartsImmediately(TestInfo testInfo) {
        ProgressMonitorFactory.MultiPartBuilder builder = this.factory.multipleParts(testInfo.getDisplayName());
        builder.build();
        InOrder order = Mockito.inOrder((Object[])new Object[]{this.indicator});
        ((Indicator)order.verify((Object)this.indicator)).startProcess(0L);
        ((Indicator)order.verify((Object)this.indicator)).progress(0, 10);
        ((Indicator)order.verify((Object)this.indicator)).completeProcess();
        order.verifyNoMoreInteractions();
    }

    @Test
    void shouldPrintADotEveryHalfPercentAndFullPercentageEveryTenPercentWithTextualIndicator(TestInfo testInfo) throws Exception {
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        ProgressListener progressListener = ProgressMonitorFactory.textual((OutputStream)stream).singlePart(testInfo.getDisplayName(), 1000L);
        for (int i = 0; i < 1000; ++i) {
            progressListener.add(1L);
        }
        Assertions.assertEquals((Object)(testInfo.getDisplayName() + System.lineSeparator() + EXPECTED_TEXTUAL_OUTPUT), (Object)stream.toString(Charset.defaultCharset().name()));
    }

    @Test
    void shouldPrintADotEveryHalfPercentAndFullPercentageEveryTenPercentEvenWhenStepResolutionIsLower(TestInfo testInfo) {
        StringWriter writer = new StringWriter();
        ProgressListener progressListener = ProgressMonitorFactory.textual((Writer)writer).singlePart(testInfo.getDisplayName(), 50L);
        for (int i = 0; i < 50; ++i) {
            progressListener.add(1L);
        }
        Assertions.assertEquals((Object)(testInfo.getDisplayName() + System.lineSeparator() + EXPECTED_TEXTUAL_OUTPUT), (Object)writer.toString());
    }

    @Test
    void shouldAllowStartingAPartBeforeCompletionOfMultiPartBuilder(TestInfo testInfo) {
        ProgressMonitorFactory.MultiPartBuilder builder = this.factory.multipleParts(testInfo.getDisplayName());
        ProgressListener part1 = builder.progressForPart("part1", 1L);
        ProgressListener part2 = builder.progressForPart("part2", 1L);
        part1.add(1L);
        builder.build();
        part2.add(1L);
        part1.done();
        part2.done();
        InOrder order = Mockito.inOrder((Object[])new Object[]{this.indicator});
        ((Indicator)order.verify((Object)this.indicator)).startPart("part1", 1L);
        ((Indicator)order.verify((Object)this.indicator)).startProcess(2L);
        ((Indicator)order.verify((Object)this.indicator)).startPart("part2", 1L);
        ((Indicator)order.verify((Object)this.indicator)).completePart("part1");
        ((Indicator)order.verify((Object)this.indicator)).completePart("part2");
        ((Indicator)order.verify((Object)this.indicator)).completeProcess();
    }

    @Test
    void shouldAllowConcurrentProgressReportingForMultipartProgress(TestInfo testInfo) {
        ProgressMonitorFactory.MultiPartBuilder builder = this.factory.multipleParts(testInfo.getDisplayName());
        int numberOfThreads = 4;
        int part1LocalCount = 12345;
        int part2LocalCount = 54321;
        int part1TotalCount = numberOfThreads * part1LocalCount;
        int part2TotalCount = numberOfThreads * part2LocalCount;
        ProgressListener part1 = builder.progressForPart("part1", (long)part1TotalCount);
        ProgressListener part2 = builder.progressForPart("part2", (long)part2TotalCount);
        Race race = new Race().withEndCondition(new BooleanSupplier[]{() -> false});
        race.addContestants(numberOfThreads, () -> part1.add(1L), part1LocalCount);
        race.addContestants(numberOfThreads, () -> part2.add(1L), part2LocalCount);
        race.goUnchecked();
        for (int i = 0; i < 10; ++i) {
            ((Indicator)Mockito.verify((Object)this.indicator)).progress(i, i + 1);
        }
    }

    @Test
    void shouldAllowConcurrentProgressReportingForMultipartProgressWithLocalReporting(TestInfo testInfo) {
        ProgressMonitorFactory.MultiPartBuilder builder = this.factory.multipleParts(testInfo.getDisplayName());
        int numberOfThreads = 4;
        int localReportingSize = 2;
        int part1ReportCount = 12345;
        int part2ReportCount = 54321;
        int part1TotalCount = numberOfThreads * part1ReportCount * localReportingSize;
        int part2TotalCount = numberOfThreads * part2ReportCount * localReportingSize;
        ProgressListener part1 = builder.progressForPart("part1", (long)part1TotalCount);
        ProgressListener part2 = builder.progressForPart("part2", (long)part2TotalCount);
        Race race = new Race();
        race.addContestants(numberOfThreads, () -> {
            ProgressListener local = part1.threadLocalReporter(localReportingSize);
            for (int i = 0; i < part1ReportCount * localReportingSize; ++i) {
                local.add(1L);
            }
            local.done();
        });
        race.addContestants(numberOfThreads, () -> {
            ProgressListener local = part2.threadLocalReporter(localReportingSize);
            for (int i = 0; i < part2ReportCount * localReportingSize; ++i) {
                local.add(1L);
            }
            local.done();
        });
        race.goUnchecked();
        for (int i = 0; i < 10; ++i) {
            ((Indicator)Mockito.verify((Object)this.indicator)).progress(i, i + 1);
        }
    }

    private static Indicator indicatorMock() {
        Indicator indicator = (Indicator)Mockito.mock(Indicator.class, (Answer)Mockito.CALLS_REAL_METHODS);
        ((Indicator)Mockito.doNothing().when((Object)indicator)).progress(ArgumentMatchers.anyInt(), ArgumentMatchers.anyInt());
        return indicator;
    }

    private static String buildExpectedOutput() {
        StringBuilder expectedTextualOutput = new StringBuilder();
        int i = 0;
        while (i < 10) {
            for (int j = 0; j < 20; ++j) {
                expectedTextualOutput.append('.');
            }
            expectedTextualOutput.append(String.format(" %3d%%%n", ++i * 10));
        }
        return expectedTextualOutput.toString();
    }
}

