/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.email;

import jakarta.mail.Address;
import jakarta.mail.BodyPart;
import jakarta.mail.Message;
import jakarta.mail.MessagingException;
import jakarta.mail.Multipart;
import jakarta.mail.Session;
import jakarta.mail.Transport;
import jakarta.mail.internet.AddressException;
import jakarta.mail.internet.InternetAddress;
import jakarta.mail.internet.MimeBodyPart;
import jakarta.mail.internet.MimeMessage;
import jakarta.mail.internet.MimeMultipart;
import jakarta.mail.internet.MimeUtility;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Map;
import java.util.Properties;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import org.jboss.logging.Logger;
import org.keycloak.common.enums.HostnameVerificationPolicy;
import org.keycloak.email.EmailAuthenticator;
import org.keycloak.email.EmailException;
import org.keycloak.email.EmailSenderProvider;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.UserModel;
import org.keycloak.services.ServicesLogger;
import org.keycloak.truststore.JSSETruststoreConfigurator;
import org.keycloak.utils.EmailValidationUtil;
import org.keycloak.utils.SMTPUtil;
import org.keycloak.utils.StringUtil;

public class DefaultEmailSenderProvider
implements EmailSenderProvider {
    private static final Logger logger = Logger.getLogger(DefaultEmailSenderProvider.class);
    private static final String SUPPORTED_SSL_PROTOCOLS = DefaultEmailSenderProvider.getSupportedSslProtocols();
    private final Map<EmailAuthenticator.AuthenticatorType, EmailAuthenticator> authenticators;
    private final KeycloakSession session;

    public DefaultEmailSenderProvider(KeycloakSession session, Map<EmailAuthenticator.AuthenticatorType, EmailAuthenticator> authenticators) {
        this.authenticators = authenticators;
        this.session = session;
    }

    public void send(Map<String, String> config, UserModel user, String subject, String textBody, String htmlBody) throws EmailException {
        String address = this.retrieveEmailAddress(user);
        if (address == null) {
            throw new EmailException("No email address configured for the user");
        }
        this.send(config, address, subject, textBody, htmlBody);
    }

    public void send(Map<String, String> config, String address, String subject, String textBody, String htmlBody) throws EmailException {
        boolean allowutf8 = DefaultEmailSenderProvider.isAllowUTF8(config);
        String convertedAddress = DefaultEmailSenderProvider.checkUserAddress(address, allowutf8);
        String from = DefaultEmailSenderProvider.checkFromAddress(config.get("from"), allowutf8);
        Session session = Session.getInstance((Properties)this.buildEmailProperties(config, from));
        Message message = this.buildMessage(session, convertedAddress, from, subject, config, this.buildMultipartBody(textBody, htmlBody));
        try (Transport transport = session.getTransport("smtp");){
            EmailAuthenticator selectedAuthenticator = this.selectAuthenticatorBasedOnConfig(config);
            selectedAuthenticator.connect(this.session, config, transport);
            transport.sendMessage(message, (Address[])new InternetAddress[]{new InternetAddress(convertedAddress)});
        }
        catch (Exception e) {
            ServicesLogger.LOGGER.failedToSendEmail(e);
            throw new EmailException("Error when attempting to send the email to the server. More information is available in the server log.", (Throwable)e);
        }
    }

    public void validate(Map<String, String> config) throws EmailException {
        DefaultEmailSenderProvider.checkFromAddress(config.get("from"), DefaultEmailSenderProvider.isAllowUTF8(config));
    }

    Properties buildEmailProperties(Map<String, String> config, String from) throws EmailException {
        boolean allowutf8;
        Properties props = new Properties();
        if (config.containsKey("host")) {
            props.setProperty("mail.smtp.host", config.get("host"));
        }
        if (config.containsKey("port") && config.get("port") != null) {
            props.setProperty("mail.smtp.port", config.get("port"));
        }
        if (this.isAuthConfigured(config)) {
            props.setProperty("mail.smtp.auth", "true");
        }
        if (this.isAuthTypeTokenConfigured(config)) {
            props.put("mail.smtp.auth.mechanisms", "XOAUTH2");
        }
        if (DefaultEmailSenderProvider.isDebugEnabled(config)) {
            props.put("mail.debug", "true");
        }
        if (DefaultEmailSenderProvider.isSslConfigured(config)) {
            props.setProperty("mail.smtp.ssl.enable", "true");
        }
        if (DefaultEmailSenderProvider.isStarttlsConfigured(config)) {
            props.setProperty("mail.smtp.starttls.enable", "true");
        }
        if (DefaultEmailSenderProvider.isSslConfigured(config) || DefaultEmailSenderProvider.isStarttlsConfigured(config) || this.isAuthConfigured(config)) {
            props.put("mail.smtp.ssl.protocols", SUPPORTED_SSL_PROTOCOLS);
            this.setupTruststore(props);
        }
        props.setProperty("mail.smtp.timeout", config.getOrDefault("timeout", "10000"));
        props.setProperty("mail.smtp.connectiontimeout", config.getOrDefault("connectionTimeout", "10000"));
        props.setProperty("mail.smtp.writetimeout", config.getOrDefault("writeTimeout", "10000"));
        String envelopeFrom = config.get("envelopeFrom");
        if (StringUtil.isNotBlank((String)envelopeFrom)) {
            props.setProperty("mail.smtp.from", envelopeFrom);
        }
        if (allowutf8 = DefaultEmailSenderProvider.isAllowUTF8(config)) {
            props.setProperty("mail.mime.allowutf8", "true");
        }
        props.setProperty("mail.from", from);
        return props;
    }

    private Message buildMessage(Session session, String address, String from, String subject, Map<String, String> config, Multipart multipart) throws EmailException {
        String fromDisplayName = config.get("fromDisplayName");
        String replyTo = config.get("replyTo");
        String replyToDisplayName = config.get("replyToDisplayName");
        try {
            MimeMessage msg = new MimeMessage(session);
            msg.setFrom((Address)this.toInternetAddress(from, fromDisplayName));
            msg.setReplyTo(new Address[]{this.toInternetAddress(from, fromDisplayName)});
            if (StringUtil.isNotBlank((String)replyTo)) {
                msg.setReplyTo(new Address[]{this.toInternetAddress(replyTo, replyToDisplayName)});
            }
            msg.setHeader("To", address);
            msg.setSubject(MimeUtility.encodeText((String)subject, (String)StandardCharsets.UTF_8.name(), null));
            msg.setContent(multipart);
            msg.saveChanges();
            msg.setSentDate(new Date());
            return msg;
        }
        catch (UnsupportedEncodingException e) {
            throw new EmailException("Failed to encode email address", (Throwable)e);
        }
        catch (AddressException e) {
            throw new EmailException("Invalid email address format", (Throwable)e);
        }
        catch (MessagingException e) {
            throw new EmailException("MessagingException occurred", (Throwable)e);
        }
    }

    private Multipart buildMultipartBody(String textBody, String htmlBody) throws EmailException {
        MimeMultipart multipart = new MimeMultipart("alternative");
        try {
            if (textBody != null) {
                MimeBodyPart textPart = new MimeBodyPart();
                textPart.setText(textBody, "UTF-8");
                multipart.addBodyPart((BodyPart)textPart);
            }
            if (htmlBody != null) {
                MimeBodyPart htmlPart = new MimeBodyPart();
                htmlPart.setContent((Object)htmlBody, "text/html; charset=UTF-8");
                multipart.addBodyPart((BodyPart)htmlPart);
            }
        }
        catch (MessagingException e) {
            throw new EmailException("Error encoding email body parts", (Throwable)e);
        }
        return multipart;
    }

    private EmailAuthenticator selectAuthenticatorBasedOnConfig(Map<String, String> config) {
        if (this.isAuthConfigured(config)) {
            String authType = config.getOrDefault("authType", "basic");
            return this.authenticators.get((Object)EmailAuthenticator.AuthenticatorType.valueOf(authType.toUpperCase()));
        }
        return this.authenticators.get((Object)EmailAuthenticator.AuthenticatorType.NONE);
    }

    private static boolean isStarttlsConfigured(Map<String, String> config) {
        return "true".equals(config.get("starttls"));
    }

    private static boolean isSslConfigured(Map<String, String> config) {
        return "true".equals(config.get("ssl"));
    }

    private static boolean isAllowUTF8(Map<String, String> config) {
        return "true".equals(config.get("allowutf8"));
    }

    private static boolean isDebugEnabled(Map<String, String> config) {
        return "true".equals(config.get("debug"));
    }

    private boolean isAuthConfigured(Map<String, String> config) {
        return "true".equals(config.get("auth"));
    }

    private boolean isAuthTypeTokenConfigured(Map<String, String> config) {
        return "token".equals(config.get("authType"));
    }

    private static String checkUserAddress(String address, boolean allowutf8) throws EmailException {
        String convertedAddress = DefaultEmailSenderProvider.convertEmail(address, allowutf8);
        if (convertedAddress == null) {
            throw new EmailException(String.format("Invalid address '%s'. If the address contains UTF-8 characters in the local part please ensure the SMTP server supports the SMTPUTF8 extension and enable 'Allow UTF-8' in the email realm configuration.", address));
        }
        return convertedAddress;
    }

    private static String checkFromAddress(String from, boolean allowutf8) throws EmailException {
        String covertedFrom = DefaultEmailSenderProvider.convertEmail(from, allowutf8);
        if (from == null) {
            throw new EmailException(String.format("Invalid sender address '%s'. If the address contains UTF-8 characters in the local part please ensure the SMTP server supports the SMTPUTF8 extension and enable 'Allow UTF-8' in the email realm configuration.", from));
        }
        return covertedFrom;
    }

    private static String convertEmail(String email, boolean allowutf8) throws EmailException {
        if (!EmailValidationUtil.isValidEmail((String)email)) {
            return null;
        }
        if (allowutf8) {
            return email;
        }
        String convertedEmail = SMTPUtil.convertIDNEmailAddress(email);
        if (convertedEmail == null || !convertedEmail.chars().allMatch(c -> c < 128)) {
            return null;
        }
        return convertedEmail;
    }

    protected InternetAddress toInternetAddress(String email, String displayName) throws UnsupportedEncodingException, AddressException, EmailException {
        if (email == null || "".equals(email.trim())) {
            throw new EmailException("Please provide a valid address", null);
        }
        if (displayName == null || "".equals(displayName.trim())) {
            return new InternetAddress(email);
        }
        return new InternetAddress(email, displayName, "utf-8");
    }

    protected String retrieveEmailAddress(UserModel user) {
        return user.getEmail();
    }

    private void setupTruststore(Properties props) {
        JSSETruststoreConfigurator configurator = new JSSETruststoreConfigurator(this.session);
        SSLSocketFactory factory = configurator.getSSLSocketFactory();
        if (factory != null) {
            props.put("mail.smtp.ssl.socketFactory", factory);
            if (configurator.getProvider().getPolicy() == HostnameVerificationPolicy.ANY) {
                props.setProperty("mail.smtp.ssl.trust", "*");
                props.put("mail.smtp.ssl.checkserveridentity", Boolean.FALSE.toString());
            } else {
                props.put("mail.smtp.ssl.checkserveridentity", Boolean.TRUE.toString());
            }
        }
    }

    public void close() {
    }

    private static String getSupportedSslProtocols() {
        try {
            CharSequence[] protocols = SSLContext.getDefault().getSupportedSSLParameters().getProtocols();
            if (protocols != null) {
                return String.join((CharSequence)" ", protocols);
            }
        }
        catch (Exception e) {
            logger.warn((Object)"Failed to get list of supported SSL protocols", (Throwable)e);
        }
        return null;
    }
}

