/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.protocol.oidc;

import io.opentelemetry.api.trace.Span;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.UriInfo;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jboss.logging.Logger;
import org.keycloak.OAuthErrorException;
import org.keycloak.Token;
import org.keycloak.TokenCategory;
import org.keycloak.TokenVerifier;
import org.keycloak.authentication.authenticators.util.AcrStore;
import org.keycloak.broker.oidc.OIDCIdentityProvider;
import org.keycloak.broker.provider.IdentityBrokerException;
import org.keycloak.broker.provider.IdentityProvider;
import org.keycloak.common.ClientConnection;
import org.keycloak.common.Profile;
import org.keycloak.common.VerificationException;
import org.keycloak.common.util.Time;
import org.keycloak.crypto.HashProvider;
import org.keycloak.crypto.SignatureProvider;
import org.keycloak.events.EventBuilder;
import org.keycloak.http.HttpRequest;
import org.keycloak.jose.jws.JWSInput;
import org.keycloak.jose.jws.JWSInputException;
import org.keycloak.jose.jws.crypto.HashUtils;
import org.keycloak.migration.migrators.MigrationUtils;
import org.keycloak.models.AuthenticatedClientSessionModel;
import org.keycloak.models.ClientModel;
import org.keycloak.models.ClientScopeModel;
import org.keycloak.models.ClientSessionContext;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.ProtocolMapperModel;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RoleModel;
import org.keycloak.models.SingleUseObjectProvider;
import org.keycloak.models.UserConsentModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.UserSessionProvider;
import org.keycloak.models.light.LightweightUserAdapter;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.models.utils.RoleUtils;
import org.keycloak.models.utils.SessionExpirationUtils;
import org.keycloak.organization.protocol.mappers.oidc.OrganizationScope;
import org.keycloak.protocol.ProtocolMapper;
import org.keycloak.protocol.ProtocolMapperUtils;
import org.keycloak.protocol.oidc.LogoutTokenValidationCode;
import org.keycloak.protocol.oidc.LogoutTokenValidationContext;
import org.keycloak.protocol.oidc.OIDCAdvancedConfigWrapper;
import org.keycloak.protocol.oidc.mappers.OIDCAccessTokenMapper;
import org.keycloak.protocol.oidc.mappers.OIDCAccessTokenResponseMapper;
import org.keycloak.protocol.oidc.mappers.OIDCIDTokenMapper;
import org.keycloak.protocol.oidc.mappers.TokenIntrospectionTokenMapper;
import org.keycloak.protocol.oidc.mappers.UserInfoTokenMapper;
import org.keycloak.protocol.oidc.utils.OIDCResponseType;
import org.keycloak.rar.AuthorizationDetails;
import org.keycloak.rar.AuthorizationRequestContext;
import org.keycloak.representations.AccessToken;
import org.keycloak.representations.AccessTokenResponse;
import org.keycloak.representations.AuthorizationDetailsJSONRepresentation;
import org.keycloak.representations.IDToken;
import org.keycloak.representations.JsonWebToken;
import org.keycloak.representations.LogoutToken;
import org.keycloak.representations.RefreshToken;
import org.keycloak.representations.dpop.DPoP;
import org.keycloak.services.ErrorResponseException;
import org.keycloak.services.Urls;
import org.keycloak.services.managers.AuthenticationManager;
import org.keycloak.services.managers.AuthenticationSessionManager;
import org.keycloak.services.managers.UserConsentManager;
import org.keycloak.services.managers.UserSessionManager;
import org.keycloak.services.resources.IdentityBrokerService;
import org.keycloak.services.util.AuthorizationContextUtil;
import org.keycloak.services.util.DPoPUtil;
import org.keycloak.services.util.DefaultClientSessionContext;
import org.keycloak.services.util.MtlsHoKTokenUtil;
import org.keycloak.services.util.UserSessionUtil;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.tracing.TracingAttributes;
import org.keycloak.tracing.TracingProvider;
import org.keycloak.util.TokenUtil;

public class TokenManager {
    private static final Logger logger = Logger.getLogger(TokenManager.class);

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public TokenValidation validateToken(KeycloakSession session, UriInfo uriInfo, ClientConnection connection, RealmModel realm, RefreshToken oldToken, HttpHeaders headers, String oldTokenScope) throws OAuthErrorException {
        DefaultClientSessionContext clientSessionCtx;
        UserModel user;
        UserSessionModel userSession = null;
        boolean offline = "Offline".equals(oldToken.getType());
        if (offline) {
            UserSessionManager sessionManager = new UserSessionManager(session);
            userSession = sessionManager.findOfflineUserSession(realm, oldToken.getSessionState());
            if (userSession == null) throw new OAuthErrorException("invalid_grant", "Offline user session not found", "Offline user session not found");
            if (!AuthenticationManager.isSessionValid(realm, userSession)) {
                sessionManager.revokeOfflineUserSession(userSession);
                throw new OAuthErrorException("invalid_grant", "Offline session not active", "Offline session not active");
            }
        } else {
            userSession = session.sessions().getUserSession(realm, oldToken.getSessionState());
            if (!AuthenticationManager.isSessionValid(realm, userSession)) {
                AuthenticationManager.backchannelLogout(session, realm, userSession, uriInfo, connection, headers, true);
                throw new OAuthErrorException("invalid_grant", "Session not active", "Session not active");
            }
        }
        if ((user = userSession.getUser()) == null) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token", "Unknown user");
        }
        if (!user.isEnabled()) {
            throw new OAuthErrorException("invalid_grant", "User disabled", "User disabled");
        }
        if (oldToken.isIssuedBeforeSessionStart((long)userSession.getStarted())) {
            logger.debug((Object)"Refresh token issued before the user session started");
            throw new OAuthErrorException("invalid_grant", "Refresh token issued before the user session started");
        }
        ClientModel client = session.getContext().getClient();
        AuthenticatedClientSessionModel clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        if (clientSession == null) {
            userSession = session.sessions().getUserSessionIfClientExists(realm, userSession.getId(), offline, client.getId());
            if (userSession == null) throw new OAuthErrorException("invalid_grant", "Session doesn't have required client", "Session doesn't have required client");
            clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        }
        if (!AuthenticationManager.isClientSessionValid(realm, client, userSession, clientSession)) {
            logger.debug((Object)"Client session not active");
            userSession.removeAuthenticatedClientSessions(Collections.singletonList(client.getId()));
            throw new OAuthErrorException("invalid_grant", "Client session not active");
        }
        if (oldToken.isIssuedBeforeSessionStart((long)clientSession.getStarted())) {
            logger.debug((Object)"refresh token issued before the client session started");
            throw new OAuthErrorException("invalid_grant", "refresh token issued before the client session started");
        }
        if (!client.getClientId().equals(oldToken.getIssuedFor())) {
            throw new OAuthErrorException("invalid_grant", "Unmatching clients", "Unmatching clients");
        }
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)oldToken).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(client), NotBeforeCheck.forModel(session, realm, user)}).verify();
        }
        catch (VerificationException e) {
            throw new OAuthErrorException("invalid_grant", "Stale token");
        }
        if (oldTokenScope == null && userSession.isOffline()) {
            logger.debugf("Migrating offline token of user '%s' for client '%s' of realm '%s'", (Object)user.getUsername(), (Object)client.getClientId(), (Object)realm.getName());
            MigrationUtils.migrateOldOfflineToken((KeycloakSession)session, (RealmModel)realm, (ClientModel)client, (UserModel)user);
            oldTokenScope = "offline_access";
        }
        if (!TokenManager.verifyConsentStillAvailable(session, user, client, (clientSessionCtx = DefaultClientSessionContext.fromClientSessionAndScopeParameter(clientSession, oldTokenScope, session)).getClientScopesStream())) {
            throw new OAuthErrorException("invalid_scope", "Client no longer has requested consent from user");
        }
        if (oldToken.getNonce() != null) {
            clientSessionCtx.setAttribute("nonce", oldToken.getNonce());
        }
        AccessToken newToken = this.createClientAccessToken(session, realm, client, user, userSession, clientSessionCtx);
        return new TokenValidation(user, userSession, clientSessionCtx, newToken);
    }

    public AccessToken checkTokenValidForIntrospection(KeycloakSession session, RealmModel realm, AccessToken token, EventBuilder eventBuilder) {
        return this.getValidUserSessionIfTokenIsValid(session, realm, token, eventBuilder) != null ? token : null;
    }

    public UserSessionModel getValidUserSessionIfTokenIsValid(KeycloakSession session, RealmModel realm, AccessToken token, EventBuilder eventBuilder) {
        UserSessionModel userSession;
        if (token == null) {
            return null;
        }
        ClientModel client = realm.getClientByClientId(token.getIssuedFor());
        if (client == null) {
            logger.debugf("Introspection access token : client with clientId %s does not exist", (Object)token.getIssuedFor());
            eventBuilder.detail("reason", String.format("Could not find client for %s", token.getIssuedFor()));
            return null;
        }
        if (!client.isEnabled()) {
            logger.debugf("Introspection access token : client with clientId %s is disabled", (Object)token.getIssuedFor());
            eventBuilder.detail("reason", String.format("Client with clientId %s is disabled", token.getIssuedFor()));
            return null;
        }
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)token).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(client), TokenVerifier.IS_ACTIVE, new TokenRevocationCheck(session)}).verify();
        }
        catch (VerificationException e) {
            logger.debugf("Introspection access token for %s client: JWT check failed: %s", (Object)token.getIssuedFor(), (Object)e.getMessage());
            eventBuilder.detail("reason", "Introspection access token for " + token.getIssuedFor() + " client: JWT check failed");
            return null;
        }
        try {
            userSession = UserSessionUtil.findValidSession(session, realm, token, eventBuilder, client);
        }
        catch (Exception e) {
            logger.debugf("Introspection access token for " + token.getIssuedFor() + " client:" + e.getMessage(), new Object[0]);
            eventBuilder.detail("reason", "Introspection access token for " + token.getIssuedFor() + " client:" + e.getMessage());
            return null;
        }
        if (!TokenManager.isUserValid(session, realm, token, userSession.getUser())) {
            logger.debugf("Could not find valid user from user", new Object[0]);
            eventBuilder.detail("reason", "Could not find valid user from user");
            return null;
        }
        String tokenType = token.getType();
        if (realm.isRevokeRefreshToken() && (tokenType.equals("Refresh") || tokenType.equals("Offline")) && !this.validateTokenReuseForIntrospection(session, realm, token)) {
            logger.debugf("Introspection access token for %s client: failed to validate Token reuse for introspection", (Object)token.getIssuedFor());
            eventBuilder.detail("reason", "Realm revoke refresh token, token type is " + tokenType + " and token is not eligible for introspection");
            return null;
        }
        return userSession;
    }

    private boolean validateTokenReuseForIntrospection(KeycloakSession session, RealmModel realm, AccessToken token) {
        UserSessionModel userSession = null;
        if (token.getType().equals("Refresh")) {
            userSession = session.sessions().getUserSession(realm, token.getSessionId());
        } else {
            UserSessionManager sessionManager = new UserSessionManager(session);
            userSession = sessionManager.findOfflineUserSession(realm, token.getSessionId());
        }
        ClientModel client = realm.getClientByClientId(token.getIssuedFor());
        AuthenticatedClientSessionModel clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        try {
            this.validateTokenReuse(session, realm, token, clientSession, false);
            return true;
        }
        catch (OAuthErrorException e) {
            logger.debug((Object)"validateTokenReuseForIntrospection is false", (Throwable)e);
            return false;
        }
    }

    public static boolean isUserValid(KeycloakSession session, RealmModel realm, AccessToken token, UserModel user) {
        if (user == null) {
            logger.debugf("User does not exists", new Object[0]);
            return false;
        }
        if (!user.isEnabled()) {
            logger.debugf("User '%s' is disabled", (Object)user.getUsername());
            return false;
        }
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)token).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(session, realm, user)}).verify();
        }
        catch (VerificationException e) {
            logger.debugf("JWT check failed: %s", (Object)e.getMessage());
            return false;
        }
        return true;
    }

    public static UserModel lookupUserFromStatelessToken(KeycloakSession session, RealmModel realm, AccessToken token) {
        UserModel user;
        UserModel userModel = user = token.getSubject() == null ? null : session.users().getUserById(realm, token.getSubject());
        if (user != null) {
            return user;
        }
        if (token.getPreferredUsername() != null) {
            return session.users().getUserByUsername(realm, token.getPreferredUsername());
        }
        return null;
    }

    public AccessTokenResponseBuilder refreshAccessToken(KeycloakSession session, UriInfo uriInfo, ClientConnection connection, RealmModel realm, ClientModel authorizedClient, String encodedRefreshToken, EventBuilder event, HttpHeaders headers, HttpRequest request, String scopeParameter) throws OAuthErrorException {
        String scopeParam;
        RefreshToken refreshToken = this.verifyRefreshToken(session, realm, authorizedClient, request, encodedRefreshToken, true);
        event.session(refreshToken.getSessionState()).detail("refresh_token_id", refreshToken.getId()).detail("refresh_token_type", refreshToken.getType());
        if (refreshToken.getSubject() != null) {
            event.detail("refresh_token_sub", refreshToken.getSubject());
        }
        String oldTokenScope = refreshToken.getScope();
        if (scopeParameter != null && !scopeParameter.isEmpty()) {
            Set<String> scopeParamScopes = Arrays.stream(scopeParameter.split(" ")).collect(Collectors.toSet());
            oldTokenScope = Arrays.stream(oldTokenScope.split(" ")).map(this.transformScopes(scopeParamScopes)).filter(Objects::nonNull).collect(Collectors.joining(" "));
        }
        TokenValidation validation = this.validateToken(session, uriInfo, connection, realm, refreshToken, headers, oldTokenScope);
        AuthenticatedClientSessionModel clientSession = validation.clientSessionCtx.getClientSession();
        OIDCAdvancedConfigWrapper clientConfig = OIDCAdvancedConfigWrapper.fromClientModel(authorizedClient);
        if (!clientSession.getClient().getId().equals(authorizedClient.getId())) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token. Token client and authorized client don't match");
        }
        this.validateTokenReuseForRefresh(session, realm, refreshToken, validation);
        event.user(validation.userSession.getUser());
        if (refreshToken.getAuthorization() != null) {
            validation.newToken.setAuthorization(refreshToken.getAuthorization());
        }
        AccessTokenResponseBuilder responseBuilder = this.responseBuilder(realm, authorizedClient, event, session, validation.userSession, validation.clientSessionCtx).offlineToken("Offline".equals(refreshToken.getType())).accessToken(validation.newToken);
        if (clientConfig.isUseRefreshToken()) {
            responseBuilder.generateRefreshToken(refreshToken, clientSession);
        }
        if (validation.newToken.getAuthorization() != null && clientConfig.isUseRefreshToken()) {
            responseBuilder.getRefreshToken().setAuthorization(validation.newToken.getAuthorization());
        }
        if (TokenUtil.isOIDCRequest((String)(scopeParam = clientSession.getNote("scope")))) {
            responseBuilder.generateIDToken().generateAccessTokenHash();
        }
        this.storeRefreshTimingInformation(event, refreshToken, validation.newToken);
        return responseBuilder;
    }

    private Function<String, String> transformScopes(Set<String> requestedScopes) {
        return scope -> {
            if (requestedScopes.contains(scope)) {
                return scope;
            }
            if (Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.ORGANIZATION)) {
                OrganizationScope oldScope = OrganizationScope.valueOfScope(scope);
                return oldScope == null ? null : oldScope.resolveName(requestedScopes, (String)scope);
            }
            return null;
        };
    }

    private void storeRefreshTimingInformation(EventBuilder event, RefreshToken refreshToken, AccessToken newToken) {
        long expirationAccessToken = newToken.getExp() - newToken.getIat();
        long ageOfRefreshToken = newToken.getIat() - refreshToken.getIat();
        event.detail("access_token_expiration_time", Long.toString(expirationAccessToken));
        event.detail("age_of_refresh_token", Long.toString(ageOfRefreshToken));
    }

    private void validateTokenReuseForRefresh(KeycloakSession session, RealmModel realm, RefreshToken refreshToken, TokenValidation validation) throws OAuthErrorException {
        if (realm.isRevokeRefreshToken()) {
            AuthenticatedClientSessionModel clientSession = validation.clientSessionCtx.getClientSession();
            try {
                this.validateTokenReuse(session, realm, (AccessToken)refreshToken, clientSession, true);
                String key = this.getReuseIdKey((AccessToken)refreshToken);
                int currentCount = clientSession.getRefreshTokenUseCount(key);
                clientSession.setRefreshTokenUseCount(key, currentCount + 1);
            }
            catch (OAuthErrorException oee) {
                if (logger.isDebugEnabled()) {
                    logger.debugf("Failed validation of refresh token %s due it was used before. Realm: %s, client: %s, user: %s, user session: %s. Will detach client session from user session", new Object[]{refreshToken.getId(), realm.getName(), clientSession.getClient().getClientId(), clientSession.getUserSession().getUser().getUsername(), clientSession.getUserSession().getId()});
                }
                clientSession.detachFromUserSession();
                throw oee;
            }
        }
    }

    private void validateTokenReuse(KeycloakSession session, RealmModel realm, AccessToken refreshToken, AuthenticatedClientSessionModel clientSession, boolean refreshFlag) throws OAuthErrorException {
        int currentCount;
        int startupTime = ((UserSessionProvider)session.getProvider(UserSessionProvider.class)).getStartupTime(realm);
        String key = this.getReuseIdKey(refreshToken);
        String refreshTokenId = clientSession.getRefreshToken(key);
        int lastRefresh = clientSession.getRefreshTokenLastRefresh(key);
        if (refreshTokenId != null && !refreshToken.getId().equals(refreshTokenId) && refreshToken.getIat() < (long)lastRefresh && startupTime <= lastRefresh) {
            throw new OAuthErrorException("invalid_grant", "Stale token");
        }
        if (!refreshToken.getId().equals(refreshTokenId)) {
            if (refreshFlag) {
                clientSession.setRefreshToken(key, refreshToken.getId());
                clientSession.setRefreshTokenUseCount(key, 0);
            } else {
                return;
            }
        }
        if ((currentCount = clientSession.getRefreshTokenUseCount(key)) > realm.getRefreshTokenMaxReuse()) {
            throw new OAuthErrorException("invalid_grant", "Maximum allowed refresh token reuse exceeded", "Maximum allowed refresh token reuse exceeded");
        }
    }

    public RefreshToken verifyRefreshToken(KeycloakSession session, RealmModel realm, ClientModel client, HttpRequest request, String encodedRefreshToken, boolean checkExpiration) throws OAuthErrorException {
        try {
            RefreshToken refreshToken = this.toRefreshToken(session, encodedRefreshToken);
            if (!"Refresh".equals(refreshToken.getType()) && !"Offline".equals(refreshToken.getType())) {
                throw new OAuthErrorException("invalid_grant", "Invalid refresh token");
            }
            TokenVerifier tokenVerifier = TokenVerifier.createWithoutSignature((JsonWebToken)refreshToken).withChecks(new TokenVerifier.Predicate[]{new TokenVerifier.RealmUrlCheck(Urls.realmIssuer(session.getContext().getUri().getBaseUri(), realm.getName()))});
            if (checkExpiration) {
                tokenVerifier.withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(realm), TokenVerifier.IS_ACTIVE});
            }
            try {
                tokenVerifier.verify();
            }
            catch (VerificationException e) {
                throw new OAuthErrorException("invalid_grant", e.getMessage());
            }
            if (!client.getClientId().equals(refreshToken.getIssuedFor())) {
                throw new OAuthErrorException("invalid_grant", "Invalid refresh token. Token client and authorized client don't match");
            }
            if (OIDCAdvancedConfigWrapper.fromClientModel(client).isUseMtlsHokToken() && !MtlsHoKTokenUtil.verifyTokenBindingWithClientCertificate((AccessToken)refreshToken, request, session)) {
                throw new OAuthErrorException("unauthorized_client", "Client certificate missing, or its thumbprint and one in the refresh token did NOT match");
            }
            if (Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DPOP)) {
                DPoP dPoP = (DPoP)session.getAttribute("dpop");
                if (client.isPublicClient() && (OIDCAdvancedConfigWrapper.fromClientModel(client).isUseDPoP() || dPoP != null)) {
                    try {
                        DPoPUtil.validateBinding((AccessToken)refreshToken, dPoP);
                    }
                    catch (VerificationException ex) {
                        throw new OAuthErrorException("invalid_grant", ex.getMessage());
                    }
                }
            }
            return refreshToken;
        }
        catch (JWSInputException e) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token", (Throwable)e);
        }
    }

    public RefreshToken toRefreshToken(KeycloakSession session, String encodedRefreshToken) throws JWSInputException, OAuthErrorException {
        RefreshToken refreshToken = (RefreshToken)session.tokens().decode(encodedRefreshToken, RefreshToken.class);
        if (refreshToken == null) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token");
        }
        return refreshToken;
    }

    public IDToken verifyIDToken(KeycloakSession session, RealmModel realm, String encodedIDToken) throws OAuthErrorException {
        IDToken idToken = (IDToken)session.tokens().decode(encodedIDToken, IDToken.class);
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)idToken).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(realm), TokenVerifier.IS_ACTIVE}).verify();
        }
        catch (VerificationException e) {
            throw new OAuthErrorException("invalid_grant", e.getMessage());
        }
        return idToken;
    }

    public IDToken verifyIDTokenSignature(KeycloakSession session, String encodedIDToken) throws OAuthErrorException {
        IDToken idToken = (IDToken)session.tokens().decode(encodedIDToken, IDToken.class);
        if (idToken == null) {
            throw new OAuthErrorException("invalid_grant", "Invalid IDToken");
        }
        return idToken;
    }

    public AccessToken createClientAccessToken(KeycloakSession session, RealmModel realm, ClientModel client, UserModel user, UserSessionModel userSession, ClientSessionContext clientSessionCtx) {
        AccessToken token = this.initToken(session, realm, client, user, userSession, clientSessionCtx, (UriInfo)session.getContext().getUri());
        token = this.transformAccessToken(session, token, userSession, clientSessionCtx);
        return token;
    }

    public static ClientSessionContext attachAuthenticationSession(KeycloakSession session, UserSessionModel userSession, AuthenticationSessionModel authSession) {
        Set<ClientScopeModel> clientScopes;
        ClientModel client = authSession.getClient();
        AuthenticatedClientSessionModel clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        RealmModel realm = userSession.getRealm();
        if (clientSession != null && !AuthenticationManager.isClientSessionValid(realm, client, userSession, clientSession)) {
            clientSession.restartClientSession();
        } else if (clientSession == null) {
            clientSession = session.sessions().createClientSession(realm, client, userSession);
        }
        clientSession.setRedirectUri(authSession.getRedirectUri());
        clientSession.setProtocol(authSession.getProtocol());
        String scopeParam = authSession.getClientNote("scope");
        if (Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DYNAMIC_SCOPES)) {
            session.getContext().setClient(client);
            clientScopes = AuthorizationContextUtil.getClientScopesStreamFromAuthorizationRequestContextWithClient(session, scopeParam).collect(Collectors.toSet());
        } else {
            clientScopes = TokenManager.getRequestedClientScopes(session, scopeParam, client, userSession.getUser()).collect(Collectors.toSet());
        }
        Map transferredNotes = authSession.getClientNotes();
        for (Map.Entry entry : transferredNotes.entrySet()) {
            clientSession.setNote((String)entry.getKey(), (String)entry.getValue());
        }
        Map transferredUserSessionNotes = authSession.getUserSessionNotes();
        for (Map.Entry entry : transferredUserSessionNotes.entrySet()) {
            userSession.setNote((String)entry.getKey(), (String)entry.getValue());
        }
        clientSession.setNote("level-of-authentication", String.valueOf(new AcrStore(session, authSession).getLevelOfAuthenticationFromCurrentAuthentication()));
        clientSession.setTimestamp(userSession.getLastSessionRefresh());
        new AuthenticationSessionManager(session).updateAuthenticationSessionAfterSuccessfulAuthentication(realm, authSession);
        return DefaultClientSessionContext.fromClientSessionAndClientScopes(clientSession, clientScopes, session);
    }

    public static void dettachClientSession(AuthenticatedClientSessionModel clientSession) {
        UserSessionModel userSession = clientSession.getUserSession();
        if (userSession == null) {
            return;
        }
        clientSession.detachFromUserSession();
    }

    public static Set<RoleModel> getAccess(UserModel user, ClientModel client, Stream<ClientScopeModel> clientScopes) {
        Set roleMappings = RoleUtils.getDeepUserRoleMappings((UserModel)user);
        if (client.isFullScopeAllowed()) {
            if (logger.isTraceEnabled()) {
                logger.tracef("Using full scope for client %s", (Object)client.getClientId());
            }
            return roleMappings;
        }
        Stream scopeMappings = client.getRolesStream();
        Stream clientScopesMappings = !logger.isTraceEnabled() ? clientScopes.flatMap(clientScope -> clientScope.getScopeMappingsStream()) : clientScopes.flatMap(clientScope -> {
            logger.tracef("Adding client scope role mappings of client scope '%s' to client '%s'", (Object)clientScope.getName(), (Object)client.getClientId());
            return clientScope.getScopeMappingsStream();
        });
        scopeMappings = Stream.concat(scopeMappings, clientScopesMappings);
        scopeMappings = RoleUtils.expandCompositeRolesStream(scopeMappings);
        roleMappings.retainAll(scopeMappings.collect(Collectors.toSet()));
        return roleMappings;
    }

    public static Stream<ClientScopeModel> getRequestedClientScopes(KeycloakSession session, String scopeParam, ClientModel client, UserModel user) {
        if (client == null) {
            return Stream.of(new ClientScopeModel[0]);
        }
        Stream<ClientModel> clientScopes = Stream.concat(client.getClientScopes(true).values().stream(), Stream.of(client)).distinct();
        if (scopeParam == null) {
            return clientScopes;
        }
        Map allOptionalScopes = client.getClientScopes(false);
        return Stream.concat(TokenManager.parseScopeParameter(scopeParam).map(name -> {
            ClientScopeModel scope = (ClientScopeModel)allOptionalScopes.get(name);
            if (scope != null) {
                return scope;
            }
            return TokenManager.tryResolveDynamicClientScope(session, scopeParam, client, user, name);
        }).filter(Objects::nonNull), clientScopes).distinct();
    }

    private static ClientScopeModel tryResolveDynamicClientScope(KeycloakSession session, String scopeParam, ClientModel client, UserModel user, String name) {
        if (Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.ORGANIZATION)) {
            OrganizationScope orgScope = OrganizationScope.valueOfScope(scopeParam);
            if (orgScope == null) {
                return null;
            }
            return orgScope.toClientScope(name, user, session);
        }
        return null;
    }

    public static boolean isValidScope(KeycloakSession session, String scopes, AuthorizationRequestContext authorizationRequestContext, ClientModel client, UserModel user) {
        Set clientScopes;
        if (scopes == null) {
            return true;
        }
        Collection rawScopes = TokenManager.parseScopeParameter(scopes).collect(Collectors.toSet());
        if (Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.ORGANIZATION) && rawScopes.stream().filter(scope -> scope.startsWith("organization")).count() > 1L) {
            return false;
        }
        if (TokenUtil.isOIDCRequest((String)scopes)) {
            rawScopes.remove("openid");
        }
        if (rawScopes.isEmpty()) {
            return true;
        }
        if (authorizationRequestContext == null) {
            clientScopes = TokenManager.getRequestedClientScopes(session, scopes, client, user).filter(((Predicate<ClientScopeModel>)ClientModel.class::isInstance).negate()).map(ClientScopeModel::getName).collect(Collectors.toSet());
        } else {
            List details = Optional.ofNullable(authorizationRequestContext.getAuthorizationDetailEntries()).orElse(List.of());
            clientScopes = details.stream().map(AuthorizationDetails::getAuthorizationDetails).map(AuthorizationDetailsJSONRepresentation::getScopeNameFromCustomData).collect(Collectors.toSet());
        }
        if (logger.isTraceEnabled()) {
            logger.tracef("Scopes to validate requested scopes against: %1s", (Object)String.join((CharSequence)" ", clientScopes));
            logger.tracef("Requested scopes: %1s", (Object)String.join((CharSequence)" ", rawScopes));
        }
        if (clientScopes.isEmpty()) {
            return false;
        }
        for (String requestedScope : rawScopes) {
            if (clientScopes.contains(requestedScope) || client.getDynamicClientScope(requestedScope) != null) continue;
            return false;
        }
        return true;
    }

    public static boolean isValidScope(KeycloakSession session, String scopes, ClientModel client, UserModel user) {
        return TokenManager.isValidScope(session, scopes, null, client, user);
    }

    public static Stream<String> parseScopeParameter(String scopeParam) {
        return Arrays.stream(scopeParam.split(" ")).distinct();
    }

    public static boolean verifyConsentStillAvailable(KeycloakSession session, UserModel user, ClientModel client, Stream<ClientScopeModel> requestedClientScopes) {
        if (!client.isConsentRequired()) {
            return true;
        }
        UserConsentModel grantedConsent = UserConsentManager.getConsentByClient(session, client.getRealm(), user, client.getId());
        return requestedClientScopes.filter(ClientScopeModel::isDisplayOnConsentScreen).noneMatch(requestedScope -> {
            if (grantedConsent == null || !grantedConsent.getGrantedClientScopes().contains(requestedScope)) {
                logger.debugf("Client '%s' no longer has requested consent from user '%s' for client scope '%s'", (Object)client.getClientId(), (Object)user.getUsername(), (Object)requestedScope.getName());
                return true;
            }
            return false;
        });
    }

    public AccessToken transformAccessToken(final KeycloakSession session, AccessToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof OIDCAccessTokenMapper).collect(new TokenCollector<AccessToken>(token){

            @Override
            protected AccessToken applyMapper(AccessToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((OIDCAccessTokenMapper)mapper.getValue()).transformAccessToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public AccessTokenResponse transformAccessTokenResponse(final KeycloakSession session, AccessTokenResponse accessTokenResponse, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof OIDCAccessTokenResponseMapper).collect(new TokenCollector<AccessTokenResponse>(accessTokenResponse){

            @Override
            protected AccessTokenResponse applyMapper(AccessTokenResponse token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((OIDCAccessTokenResponseMapper)mapper.getValue()).transformAccessTokenResponse(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public AccessToken transformUserInfoAccessToken(final KeycloakSession session, AccessToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof UserInfoTokenMapper).collect(new TokenCollector<AccessToken>(token){

            @Override
            protected AccessToken applyMapper(AccessToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((UserInfoTokenMapper)mapper.getValue()).transformUserInfoToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public AccessToken transformIntrospectionAccessToken(final KeycloakSession session, AccessToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof TokenIntrospectionTokenMapper).collect(new TokenCollector<AccessToken>(token){

            @Override
            protected AccessToken applyMapper(AccessToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((TokenIntrospectionTokenMapper)mapper.getValue()).transformIntrospectionToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public Map<String, Object> generateUserInfoClaims(AccessToken userInfo, UserModel userModel) {
        HashMap<String, Object> claims = new HashMap<String, Object>();
        claims.put("sub", userInfo.getSubject() == null ? userModel.getId() : userInfo.getSubject());
        if (userInfo.getIssuer() != null) {
            claims.put("iss", userInfo.getIssuer());
        }
        if (userInfo.getAudience() != null) {
            claims.put("aud", userInfo.getAudience());
        }
        if (userInfo.getName() != null) {
            claims.put("name", userInfo.getName());
        }
        if (userInfo.getGivenName() != null) {
            claims.put("given_name", userInfo.getGivenName());
        }
        if (userInfo.getFamilyName() != null) {
            claims.put("family_name", userInfo.getFamilyName());
        }
        if (userInfo.getMiddleName() != null) {
            claims.put("middle_name", userInfo.getMiddleName());
        }
        if (userInfo.getNickName() != null) {
            claims.put("nickname", userInfo.getNickName());
        }
        if (userInfo.getPreferredUsername() != null) {
            claims.put("preferred_username", userInfo.getPreferredUsername());
        }
        if (userInfo.getProfile() != null) {
            claims.put("profile", userInfo.getProfile());
        }
        if (userInfo.getPicture() != null) {
            claims.put("picture", userInfo.getPicture());
        }
        if (userInfo.getWebsite() != null) {
            claims.put("website", userInfo.getWebsite());
        }
        if (userInfo.getEmail() != null) {
            claims.put("email", userInfo.getEmail());
        }
        if (userInfo.getEmailVerified() != null) {
            claims.put("email_verified", userInfo.getEmailVerified());
        }
        if (userInfo.getGender() != null) {
            claims.put("gender", userInfo.getGender());
        }
        if (userInfo.getBirthdate() != null) {
            claims.put("birthdate", userInfo.getBirthdate());
        }
        if (userInfo.getZoneinfo() != null) {
            claims.put("zoneinfo", userInfo.getZoneinfo());
        }
        if (userInfo.getLocale() != null) {
            claims.put("locale", userInfo.getLocale());
        }
        if (userInfo.getPhoneNumber() != null) {
            claims.put("phone_number", userInfo.getPhoneNumber());
        }
        if (userInfo.getPhoneNumberVerified() != null) {
            claims.put("phone_number_verified", userInfo.getPhoneNumberVerified());
        }
        if (userInfo.getAddress() != null) {
            claims.put("address", userInfo.getAddress());
        }
        if (userInfo.getUpdatedAt() != null) {
            claims.put("updated_at", userInfo.getUpdatedAt());
        }
        if (userInfo.getClaimsLocales() != null) {
            claims.put("claims_locales", userInfo.getClaimsLocales());
        }
        claims.putAll(userInfo.getOtherClaims());
        if (userInfo.getRealmAccess() != null) {
            HashMap<String, Set> realmAccess = new HashMap<String, Set>();
            realmAccess.put("roles", userInfo.getRealmAccess().getRoles());
            claims.put("realm_access", realmAccess);
        }
        if (userInfo.getResourceAccess() != null && !userInfo.getResourceAccess().isEmpty()) {
            HashMap resourceAccessMap = new HashMap();
            for (Map.Entry resourceAccessMapEntry : userInfo.getResourceAccess().entrySet()) {
                HashMap<String, Set> resourceAccess = new HashMap<String, Set>();
                resourceAccess.put("roles", ((AccessToken.Access)resourceAccessMapEntry.getValue()).getRoles());
                resourceAccessMap.put((String)resourceAccessMapEntry.getKey(), resourceAccess);
            }
            claims.put("resource_access", resourceAccessMap);
        }
        return claims;
    }

    public IDToken transformIDToken(final KeycloakSession session, IDToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof OIDCIDTokenMapper).collect(new TokenCollector<IDToken>(token){

            @Override
            protected IDToken applyMapper(IDToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((OIDCIDTokenMapper)mapper.getValue()).transformIDToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    protected AccessToken initToken(KeycloakSession session, RealmModel realm, ClientModel client, UserModel user, UserSessionModel userSession, ClientSessionContext clientSessionCtx, UriInfo uriInfo) {
        AccessToken token = new AccessToken();
        token.id(KeycloakModelUtils.generateId());
        token.type(this.formatTokenType(client, token));
        if (UserSessionModel.SessionPersistenceState.TRANSIENT.equals((Object)userSession.getPersistenceState())) {
            token.subject(user.getId());
        }
        token.issuedNow();
        token.issuedFor(client.getClientId());
        AuthenticatedClientSessionModel clientSession = clientSessionCtx.getClientSession();
        token.issuer(clientSession.getNote("iss"));
        token.setScope(clientSessionCtx.getScopeString());
        if (!Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.STEP_UP_AUTHENTICATION)) {
            String acr = AuthenticationManager.isSSOAuthentication(clientSession) ? "0" : "1";
            token.setAcr(acr);
        }
        token.setSessionId(userSession.getId());
        ClientScopeModel offlineAccessScope = KeycloakModelUtils.getClientScopeByName((RealmModel)realm, (String)"offline_access");
        boolean offlineTokenRequested = offlineAccessScope == null ? false : clientSessionCtx.getClientScopeIds().contains(offlineAccessScope.getId());
        token.exp(this.getTokenExpiration(realm, client, userSession, clientSession, offlineTokenRequested));
        TracingProvider tracing = (TracingProvider)session.getProvider(TracingProvider.class);
        Span span = tracing.getCurrentSpan();
        if (span.isRecording()) {
            span.setAttribute(TracingAttributes.TOKEN_ISSUER, (Object)token.getIssuer());
            span.setAttribute(TracingAttributes.TOKEN_SID, (Object)token.getSessionId());
            span.setAttribute(TracingAttributes.TOKEN_ID, (Object)token.getId());
        }
        return token;
    }

    private Long getTokenExpiration(RealmModel realm, ClientModel client, UserSessionModel userSession, AuthenticatedClientSessionModel clientSession, boolean offlineTokenRequested) {
        String clientLifespan;
        boolean implicitFlow = false;
        String responseType = clientSession.getNote("response_type");
        if (responseType != null) {
            implicitFlow = OIDCResponseType.parse(responseType).isImplicitFlow();
        }
        int tokenLifespan = implicitFlow ? realm.getAccessTokenLifespanForImplicitFlow() : ((clientLifespan = client.getAttribute("access.token.lifespan")) != null && !clientLifespan.trim().isEmpty() ? Integer.parseInt(clientLifespan) : realm.getAccessTokenLifespan());
        long expiration = tokenLifespan == -1 ? TimeUnit.SECONDS.toMillis(userSession.getStarted() + (userSession.isRememberMe() && realm.getSsoSessionMaxLifespanRememberMe() > 0 ? realm.getSsoSessionMaxLifespanRememberMe() : realm.getSsoSessionMaxLifespan())) : Time.currentTimeMillis() + TimeUnit.SECONDS.toMillis(tokenLifespan);
        long sessionExpires = SessionExpirationUtils.calculateClientSessionMaxLifespanTimestamp((userSession.isOffline() || offlineTokenRequested ? 1 : 0) != 0, (boolean)userSession.isRememberMe(), (long)TimeUnit.SECONDS.toMillis(clientSession.getStarted()), (long)TimeUnit.SECONDS.toMillis(userSession.getStarted()), (RealmModel)realm, (ClientModel)client);
        expiration = sessionExpires > 0L ? Math.min(expiration, sessionExpires) : expiration;
        return TimeUnit.MILLISECONDS.toSeconds(expiration);
    }

    public AccessTokenResponseBuilder responseBuilder(RealmModel realm, ClientModel client, EventBuilder event, KeycloakSession session, UserSessionModel userSession, ClientSessionContext clientSessionCtx) {
        return new AccessTokenResponseBuilder(realm, client, event, session, userSession, clientSessionCtx);
    }

    private String formatTokenType(ClientModel client, AccessToken accessToken) {
        String tokenType = Optional.ofNullable(accessToken).map(JsonWebToken::getType).orElse("Bearer");
        if (OIDCAdvancedConfigWrapper.fromClientModel(client).isUseLowerCaseInTokenResponse()) {
            return tokenType.toLowerCase();
        }
        return tokenType;
    }

    public LogoutTokenValidationContext verifyLogoutToken(KeycloakSession session, String encodedLogoutToken) {
        Optional<LogoutToken> logoutTokenOptional = this.toLogoutToken(encodedLogoutToken);
        if (logoutTokenOptional.isEmpty()) {
            return LogoutTokenValidationCode.DECODE_TOKEN_FAILED.toCtx();
        }
        LogoutToken logoutToken = logoutTokenOptional.get();
        List<OIDCIdentityProvider> identityProviders = this.getOIDCIdentityProviders(logoutToken, session).toList();
        if (identityProviders.isEmpty()) {
            return LogoutTokenValidationCode.COULD_NOT_FIND_IDP.toCtx();
        }
        List<OIDCIdentityProvider> validOidcIdentityProviders = this.validateLogoutTokenAgainstIdpProvider(identityProviders.stream(), encodedLogoutToken).toList();
        if (validOidcIdentityProviders.isEmpty()) {
            return LogoutTokenValidationCode.TOKEN_VERIFICATION_WITH_IDP_FAILED.toCtx();
        }
        if (logoutToken.getSubject() == null && logoutToken.getSid() == null) {
            return LogoutTokenValidationCode.MISSING_SID_OR_SUBJECT.toCtx();
        }
        if (!this.checkLogoutTokenForEvents(logoutToken)) {
            return LogoutTokenValidationCode.BACKCHANNEL_LOGOUT_EVENT_MISSING.toCtx();
        }
        if (logoutToken.getOtherClaims().get("nonce") != null) {
            return LogoutTokenValidationCode.NONCE_CLAIM_IN_TOKEN.toCtx();
        }
        if (logoutToken.getId() == null) {
            return LogoutTokenValidationCode.LOGOUT_TOKEN_ID_MISSING.toCtx();
        }
        if (logoutToken.getIat() == null) {
            return LogoutTokenValidationCode.MISSING_IAT_CLAIM.toCtx();
        }
        return new LogoutTokenValidationContext(LogoutTokenValidationCode.VALIDATION_SUCCESS, logoutToken, validOidcIdentityProviders);
    }

    public Optional<LogoutToken> toLogoutToken(String encodedLogoutToken) {
        try {
            JWSInput jws = new JWSInput(encodedLogoutToken);
            return Optional.of((LogoutToken)jws.readJsonContent(LogoutToken.class));
        }
        catch (JWSInputException e) {
            return Optional.empty();
        }
    }

    public Stream<OIDCIdentityProvider> validateLogoutTokenAgainstIdpProvider(Stream<OIDCIdentityProvider> oidcIdps, String encodedLogoutToken) {
        return oidcIdps.filter(oidcIdp -> {
            try {
                oidcIdp.validateToken(encodedLogoutToken);
                return true;
            }
            catch (IdentityBrokerException e) {
                logger.debugf("LogoutToken verification with identity provider failed", (Object)e.getMessage());
                return false;
            }
        });
    }

    private Stream<OIDCIdentityProvider> getOIDCIdentityProviders(LogoutToken logoutToken, KeycloakSession session) {
        try {
            return session.identityProviders().getAllStream(Map.of("issuer", logoutToken.getIssuer()), Integer.valueOf(-1), Integer.valueOf(-1)).map(model -> {
                IdentityProvider<?> idp = IdentityBrokerService.getIdentityProvider(session, model.getAlias());
                if (idp instanceof OIDCIdentityProvider) {
                    OIDCIdentityProvider oidcIdp = (OIDCIdentityProvider)idp;
                    return oidcIdp;
                }
                return null;
            }).filter(Objects::nonNull);
        }
        catch (IdentityBrokerException e) {
            logger.warnf("LogoutToken verification with identity provider failed", (Object)e.getMessage());
            return Stream.empty();
        }
    }

    private boolean checkLogoutTokenForEvents(LogoutToken logoutToken) {
        for (String eventKey : logoutToken.getEvents().keySet()) {
            if (!"http://schemas.openid.net/event/backchannel-logout".equals(eventKey)) continue;
            return true;
        }
        return false;
    }

    private String getReuseIdKey(AccessToken refreshToken) {
        return Optional.ofNullable(refreshToken.getOtherClaims().get("reuse_id")).map(String::valueOf).orElse("");
    }

    public static class NotBeforeCheck
    implements TokenVerifier.Predicate<JsonWebToken> {
        private final int notBefore;

        public NotBeforeCheck(int notBefore) {
            this.notBefore = notBefore;
        }

        public boolean test(JsonWebToken t) throws VerificationException {
            if (t.getIat() < (long)this.notBefore) {
                throw new VerificationException("Stale token");
            }
            return true;
        }

        public static NotBeforeCheck forModel(ClientModel clientModel) {
            if (clientModel != null) {
                int notBeforeClient = clientModel.getNotBefore();
                int notBeforeRealm = clientModel.getRealm().getNotBefore();
                int notBefore = notBeforeClient == 0 ? notBeforeRealm : (notBeforeRealm == 0 ? notBeforeClient : Math.min(notBeforeClient, notBeforeRealm));
                return new NotBeforeCheck(notBefore);
            }
            return new NotBeforeCheck(0);
        }

        public static NotBeforeCheck forModel(RealmModel realmModel) {
            return new NotBeforeCheck(realmModel == null ? 0 : realmModel.getNotBefore());
        }

        public static NotBeforeCheck forModel(KeycloakSession session, RealmModel realmModel, UserModel userModel) {
            return LightweightUserAdapter.isLightweightUser((UserModel)userModel) ? new NotBeforeCheck((int)(((LightweightUserAdapter)userModel).getCreatedTimestamp() / 1000L)) : new NotBeforeCheck(session.users().getNotBeforeOfUser(realmModel, userModel));
        }
    }

    public static class TokenValidation {
        public final UserModel user;
        public final UserSessionModel userSession;
        public final ClientSessionContext clientSessionCtx;
        public final AccessToken newToken;

        public TokenValidation(UserModel user, UserSessionModel userSession, ClientSessionContext clientSessionCtx, AccessToken newToken) {
            this.user = user;
            this.userSession = userSession;
            this.clientSessionCtx = clientSessionCtx;
            this.newToken = newToken;
        }
    }

    public static class TokenRevocationCheck
    implements TokenVerifier.Predicate<JsonWebToken> {
        private final KeycloakSession session;

        public TokenRevocationCheck(KeycloakSession session) {
            this.session = session;
        }

        public boolean test(JsonWebToken token) {
            SingleUseObjectProvider singleUseStore = this.session.singleUseObjects();
            return !singleUseStore.contains(token.getId() + ".revoked");
        }
    }

    public class AccessTokenResponseBuilder {
        RealmModel realm;
        ClientModel client;
        EventBuilder event;
        KeycloakSession session;
        UserSessionModel userSession;
        ClientSessionContext clientSessionCtx;
        AccessToken accessToken;
        RefreshToken refreshToken;
        IDToken idToken;
        String responseTokenType;
        boolean generateAccessTokenHash = false;
        String codeHash;
        String stateHash;
        boolean offlineToken = false;
        private AccessTokenResponse response;

        public AccessTokenResponseBuilder(RealmModel realm, ClientModel client, EventBuilder event, KeycloakSession session, UserSessionModel userSession, ClientSessionContext clientSessionCtx) {
            this.realm = realm;
            this.client = client;
            this.event = event;
            this.session = session;
            this.userSession = userSession;
            this.clientSessionCtx = clientSessionCtx;
            this.responseTokenType = TokenManager.this.formatTokenType(client, null);
        }

        public AccessToken getAccessToken() {
            return this.accessToken;
        }

        public RefreshToken getRefreshToken() {
            return this.refreshToken;
        }

        public IDToken getIdToken() {
            return this.idToken;
        }

        public AccessTokenResponseBuilder accessToken(AccessToken accessToken) {
            this.accessToken = accessToken;
            this.responseTokenType = TokenManager.this.formatTokenType(this.client, accessToken);
            return this;
        }

        public AccessTokenResponseBuilder refreshToken(RefreshToken refreshToken) {
            this.refreshToken = refreshToken;
            return this;
        }

        public AccessTokenResponseBuilder responseTokenType(String responseTokenType) {
            this.responseTokenType = responseTokenType;
            return this;
        }

        public AccessTokenResponseBuilder offlineToken(boolean offlineToken) {
            this.offlineToken = offlineToken;
            return this;
        }

        public AccessTokenResponseBuilder generateAccessToken() {
            UserModel user = this.userSession.getUser();
            this.accessToken = TokenManager.this.createClientAccessToken(this.session, this.realm, this.client, user, this.userSession, this.clientSessionCtx);
            this.responseTokenType = TokenManager.this.formatTokenType(this.client, this.accessToken);
            return this;
        }

        public AccessTokenResponseBuilder generateRefreshToken() {
            if (this.accessToken == null) {
                throw new IllegalStateException("accessToken not set");
            }
            ClientScopeModel offlineAccessScope = KeycloakModelUtils.getClientScopeByName((RealmModel)this.realm, (String)"offline_access");
            boolean offlineTokenRequested = offlineAccessScope == null ? false : this.clientSessionCtx.getClientScopeIds().contains(offlineAccessScope.getId());
            this.generateRefreshToken(offlineTokenRequested);
            this.refreshToken.setScope(this.clientSessionCtx.getScopeString(true));
            if (this.realm.isRevokeRefreshToken()) {
                this.refreshToken.getOtherClaims().put("reuse_id", KeycloakModelUtils.generateId());
            }
            return this;
        }

        public AccessTokenResponseBuilder generateRefreshToken(RefreshToken oldRefreshToken, AuthenticatedClientSessionModel clientSession) {
            if (this.accessToken == null) {
                throw new IllegalStateException("accessToken not set");
            }
            String scope = oldRefreshToken.getScope();
            Object reuseId = oldRefreshToken.getOtherClaims().get("reuse_id");
            boolean offlineTokenRequested = Arrays.asList(scope.split(" ")).contains("offline_access");
            if (offlineTokenRequested) {
                this.clientSessionCtx = DefaultClientSessionContext.fromClientSessionAndScopeParameter(clientSession, scope, this.session);
                if (oldRefreshToken.getNonce() != null) {
                    this.clientSessionCtx.setAttribute("nonce", (Object)oldRefreshToken.getNonce());
                }
            }
            this.generateRefreshToken(offlineTokenRequested);
            if (this.realm.isRevokeRefreshToken()) {
                this.refreshToken.getOtherClaims().put("reuse_id", reuseId);
                clientSession.setRefreshTokenLastRefresh(TokenManager.this.getReuseIdKey((AccessToken)oldRefreshToken), this.refreshToken.getIat().intValue());
            }
            this.refreshToken.setScope(scope);
            return this;
        }

        private void generateRefreshToken(boolean offlineTokenRequested) {
            AuthenticatedClientSessionModel clientSession = this.clientSessionCtx.getClientSession();
            AccessToken.Confirmation confirmation = this.getConfirmation(clientSession, this.accessToken);
            this.refreshToken = new RefreshToken(this.accessToken, confirmation);
            this.refreshToken.id(KeycloakModelUtils.generateId());
            this.refreshToken.issuedNow();
            clientSession.setTimestamp(this.refreshToken.getIat().intValue());
            UserSessionModel userSession = clientSession.getUserSession();
            userSession.setLastSessionRefresh(this.refreshToken.getIat().intValue());
            if (offlineTokenRequested) {
                UserSessionManager sessionManager = new UserSessionManager(this.session);
                if (!sessionManager.isOfflineTokenAllowed(this.clientSessionCtx)) {
                    this.event.detail("reason", "Offline tokens not allowed for the user or client");
                    this.event.error("not_allowed");
                    throw new ErrorResponseException("not_allowed", "Offline tokens not allowed for the user or client", Response.Status.BAD_REQUEST);
                }
                this.refreshToken.type("Offline");
                if (this.realm.isOfflineSessionMaxLifespanEnabled()) {
                    this.refreshToken.exp(this.getExpiration(true));
                }
                sessionManager.createOrUpdateOfflineSession(this.clientSessionCtx.getClientSession(), userSession);
            } else {
                this.refreshToken.exp(this.getExpiration(false));
            }
        }

        private AccessToken.Confirmation getConfirmation(AuthenticatedClientSessionModel clientSession, AccessToken accessToken) {
            boolean isPublicClient = clientSession.getClient().isPublicClient();
            return isPublicClient ? accessToken.getConfirmation() : null;
        }

        private Long getExpiration(boolean offline) {
            long expiration = SessionExpirationUtils.calculateClientSessionIdleTimestamp((boolean)offline, (boolean)this.userSession.isRememberMe(), (long)TimeUnit.SECONDS.toMillis(this.clientSessionCtx.getClientSession().getTimestamp()), (RealmModel)this.realm, (ClientModel)this.client);
            long lifespan = SessionExpirationUtils.calculateClientSessionMaxLifespanTimestamp((boolean)offline, (boolean)this.userSession.isRememberMe(), (long)TimeUnit.SECONDS.toMillis(this.clientSessionCtx.getClientSession().getStarted()), (long)TimeUnit.SECONDS.toMillis(this.userSession.getStarted()), (RealmModel)this.realm, (ClientModel)this.client);
            expiration = lifespan > 0L ? Math.min(expiration, lifespan) : expiration;
            return TimeUnit.MILLISECONDS.toSeconds(expiration);
        }

        public AccessTokenResponseBuilder generateIDToken() {
            return this.generateIDToken(false);
        }

        public AccessTokenResponseBuilder generateIDToken(boolean isIdTokenAsDetachedSignature) {
            if (this.accessToken == null) {
                throw new IllegalStateException("accessToken not set");
            }
            this.idToken = new IDToken();
            this.idToken.id(KeycloakModelUtils.generateId());
            this.idToken.type("ID");
            this.idToken.subject(this.userSession.getUser().getId());
            this.idToken.audience(new String[]{this.client.getClientId()});
            this.idToken.issuedNow();
            this.idToken.issuedFor(this.accessToken.getIssuedFor());
            this.idToken.issuer(this.accessToken.getIssuer());
            this.idToken.setNonce((String)this.clientSessionCtx.getAttribute("nonce", String.class));
            this.idToken.setSessionId(this.accessToken.getSessionId());
            this.idToken.exp(this.accessToken.getExp());
            if (!Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.STEP_UP_AUTHENTICATION)) {
                this.idToken.setAcr(this.accessToken.getAcr());
            }
            if (!isIdTokenAsDetachedSignature) {
                this.idToken = TokenManager.this.transformIDToken(this.session, this.idToken, this.userSession, this.clientSessionCtx);
            }
            return this;
        }

        public AccessTokenResponseBuilder generateAccessTokenHash() {
            this.generateAccessTokenHash = true;
            return this;
        }

        public AccessTokenResponseBuilder generateCodeHash(String code) {
            this.codeHash = this.generateOIDCHash(code);
            return this;
        }

        public AccessTokenResponseBuilder generateStateHash(String state) {
            this.stateHash = this.generateOIDCHash(state);
            return this;
        }

        public boolean isOfflineToken() {
            return this.offlineToken;
        }

        public AccessTokenResponse build() {
            int userNotBefore;
            UserModel user;
            String encodedToken;
            if (this.response != null) {
                return this.response;
            }
            if (this.accessToken != null) {
                this.event.detail("token_id", this.accessToken.getId());
            }
            if (this.refreshToken != null) {
                if (this.event.getEvent().getDetails().containsKey("refresh_token_id")) {
                    this.event.detail("updated_refresh_token_id", this.refreshToken.getId());
                } else {
                    this.event.detail("refresh_token_id", this.refreshToken.getId());
                }
                this.event.detail("refresh_token_type", this.refreshToken.getType());
            }
            AccessTokenResponse res = new AccessTokenResponse();
            if (this.accessToken != null) {
                encodedToken = this.session.tokens().encode((Token)this.accessToken);
                res.setToken(encodedToken);
                res.setTokenType(this.responseTokenType);
                res.setSessionState(this.accessToken.getSessionState());
                if (this.accessToken.getExp() != 0L) {
                    res.setExpiresIn(this.accessToken.getExp() - (long)Time.currentTime());
                }
            }
            if (this.generateAccessTokenHash) {
                String atHash = this.generateOIDCHash(res.getToken());
                this.idToken.setAccessTokenHash(atHash);
            }
            if (this.codeHash != null) {
                this.idToken.setCodeHash(this.codeHash);
            }
            if (this.stateHash != null) {
                this.idToken.setStateHash(this.stateHash);
            }
            if (this.idToken != null) {
                encodedToken = this.session.tokens().encodeAndEncrypt((Token)this.idToken);
                res.setIdToken(encodedToken);
            }
            if (this.refreshToken != null) {
                encodedToken = this.session.tokens().encode((Token)this.refreshToken);
                res.setRefreshToken(encodedToken);
                Long exp = this.refreshToken.getExp();
                if (exp != null && exp > 0L) {
                    res.setRefreshExpiresIn(exp - (long)Time.currentTime());
                }
            }
            int notBefore = this.realm.getNotBefore();
            if (this.client.getNotBefore() > notBefore) {
                notBefore = this.client.getNotBefore();
            }
            if (!LightweightUserAdapter.isLightweightUser((UserModel)(user = this.userSession.getUser())) && (userNotBefore = this.session.users().getNotBeforeOfUser(this.realm, user)) > notBefore) {
                notBefore = userNotBefore;
            }
            res.setNotBeforePolicy(notBefore);
            res = TokenManager.this.transformAccessTokenResponse(this.session, res, this.userSession, this.clientSessionCtx);
            String responseScope = this.clientSessionCtx.getScopeString();
            res.setScope(responseScope);
            this.event.detail("scope", responseScope);
            this.response = res;
            return this.response;
        }

        private String generateOIDCHash(String input) {
            String signatureAlgorithm = this.session.tokens().signatureAlgorithm(TokenCategory.ID);
            SignatureProvider signatureProvider = (SignatureProvider)this.session.getProvider(SignatureProvider.class, signatureAlgorithm);
            String hashAlgorithm = signatureProvider.signer().getHashAlgorithm();
            HashProvider hashProvider = (HashProvider)this.session.getProvider(HashProvider.class, hashAlgorithm);
            byte[] hash = hashProvider.hash(input);
            return HashUtils.encodeHashToOIDC((byte[])hash);
        }
    }

    private static abstract class TokenCollector<T>
    implements Collector<Map.Entry<ProtocolMapperModel, ProtocolMapper>, TokenCollector<T>, T> {
        private T token;

        public TokenCollector(T token) {
            this.token = token;
        }

        @Override
        public Supplier<TokenCollector<T>> supplier() {
            return () -> this;
        }

        @Override
        public Function<TokenCollector<T>, T> finisher() {
            return idTokenWrapper -> idTokenWrapper.token;
        }

        @Override
        public Set<Collector.Characteristics> characteristics() {
            return Collections.emptySet();
        }

        @Override
        public BinaryOperator<TokenCollector<T>> combiner() {
            return (tMutableWrapper, tMutableWrapper2) -> {
                throw new IllegalStateException("can't combine");
            };
        }

        @Override
        public BiConsumer<TokenCollector<T>, Map.Entry<ProtocolMapperModel, ProtocolMapper>> accumulator() {
            return (idToken, mapper) -> {
                idToken.token = this.applyMapper(idToken.token, (Map.Entry<ProtocolMapperModel, ProtocolMapper>)mapper);
            };
        }

        protected abstract T applyMapper(T var1, Map.Entry<ProtocolMapperModel, ProtocolMapper> var2);
    }
}

