/*
 * Copyright 2024 C Thing Software
 * SPDX-License-Identifier: Apache-2.0
 */

package org.cthing.gradle.plugins.locc.reports;

import java.io.File;
import java.nio.file.Path;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import org.gradle.api.Task;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.model.ObjectFactory;
import org.gradle.api.provider.Property;
import org.gradle.api.reporting.Report;

import groovy.lang.Closure;
import groovy.lang.GroovyObjectSupport;


/**
 * Base class for the reports generated by this plugin.
 */
abstract class AbstractLoccReport extends GroovyObjectSupport implements LoccReport {

    protected static final String REPORT_BASE_NAME = "locc";

    protected final Task task;

    private final String name;
    private final String displayName;
    private final RegularFileProperty destination;
    private final Path rootProjectPath;
    private final Property<Boolean> required;
    private final Property<Boolean> showRelativePaths;

    protected AbstractLoccReport(final Task task, final String name, final String displayName,
                                 final boolean required) {
        this.task = task;
        this.name = name;
        this.displayName = displayName;
        this.rootProjectPath = task.getProject().getRootProject().getProjectDir().toPath();

        final ObjectFactory objects = task.getProject().getObjects();
        this.destination = objects.fileProperty();
        this.required = objects.property(Boolean.class).convention(required);
        this.showRelativePaths = objects.property(Boolean.class).convention(Boolean.TRUE);
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public String getDisplayName() {
        return this.displayName;
    }

    @Override
    public OutputType getOutputType() {
        return OutputType.FILE;
    }


    @Override
    public Property<Boolean> getRequired() {
        return this.required;
    }

    @Override
    public Property<Boolean> getShowRelativePaths() {
        return this.showRelativePaths;
    }

    @Override
    public RegularFileProperty getOutputLocation() {
        return this.destination;
    }

    @Override
    @Deprecated
    public void setDestination(final File file) {
        this.destination.set(file);
    }

    /**
     * Sets whether the report is required to be generated. This method is only used by Groovy configuration
     * of the report.
     *
     * @param required {@code true} if the report should be generated
     */
    void setRequired(final boolean required) {
        this.required.set(required);
    }

    /**
     * Sets whether project relative paths should be reported.
     *
     * @param relative {@code true} if the project relative paths should be reported
     */
    void setShowRelativePaths(final boolean relative) {
        this.showRelativePaths.set(relative);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public Report configure(final Closure closure) {
        final Closure cl = (Closure)closure.clone();
        cl.setResolveStrategy(Closure.DELEGATE_FIRST);
        cl.setDelegate(this);
        cl.call(this);
        return this;
    }

    /**
     * Provides an ISO 8601 format timestamp.
     *
     * @return ISO 8601 format timestamp.
     */
    protected String timestamp() {
        return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXXX", Locale.US).format(new Date());
    }

    /**
     * If the report is to show relative pathnames, this method transforms the specified pathname into a path
     * relative to the root of the Gradle project. Otherwise, the specified pathname is returned unchanged.
     *
     * @param pathname Pathname to prepare
     * @return Pathname relative to the Gradle project root, if desired.
     */
    protected Path preparePathname(final Path pathname) {
        if (this.showRelativePaths.get()) {
            return this.rootProjectPath.relativize(pathname);
        }
        return pathname;
    }
}
