/*
 * Copyright 2024 C Thing Software
 * SPDX-License-Identifier: Apache-2.0
 */

package org.cthing.gradle.plugins.locc;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;

import org.cthing.gradle.plugins.locc.reports.CsvReport;
import org.cthing.gradle.plugins.locc.reports.HtmlReport;
import org.cthing.gradle.plugins.locc.reports.JsonReport;
import org.cthing.gradle.plugins.locc.reports.LoccReport;
import org.cthing.gradle.plugins.locc.reports.TextReport;
import org.cthing.gradle.plugins.locc.reports.XmlReport;
import org.cthing.gradle.plugins.locc.reports.YamlReport;
import org.gradle.api.Action;
import org.gradle.api.DomainObjectCollection;
import org.gradle.api.NamedDomainObjectCollectionSchema;
import org.gradle.api.NamedDomainObjectProvider;
import org.gradle.api.NamedDomainObjectSet;
import org.gradle.api.Namer;
import org.gradle.api.Rule;
import org.gradle.api.Task;
import org.gradle.api.UnknownDomainObjectException;
import org.gradle.api.file.DirectoryProperty;
import org.gradle.api.provider.Provider;
import org.gradle.api.reporting.Report;
import org.gradle.api.reporting.ReportContainer;
import org.gradle.api.specs.Spec;
import org.gradle.api.tasks.Internal;
import org.jspecify.annotations.Nullable;

import groovy.lang.Closure;
import groovy.lang.GroovyObjectSupport;


/**
 * Container for the reports generated by the plugin.
 */
public class LoccReports extends GroovyObjectSupport implements ReportContainer<LoccReport> {

    private final NamedDomainObjectSet<LoccReport> reports;
    private final NamedDomainObjectSet<LoccReport> enabled;

    public LoccReports(final Task task, final DirectoryProperty reportsDir) {
        this.reports = task.getProject().getObjects().namedDomainObjectSet(LoccReport.class);
        this.enabled = this.reports.matching(report -> report.getRequired().get());

        this.reports.add(new XmlReport(task, reportsDir));
        this.reports.add(new HtmlReport(task, reportsDir));
        this.reports.add(new YamlReport(task, reportsDir));
        this.reports.add(new JsonReport(task, reportsDir));
        this.reports.add(new CsvReport(task, reportsDir));
        this.reports.add(new TextReport(task, reportsDir));
    }

    @Internal
    public LoccReport getXml() {
        return getByName("xml");
    }

    @Internal
    public LoccReport getHtml() {
        return getByName("html");
    }

    @Internal
    public LoccReport getYaml() {
        return getByName("yaml");
    }

    @Internal
    public LoccReport getJson() {
        return getByName("json");
    }

    @Internal
    public LoccReport getCsv() {
        return getByName("csv");
    }

    @Internal
    public LoccReport getText() {
        return getByName("text");
    }

    @Override
    @SuppressWarnings("AssignmentOrReturnOfFieldWithMutableType")
    public NamedDomainObjectSet<LoccReport> getEnabled() {
        return this.enabled;
    }

    @Override
    public boolean add(final LoccReport report) {
        throw new ImmutableViolationException();
    }

    @Override
    public boolean addAll(final Collection<? extends LoccReport> reps) {
        throw new ImmutableViolationException();
    }

    @Override
    public void addLater(final Provider<? extends LoccReport> provider) {
        throw new ImmutableViolationException();
    }

    @Override
    public void addAllLater(final Provider<? extends Iterable<LoccReport>> provider) {
        throw new ImmutableViolationException();
    }

    @Override
    public boolean remove(final Object report) {
        throw new ImmutableViolationException();
    }

    @Override
    public boolean removeAll(final Collection<?> reps) {
        throw new ImmutableViolationException();
    }

    @Override
    public boolean retainAll(final Collection<?> reps) {
        throw new ImmutableViolationException();
    }

    @Override
    public void clear() {
        throw new ImmutableViolationException();
    }

    @Override
    public boolean containsAll(final Collection<?> reps) {
        return this.reports.containsAll(reps);
    }

    @Override
    public Namer<LoccReport> getNamer() {
        return Report::getName;
    }

    @Override
    public SortedMap<String, LoccReport> getAsMap() {
        return this.reports.getAsMap();
    }

    @Override
    public SortedSet<String> getNames() {
        return this.reports.getNames();
    }

    @Override
    @Nullable
    public LoccReport findByName(final String name) {
        return this.reports.findByName(name);
    }

    @Override
    public LoccReport getByName(final String name) throws UnknownDomainObjectException {
        return this.reports.getByName(name);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public LoccReport getByName(final String name, final Closure configureClosure)
            throws UnknownDomainObjectException {
        return this.reports.getByName(name, configureClosure);
    }

    @Override
    public LoccReport getByName(final String name, final Action<? super LoccReport> configureAction)
            throws UnknownDomainObjectException {
        return this.reports.getByName(name, configureAction);
    }

    @Override
    public LoccReport getAt(final String name) throws UnknownDomainObjectException {
        return this.reports.getAt(name);
    }

    @Override
    public Rule addRule(final Rule rule) {
        return this.reports.addRule(rule);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public Rule addRule(final String description, final Closure ruleAction) {
        return this.reports.addRule(description, ruleAction);
    }

    @Override
    public Rule addRule(final String description, final Action<String> ruleAction) {
        return this.reports.addRule(description, ruleAction);
    }

    @Override
    public List<Rule> getRules() {
        return this.reports.getRules();
    }

    @Override
    public int size() {
        return this.reports.size();
    }

    @Override
    public boolean isEmpty() {
        return this.reports.isEmpty();
    }

    @Override
    public boolean contains(final Object report) {
        return this.reports.contains(report);
    }

    @Override
    public Iterator<LoccReport> iterator() {
        return this.reports.iterator();
    }

    @Override
    public Object[] toArray() {
        return this.reports.toArray();
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T[] toArray(final T[] arr) {
        return (T[])this.reports.toArray((LoccReport[])arr);
    }

    @Override
    public Map<String, LoccReport> getEnabledReports() {
        return this.enabled.getAsMap();
    }

    @Override
    public <S extends LoccReport> NamedDomainObjectSet<S> withType(final Class<S> type) {
        return this.reports.withType(type);
    }

    @Override
    public <S extends LoccReport> DomainObjectCollection<S> withType(final Class<S> type,
                                                                     final Action<? super S> configureAction) {
        return this.reports.withType(type, configureAction);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public <S extends LoccReport> DomainObjectCollection<S> withType(final Class<S> type,
                                                                           final Closure configureClosure) {
        return this.reports.withType(type, configureClosure);
    }

    @Override
    public NamedDomainObjectSet<LoccReport> matching(final Spec<? super LoccReport> spec) {
        return this.reports.matching(spec);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public NamedDomainObjectSet<LoccReport> matching(final Closure spec) {
        return this.reports.matching(spec);
    }

    @Override
    public Action<? super LoccReport> whenObjectAdded(final Action<? super LoccReport> action) {
        return this.reports.whenObjectAdded(action);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public void whenObjectAdded(final Closure action) {
        this.reports.whenObjectAdded(action);
    }

    @Override
    public Action<? super LoccReport> whenObjectRemoved(final Action<? super LoccReport> action) {
        return this.reports.whenObjectRemoved(action);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public void whenObjectRemoved(final Closure action) {
        this.reports.whenObjectRemoved(action);
    }

    @Override
    public void all(final Action<? super LoccReport> action) {
        this.reports.all(action);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public void all(final Closure action) {
        this.reports.all(action);
    }

    @Override
    public void configureEach(final Action<? super LoccReport> action) {
        this.reports.configureEach(action);
    }

    @Override
    public NamedDomainObjectProvider<LoccReport> named(final String name) throws UnknownDomainObjectException {
        return this.reports.named(name);
    }

    @Override
    public NamedDomainObjectProvider<LoccReport> named(final String name,
                                                       final Action<? super LoccReport> configurationAction)
            throws UnknownDomainObjectException {
        return this.reports.named(name, configurationAction);
    }

    @Override
    public <S extends LoccReport> NamedDomainObjectProvider<S> named(final String name, final Class<S> type)
            throws UnknownDomainObjectException {
        return this.reports.named(name, type);
    }

    @Override
    public <S extends LoccReport> NamedDomainObjectProvider<S> named(final String name, final Class<S> type,
                                                                     final Action<? super S> configurationAction)
            throws UnknownDomainObjectException {
        return this.reports.named(name, type, configurationAction);
    }

    @Override
    public NamedDomainObjectSet<LoccReport> named(final Spec<String> nameFilter) {
        return this.reports.named(nameFilter);
    }

    @Override
    public NamedDomainObjectCollectionSchema getCollectionSchema() {
        return this.reports.getCollectionSchema();
    }

    @Override
    @SuppressWarnings("rawtypes")
    public Set<LoccReport> findAll(final Closure spec) {
        return this.reports.findAll(spec);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public ReportContainer<LoccReport> configure(final Closure closure) {
        final Closure cl = (Closure)closure.clone();
        cl.setResolveStrategy(Closure.DELEGATE_FIRST);
        cl.setDelegate(this);
        cl.call(this);
        return this;
    }
}
