/*
 * Copyright 2020-2025 Barfuin and the gradle-taskinfo contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.barfuin.gradle.taskinfo.tasks;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;

import org.barfuin.gradle.taskinfo.EntryNodeProvider;
import org.barfuin.gradle.taskinfo.TaskInfoDto;
import org.barfuin.gradle.taskinfo.TaskProbe;
import org.barfuin.gradle.taskinfo.util.TaskNodeHolder;
import org.barfuin.texttree.api.CycleProtection;
import org.barfuin.texttree.api.IdentityScheme;
import org.barfuin.texttree.api.TextTree;
import org.barfuin.texttree.api.TreeOptions;
import org.barfuin.texttree.api.style.AnnotationPosition;
import org.gradle.api.execution.TaskExecutionGraph;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.TaskAction;


/**
 * Task which gathers the required task info and shows it as a dependency tree.
 */
public abstract class AbstractTaskInfoTreeTask
    extends AbstractInfoTask
{
    private static final long NODES_COUNT_TO_FORCE_CLIPPED = 1000;

    private final Property<Boolean> isAnyTaskFromAnotherBuild;

    private final Property<Boolean> isNonTaskNodesPresent;



    public AbstractTaskInfoTreeTask()
    {
        super();
        setDescription("Displays task dependencies of a given task as a tree, and their task types.");
        isAnyTaskFromAnotherBuild = getProject().getObjects().property(Boolean.class);
        isNonTaskNodesPresent = getProject().getObjects().property(Boolean.class);
    }



    @Override
    public void configureOnTaskGraph(@Nonnull final TaskExecutionGraph pTaskGraph)
    {
        final TaskNodeHolder entryNode = getEffectiveEntryNode(pTaskGraph);
        if (entryNode != null) { // this is null only if we were not invoked
            final TaskProbe taskProbe = buildTaskProbe();
            getTaskInfo().set(taskProbe.buildHierarchy(entryNode));
            getIsAnyTaskFromAnotherBuild().set(taskProbe.isAnyTaskFromAnotherBuild());
            getIsNonTaskNodesPresent().set(taskProbe.isNonTaskNodesPresent());
        }
    }



    @CheckForNull
    TaskNodeHolder getEffectiveEntryNode(@Nonnull final TaskExecutionGraph pTaskGraph)
    {
        final EntryNodeProvider ep = new EntryNodeProvider(getProject(), pTaskGraph);
        return ep.getEffectiveEntryNode();
    }



    @Override
    @TaskAction
    public void execute()
    {
        outputTaskInfo(getTaskInfo().get());
        if (getIsNonTaskNodesPresent().get() && getCfgInternal().get()) {
            printHintOnInternalNodes("Nodes", getCfgColor().get());
        }
        if (getIsAnyTaskFromAnotherBuild().get()) {
            String msg = "The above includes at least one task from another build. Tasks from the included build(s) "
                + "have probably been executed,\nand may not show their dependencies here. You are seeing this because "
                + "taskinfo.disableSafeguard=true.";
            getLogger().lifecycle(msg);
        }
    }



    public TaskProbe buildTaskProbe()
    {
        return new TaskProbe(getProject(), getCfgColor().get(), getCfgInternal().get());
    }



    private void outputTaskInfo(final TaskInfoDto pTaskInfo)
    {
        boolean clipped = getCfgClipped().get();
        if (!clipped) {
            long nodesCount = countTreeNodes(pTaskInfo);
            if (nodesCount > NODES_COUNT_TO_FORCE_CLIPPED) {
                clipped = true;
                getLogger().lifecycle(
                    "The number of nodes in a fully expanded tree is {}, which exceeds a limit of {}. "
                        + "Repeated tasks will be collapsed by forcing the 'clipped' configuration option to 'true'.",
                    nodesCount, NODES_COUNT_TO_FORCE_CLIPPED);
            }
        }

        final TreeOptions options = buildTreeOptions(clipped);
        String tree = TextTree.newInstance(options).render(pTaskInfo);
        getLogger().lifecycle(tree);
    }



    private long countTreeNodes(@Nonnull final TaskInfoDto pTaskInfo)
    {
        return 1 + pTaskInfo.getDependencies().stream()
            .mapToLong(this::countTreeNodes)
            .sum();
    }



    @Nonnull
    private TreeOptions buildTreeOptions(final boolean pClipped)
    {
        TreeOptions options = new TreeOptions();
        options.setAnnotationPosition(
            getCfgShowTaskTypes().get() ? AnnotationPosition.Aligned : AnnotationPosition.None);
        options.setEnableDefaultColoring(getCfgColor().get());
        options.setIdentityScheme(IdentityScheme.ByKey);
        options.setCycleProtection(pClipped ? CycleProtection.PruneRepeating : CycleProtection.On);
        options.setCycleAsPruned(true);
        return options;
    }



    @Nonnull
    @Internal
    Property<Boolean> getIsAnyTaskFromAnotherBuild()
    {
        return isAnyTaskFromAnotherBuild;
    }



    @Nonnull
    @Internal
    Property<Boolean> getIsNonTaskNodesPresent()
    {
        return isNonTaskNodesPresent;
    }
}
