/*
 * Copyright 2020-2025 Barfuin and the gradle-taskinfo contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.barfuin.gradle.taskinfo.tasks;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.StringUtils;
import org.barfuin.gradle.taskinfo.ColorizableAppender;
import org.barfuin.gradle.taskinfo.EntryNodeProvider;
import org.barfuin.gradle.taskinfo.TaskInfoDto;
import org.barfuin.gradle.taskinfo.TaskProbe;
import org.barfuin.gradle.taskinfo.util.TaskNodeHolder;
import org.gradle.api.execution.TaskExecutionGraph;
import org.gradle.api.provider.ListProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.TaskAction;


/**
 * Task which gathers the required task info and shows it as the execution queue.
 */
public abstract class AbstractTaskInfoOrderTask
    extends AbstractInfoTask
{
    private final ListProperty<TaskInfoDto> taskInfos;

    private final Property<Boolean> containsIncludedBuilds;

    private final Property<Boolean> isNonTaskNodesPresent;

    private final Property<String> entryNodeLabel;



    public AbstractTaskInfoOrderTask()
    {
        super();
        setDescription("Displays which tasks would be executed in what order for a given task, and their task types.");
        taskInfos = getProject().getObjects().listProperty(TaskInfoDto.class);
        containsIncludedBuilds = getProject().getObjects().property(Boolean.class);
        isNonTaskNodesPresent = getProject().getObjects().property(Boolean.class);
        entryNodeLabel = getProject().getObjects().property(String.class);
    }



    @Override
    public void configureOnTaskGraph(@Nonnull final TaskExecutionGraph pTaskGraph)
    {
        final EntryNodeProvider ep = new EntryNodeProvider(getProject(), pTaskGraph);
        entryNodeLabel.set(ep.getLabel());

        final TaskProbe taskProbe = new TaskProbe(getProject(), getCfgColor().get(), getCfgInternal().get());
        List<TaskNodeHolder> taskNodeHolders = taskProbe.buildOrder();
        taskInfos.set(taskNodeHolders.stream().map(th -> th.asTaskInfoDto(false, getCfgColor().get())).toList());
        containsIncludedBuilds.set(!getProject().getGradle().getIncludedBuilds().isEmpty());
        isNonTaskNodesPresent.set(taskProbe.isNonTaskNodesPresent());
    }



    @Override
    @TaskAction
    public void execute()
    {
        outputTaskInfo(taskInfos.get());
        if (isNonTaskNodesPresent.get() && getCfgInternal().get()) {
            printHintOnInternalNodes("Items", getCfgColor().get());
        }
        if (containsIncludedBuilds.get()) {
            String msg = "The above may be missing information on tasks from included builds. The tiTree task may have "
                + "a little more information. You are seeing this because taskinfo.disableSafeguard=true.";
            getLogger().lifecycle(msg);
        }
    }



    private void outputTaskInfo(final List<TaskInfoDto> pTaskInfos)
    {
        String output = renderTaskInfo(pTaskInfos);
        getLogger().lifecycle(output);
    }



    private String renderTaskInfo(final List<TaskInfoDto> pTaskInfos)
    {
        final int m = getLengthOfLongestTaskName(pTaskInfos);
        final ColorizableAppender appender = new ColorizableAppender(getCfgColor().get());
        appender.newline();
        appender.append("In order to execute [");
        appender.append(entryNodeLabel.get());
        appender.append("], the following tasks would be executed in this order:");
        appender.newline();
        appender.newline();

        for (int i = 0; i < pTaskInfos.size(); i++) {
            final TaskInfoDto taskInfo = pTaskInfos.get(i);
            final String path = taskInfo.getPath();

            appender.colorNumbers();
            appender.append(String.format("%3d", i + 1));
            appender.append('.');
            appender.colorReset();

            appender.append(' ');
            if (!taskInfo.isTaskNode()) {
                appender.colorAnnotations();
            }
            appender.append(path);
            if (!taskInfo.isTaskNode()) {
                appender.colorReset();
            }
            // finalizer flag is meaningless for ordered output

            if (getCfgShowTaskTypes().get()) {
                appender.append(StringUtils.repeat(' ', m - path.length() + 1));
                appender.colorAnnotations();
                appender.append('(');
                appender.append(taskInfo.getType());
                appender.append(')');
                appender.colorReset();
            }
            appender.newline();
        }
        return appender.toString();
    }



    private int getLengthOfLongestTaskName(final List<TaskInfoDto> pTaskInfos)
    {
        return Collections.max(pTaskInfos, Comparator.comparing(dto -> dto.getPath().length())).getPath().length();
    }
}
