/*
 * Copyright 2020-2025 Barfuin and the gradle-taskinfo contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.barfuin.gradle.taskinfo.tasks;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.barfuin.gradle.taskinfo.EntryNodeProvider;
import org.barfuin.gradle.taskinfo.TaskInfoDto;
import org.barfuin.gradle.taskinfo.TaskProbe;
import org.barfuin.gradle.taskinfo.util.TaskNodeHolder;
import org.gradle.api.GradleException;
import org.gradle.api.Task;
import org.gradle.api.execution.TaskExecutionGraph;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.TaskAction;


/**
 * Task which gathers the required task info and writes it to a JSON file.
 */
public abstract class AbstractTaskInfoJsonTask
    extends AbstractInfoTask
{
    private RegularFileProperty outputFile = null;



    public AbstractTaskInfoJsonTask()
    {
        super();
        setDescription("Creates a JSON file with task dependencies of a given task, and their task types.");
        outputFile = getProject().getObjects().fileProperty();
    }



    @Override
    public void configureOnTaskGraph(@Nonnull final TaskExecutionGraph pTaskGraph)
    {
        final EntryNodeProvider ep = new EntryNodeProvider(getProject(), pTaskGraph);
        final TaskNodeHolder entryNode = ep.getEffectiveEntryNode();

        if (entryNode != null) {  // this is null only if we were not invoked
            final Task entryTask = entryNode.getTask();
            outputFile.set(new File(getProject().getLayout().getBuildDirectory().get().getAsFile(),
                "taskinfo/taskinfo-" + (entryTask != null ? entryTask.getName() : "null") + ".json"));

            // color is always false for the JSON output
            getTaskInfo().set(new TaskProbe(getProject(), false, getCfgInternal().get()).buildHierarchy(entryNode));
        }
    }



    @Override
    @TaskAction
    public void execute()
    {
        try {
            Files.createDirectories(outputFile.get().getAsFile().getParentFile().toPath());
        }
        catch (IOException e) {
            throw new GradleException("Failed to create directory: "
                + outputFile.get().getAsFile().getParentFile().getAbsolutePath(), e);
        }

        outputTaskInfo();
    }



    private void outputTaskInfo()
    {
        try {
            new ObjectMapper().writerWithDefaultPrettyPrinter()
                .writeValue(outputFile.get().getAsFile(), getTaskInfo().get());
        }
        catch (IOException | RuntimeException e) {
            throw new GradleException("Failed to write output JSON file: "
                + outputFile.get().getAsFile().getAbsolutePath(), e);
        }
    }



    @Nonnull
    @Internal
    public Property<TaskInfoDto> getTaskInfoData()
    {
        return getTaskInfo();
    }



    @Nonnull
    @OutputFile
    public RegularFileProperty getOutputFile()
    {
        return outputFile;
    }
}
