/*
 * Copyright 2020-2025 Barfuin and the gradle-taskinfo contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.barfuin.gradle.taskinfo.tasks;

import javax.annotation.Nonnull;

import org.barfuin.gradle.taskinfo.TaskInfoDto;
import org.fusesource.jansi.Ansi;
import org.gradle.api.DefaultTask;
import org.gradle.api.execution.TaskExecutionGraph;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Internal;


/**
 * Common superclass of all tasks provided by this plugin
 */
public abstract class AbstractInfoTask
    extends DefaultTask
{
    private final Property<TaskInfoDto> taskInfo;



    protected AbstractInfoTask()
    {
        super();
        getOutputs().upToDateWhen(task -> false);   // always run when invoked
        taskInfo = getProject().getObjects().property(TaskInfoDto.class);
    }



    @Input
    @Nonnull
    public abstract Property<Boolean> getCfgClipped();


    @Input
    @Nonnull
    public abstract Property<Boolean> getCfgColor();


    @Input
    @Nonnull
    public abstract Property<Boolean> getCfgShowTaskTypes();


    @Input
    @Nonnull
    public abstract Property<Boolean> getCfgInternal();



    @Nonnull
    @Internal
    protected Property<TaskInfoDto> getTaskInfo()
    {
        return taskInfo;
    }



    public abstract void configureOnTaskGraph(@Nonnull final TaskExecutionGraph pTaskGraph);


    public abstract void execute();



    protected void printHintOnInternalNodes(@Nonnull final String pSubject, final boolean pIsColor)
    {
        String msg;
        if (pIsColor) {
            msg = Ansi.ansi().a(pSubject + " in ").fgBrightBlack().a("gray").reset().a(" are not backed by tasks "
                + "but by internal nodes added by Gradle.").toString();
        }
        else {
            msg = "The above includes internal nodes added by Gradle which are not backed by tasks.";
        }
        getLogger().lifecycle(msg);
    }
}
